<?php
// rrhh_home/polilla_html_range.php
// Polilla de horas extra por RANGO de fechas (impresión)
// PHP 8.1.33 – UTF-8

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');

if (!isset($_SESSION["idusuario"])) { die('Sesión expirada'); }

require_once __DIR__ . '/dbcon.php';
if (!isset($con) || !($con instanceof mysqli)) { die('Sin conexión BD'); }
mysqli_set_charset($con,'utf8mb4');

/* ===== Input ===== */
$idempleado = isset($_GET['idempleado']) ? (int)$_GET['idempleado'] : 0;
$desde = $_GET['desde'] ?? '';
$hasta = $_GET['hasta'] ?? '';

if ($idempleado<=0 || !preg_match('/^\d{4}\-\d{2}\-\d{2}$/',$desde) || !preg_match('/^\d{4}\-\d{2}\-\d{2}$/',$hasta) || $hasta<$desde) {
  die('Parámetros inválidos');
}

/* ===== Empleado ===== */
$emp = ['nombre'=>'','cedula'=>'','categoria'=>'','salario_bruto'=>0.0];
if ($st = mysqli_prepare($con,"SELECT nombre_completo, cedula, categoria, COALESCE(salario_bruto,0) 
                               FROM empleados_planilla WHERE id=?")) {
  mysqli_stmt_bind_param($st,'i',$idempleado);
  mysqli_stmt_execute($st);
  mysqli_stmt_bind_result($st,$n,$c,$cat,$sb);
  if (mysqli_stmt_fetch($st)) $emp=['nombre'=>$n,'cedula'=>$c,'categoria'=>$cat,'salario_bruto'=>(float)$sb];
  mysqli_stmt_close($st);
}

/* ===== Preferencias ===== */
$pref=['horas_dia'=>8.0,'dias_mes'=>30.0];
if ($rs=mysqli_query($con,"SELECT horas_dia, dias_mes FROM payroll_pref WHERE activo=1 ORDER BY id DESC LIMIT 1")) {
  if ($r=mysqli_fetch_assoc($rs)) $pref=['horas_dia'=>(float)$r['horas_dia'],'dias_mes'=>(float)$r['dias_mes']];
  mysqli_free_result($rs);
}
$den = max(1.0, ($pref['horas_dia']*$pref['dias_mes']));
$salario_hora = ((float)$emp['salario_bruto']) / $den;

/* ===== Reglas legales (factores) ===== */
$rules = ['factor_overtime'=>1.5,'factor_holiday'=>2.0,'factor_custom'=>3.0];
if ($rs = mysqli_query($con,"SELECT factor_overtime, factor_holiday, factor_custom 
                             FROM payroll_legal_rules WHERE activo=1 ORDER BY id DESC LIMIT 1")) {
  if ($r=mysqli_fetch_assoc($rs)) {
    $rules['factor_overtime']=(float)$r['factor_overtime'];
    $rules['factor_holiday'] =(float)$r['factor_holiday'];
    $rules['factor_custom']  =(float)$r['factor_custom'];
  }
  mysqli_free_result($rs);
}

/* ===== Detalle (rango) ===== */
$detalle=[];
$sql = "SELECT d.fecha, d.horas, d.factor, d.descripcion
        FROM extras_detalle d
        INNER JOIN extras_mes m ON m.id = d.id_extras_mes
        WHERE m.idempleado=? AND d.fecha BETWEEN ? AND ?
        ORDER BY d.fecha ASC";
if ($st = mysqli_prepare($con,$sql)) {
  mysqli_stmt_bind_param($st,'iss',$idempleado,$desde,$hasta);
  mysqli_stmt_execute($st);
  $rs=mysqli_stmt_get_result($st);
  while($r=mysqli_fetch_assoc($rs)) {
    $detalle[]=[
      'fecha'=>$r['fecha'],
      'horas'=>(float)$r['horas'],
      'factor'=>(string)$r['factor'],
      'descripcion'=>$r['descripcion'] ?? ''
    ];
  }
  mysqli_stmt_close($st);
}

/* ===== Totales por factor ===== */
$tot=['h15'=>0.0,'h20'=>0.0,'h30'=>0.0,'hTot'=>0.0];
foreach($detalle as $d){
  $tot['hTot'] += $d['horas'];
  if ($d['factor']=='1.5') $tot['h15'] += $d['horas'];
  elseif ($d['factor']=='2.0') $tot['h20'] += $d['horas'];
  elseif ($d['factor']=='3.0') $tot['h30'] += $d['horas'];
}

/* ===== Montos ===== */
$m15 = $tot['h15'] * $salario_hora * (float)$rules['factor_overtime'];
$m20 = $tot['h20'] * $salario_hora * (float)$rules['factor_holiday'];
$m30 = $tot['h30'] * $salario_hora * (float)$rules['factor_custom'];
$mt  = $m15 + $m20 + $m30;

function money($n){ return '₡'.number_format((float)$n,2,'.',','); }
function esc($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
?>
<!DOCTYPE html>
<html lang="es">
<head>
<meta charset="utf-8">
<title>Polilla por Rango – <?= esc($emp['nombre']) ?></title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<style>
  body{font-family:Arial,Helvetica,sans-serif; font-size:13px; color:#111; margin:16px;}
  h1{font-size:18px; margin:0 0 8px}
  h2{font-size:14px; margin:12px 0 6px}
  .muted{color:#666}
  table{border-collapse:collapse; width:100%}
  th,td{border:1px solid #e5e7eb; padding:6px 8px; vertical-align:top}
  th{background:#f8fafc; text-align:left}
  .right{text-align:right}
  .tot{font-weight:bold; background:#f1f5f9}
  .box{border:1px solid #e5e7eb; border-radius:8px; padding:10px; margin-bottom:12px}
  .small{font-size:12px}
  @media print{ .noprint{display:none} body{margin:0} }
</style>
</head>
<body>

<div class="noprint" style="margin-bottom:8px">
  <button onclick="window.print()">Imprimir</button>
</div>

<h1>Polilla de horas extra (Rango)</h1>
<div class="box">
  <div><b>Empleado:</b> <?= esc($emp['nombre']) ?> <span class="muted">|</span> <b>Cédula:</b> <?= esc($emp['cedula']) ?></div>
  <div><b>Categoría:</b> <?= esc($emp['categoria']) ?> <span class="muted">|</span> <b>Rango:</b> <?= esc($desde) ?> → <?= esc($hasta) ?></div>
  <div class="small muted">Salario bruto: <?= money($emp['salario_bruto']) ?> · Pref: <?=$pref['horas_dia']?> h/día, <?=$pref['dias_mes']?> días/mes · Salario hora: <?= money($salario_hora) ?></div>
  <div class="small muted">Factores: OT=<?=$rules['factor_overtime']?>x · Feriado=<?=$rules['factor_holiday']?>x · Custom=<?=$rules['factor_custom']?>x</div>
</div>

<h2>Detalle</h2>
<table>
  <thead>
    <tr>
      <th style="width:110px">Fecha</th>
      <th style="width:60px" class="right">Horas</th>
      <th style="width:60px" class="right">Factor</th>
      <th>Descripción</th>
      <th style="width:110px" class="right">Monto</th>
    </tr>
  </thead>
  <tbody>
  <?php if(!$detalle): ?>
    <tr><td colspan="5" class="muted">No hay registros en el rango indicado.</td></tr>
  <?php else: ?>
    <?php foreach($detalle as $d):
      $fac = ($d['factor']=='1.5') ? (float)$rules['factor_overtime'] :
             (($d['factor']=='2.0') ? (float)$rules['factor_holiday'] : (float)$rules['factor_custom']);
      $monto = $d['horas'] * $salario_hora * $fac;
    ?>
      <tr>
        <td><?= esc($d['fecha']) ?></td>
        <td class="right"><?= number_format($d['horas'],2,'.',',') ?></td>
        <td class="right"><?= esc($d['factor']) ?>x</td>
        <td><?= esc($d['descripcion']) ?></td>
        <td class="right"><?= money($monto) ?></td>
      </tr>
    <?php endforeach; ?>
  <?php endif; ?>
  </tbody>
  <tfoot>
    <tr class="tot">
      <td>Total</td>
      <td class="right"><?= number_format($tot['hTot'],2,'.',',') ?></td>
      <td class="right">—</td>
      <td class="right">1.5x: <?= number_format($tot['h15'],2,'.',',') ?> · 2.0x: <?= number_format($tot['h20'],2,'.',',') ?> · 3.0x: <?= number_format($tot['h30'],2,'.',',') ?></td>
      <td class="right"><?= money($mt) ?></td>
    </tr>
  </tfoot>
</table>

</body>
</html>
