<?php
// rrhh_home/polilla_html.php
// Polilla: Mes / Quincena (q1, q2) + switches (extras/ausencias) y deducciones legales + ISR desde tablas.
// Con AJAX (ajax=1) para recalcular sin recargar.
// PHP 8.1.33 – UTF-8

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');
if (!isset($_SESSION["idusuario"])) { header("Location:index.html"); exit; }
$ES_SUPER = isset($_SESSION["superadmin"]) && $_SESSION["superadmin"] === "S";

require_once __DIR__ . '/dbcon.php';
mysqli_set_charset($con,'utf8mb4');

/* ============================= Helpers ============================= */
function ensure_header(mysqli $con, int $idempleado, string $mes): int {
  $id = 0;
  $q = "SELECT id FROM extras_mes WHERE idempleado=? AND mes=? LIMIT 1";
  if($st = mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'is',$idempleado,$mes);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$idtmp);
    if(mysqli_stmt_fetch($st)) $id = (int)$idtmp;
    mysqli_stmt_close($st);
  }
  if($id>0) return $id;
  $uid = (int)($_SESSION['idusuario'] ?? 0);
  $ins = "INSERT INTO extras_mes (idempleado, mes, creado_por) VALUES (?,?,?)";
  if($st = mysqli_prepare($con,$ins)){
    mysqli_stmt_bind_param($st,'isi',$idempleado,$mes,$uid);
    mysqli_stmt_execute($st);
    $id = mysqli_insert_id($con);
    mysqli_stmt_close($st);
  }
  return (int)$id;
}

function ensure_abs_header(mysqli $con, int $idempleado, string $mes): ?int {
  $exists = false;
  if ($rs = mysqli_query($con, "SHOW TABLES LIKE 'ausencias_mes'")) { $exists = (bool)mysqli_fetch_row($rs); mysqli_free_result($rs); }
  if (!$exists) return null;
  $id = null;
  $q = "SELECT id FROM ausencias_mes WHERE idempleado=? AND mes=? LIMIT 1";
  if($st = mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'is',$idempleado,$mes);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$idtmp);
    if(mysqli_stmt_fetch($st)) $id = (int)$idtmp;
    mysqli_stmt_close($st);
  }
  return $id;
}

function get_pref(mysqli $con): array {
  $pref=['horas_dia'=>8.0,'dias_mes'=>30.0];
  $qp="SELECT horas_dia, dias_mes FROM payroll_pref WHERE activo=1 ORDER BY id DESC LIMIT 1";
  if($rs=mysqli_query($con,$qp)){
    if($r=mysqli_fetch_assoc($rs)){ $pref=['horas_dia'=>(float)$r['horas_dia'],'dias_mes'=>(float)$r['dias_mes']]; }
    mysqli_free_result($rs);
  }
  return $pref;
}

function get_salario_hora(float $sb, array $pref): float {
  $h=($pref['horas_dia']??0)*($pref['dias_mes']??0);
  return $h>0?($sb/$h):0.0;
}

function compute_totals_extras_range(mysqli $con, int $id_extras_mes, string $desde, string $hasta): array {
  $out = ['h15'=>0.0,'h20'=>0.0,'h30'=>0.0,'hTot'=>0.0];
  $q = "SELECT
          COALESCE(SUM(CASE WHEN factor=1.5 THEN horas ELSE 0 END),0) AS h15,
          COALESCE(SUM(CASE WHEN factor=2.0 THEN horas ELSE 0 END),0) AS h20,
          COALESCE(SUM(CASE WHEN factor=3.0 THEN horas ELSE 0 END),0) AS h30,
          COALESCE(SUM(horas),0) AS hTot
        FROM extras_detalle
        WHERE id_extras_mes=? AND fecha BETWEEN ? AND ?";
  if($st = mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'iss',$id_extras_mes,$desde,$hasta);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$a,$b,$c,$t);
    if(mysqli_stmt_fetch($st)) $out = ['h15'=>(float)$a,'h20'=>(float)$b,'h30'=>(float)$c,'hTot'=>(float)$t];
    mysqli_stmt_close($st);
  }
  return $out;
}

function get_legal_rules_fallback(mysqli $con): array {
  $r = ['factor_regular'=>1.00,'factor_overtime'=>1.50,'factor_holiday'=>2.00,'factor_custom'=>3.00];
  $sql = "SELECT factor_regular,factor_overtime,factor_holiday,factor_custom
          FROM payroll_legal_rules WHERE activo=1 ORDER BY id DESC LIMIT 1";
  if ($rs = mysqli_query($con,$sql)) {
    if ($row = mysqli_fetch_assoc($rs)) { foreach ($r as $k=>$_) if(isset($row[$k])) $r[$k]=(float)$row[$k]; }
    mysqli_free_result($rs);
  }
  return $r;
}

function get_active_legal_set(mysqli $con): ?array {
  $set = null;
  if ($rs = mysqli_query($con,"SELECT id, nombre FROM payroll_legal_set WHERE activo=1 ORDER BY id DESC LIMIT 1")) {
    $set = mysqli_fetch_assoc($rs) ?: null; mysqli_free_result($rs);
  }
  if (!$set) return null;

  $rates=[]; $tramos=[];
  if ($st = mysqli_prepare($con,"SELECT nombre, rol, tasa FROM payroll_legal_rate WHERE id_set=? ORDER BY id ASC")){
    mysqli_stmt_bind_param($st,'i',$set['id']); mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st); while($r=mysqli_fetch_assoc($rs)) $rates[]=$r; mysqli_stmt_close($st);
  }
  if ($st = mysqli_prepare($con,"SELECT desde, hasta, pct FROM payroll_isr_tramo WHERE set_id=? ORDER BY desde ASC")){
    mysqli_stmt_bind_param($st,'i',$set['id']); mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    while($r=mysqli_fetch_assoc($rs)){
      $tramos[]=['desde'=>(float)$r['desde'],'hasta'=>isset($r['hasta'])?(float)$r['hasta']:null,'porcentaje'=>(float)$r['pct']];
    }
    mysqli_stmt_close($st);
  }
  return ['id'=>(int)$set['id'],'nombre'=>$set['nombre'],'rates'=>$rates,'tramos'=>$tramos];
}

/**
 * Calcula ISR progresivo con desglose. Escala tramos si el período es quincena.
 * Devuelve:
 *   monto (float)
 *   aplicados: [{desde,hasta|null,pct,monto}]
 *   top: {pct,...} tramo superior utilizado
 */
function compute_isr_progressive_scaled_with_breakdown(float $base, array $tramos, float $factor): array {
  $isr = 0.0; $aplicados=[]; $top=null;

  foreach ($tramos as $t) {
    $desde = (float)$t['desde'] * $factor;
    $hasta = is_null($t['hasta']) ? INF : (float)$t['hasta'] * $factor;
    $pct   = max(0.0,(float)$t['porcentaje'])/100.0;

    if ($base > $desde) {
      $aplicaHasta = min($base, $hasta);
      $exceso = max(0.0, $aplicaHasta - $desde);
      if ($exceso > 0) {
        $m = $exceso * $pct;          // puede ser 0 si el tramo es 0 %
        $isr += $m;
        $aplicados[] = [
          'desde'=>$desde,'hasta'=>is_finite($hasta)?$hasta:null,
          'pct'=>$t['porcentaje'],'monto'=>$m
        ];
        $top = $t; // último tramo alcanzado (aunque sea 0 %)
      }
    }
  }

  return ['monto'=>$isr,'aplicados'=>$aplicados,'top'=>$top];
}

function pick_core_employee_rates(?array $legal_set): array {
  $core = ['IVM'=>null,'SEM/EM'=>null,'Banco Popular'=>null];
  if ($legal_set && !empty($legal_set['rates'])) {
    foreach ($legal_set['rates'] as $r) {
      $rol = strtoupper((string)($r['rol'] ?? ''));
      if ($rol !== 'EMPLOYEE') continue;
      $nombre = strtolower((string)$r['nombre']); $tasa = (float)$r['tasa'];
      if (strpos($nombre,'ivm') !== false)                                       $core['IVM'] = ['nombre'=>'IVM','tasa'=>$tasa];
      elseif (strpos($nombre,'sem')!==false || strpos($nombre,'enfer')!==false)  $core['SEM/EM'] = ['nombre'=>'Enfermedad y Maternidad','tasa'=>$tasa];
      elseif (strpos($nombre,'popular')!==false)                                 $core['Banco Popular'] = ['nombre'=>'Banco Popular','tasa'=>$tasa];
    }
  }
  // Defaults si no vienen en BD:
  if (!$core['IVM'])           $core['IVM']           = ['nombre'=>'IVM','tasa'=>4.17];
  if (!$core['SEM/EM'])        $core['SEM/EM']        = ['nombre'=>'Enfermedad y Maternidad','tasa'=>5.50];
  if (!$core['Banco Popular']) $core['Banco Popular'] = ['nombre'=>'Banco Popular','tasa'=>1.00];
  return array_values($core);
}

/* -------- función central que arma el cálculo (reutilizable para AJAX) -------- */
function do_calc(mysqli $con, int $idempleado, string $mes, string $modo, int $inc_extras, int $inc_aus): array {
  // Empleado
  $emp = null;
  $qe = "SELECT id, nombre_completo, cedula, categoria, COALESCE(salario_bruto,0) AS salario_bruto
         FROM empleados_planilla WHERE id=? LIMIT 1";
  if ($st = mysqli_prepare($con,$qe)) {
    mysqli_stmt_bind_param($st,'i',$idempleado);
    mysqli_stmt_execute($st);
    $r = mysqli_stmt_get_result($st);
    $emp = mysqli_fetch_assoc($r);
    mysqli_stmt_close($st);
  }
  if(!$emp) return ['ok'=>false,'msg'=>'Empleado no existe.'];

  $pref             = get_pref($con);
  $sal_h            = get_salario_hora((float)$emp['salario_bruto'], $pref);
  $salario_mensual  = (float)$emp['salario_bruto'];

  // Rango (mes / quincena)
  $y=(int)substr($mes,0,4); $m=(int)substr($mes,5,2);
  $desdeMes = sprintf('%04d-%02d-01',$y,$m);
  $hastaMes = date('Y-m-t', strtotime($desdeMes));
  $desde = $desdeMes; $hasta = $hastaMes; $labelPeriodo='Mes completo'; $factorPeriodo=1.0;
  if ($modo==='q1'){ $desde=$desdeMes; $hasta=sprintf('%04d-%02d-15',$y,$m); $labelPeriodo='Quincena 1 (1–15)'; $factorPeriodo=0.5; }
  if ($modo==='q2'){ $desde=sprintf('%04d-%02d-16',$y,$m); $hasta=$hastaMes; $labelPeriodo='Quincena 2 (16–fin)'; $factorPeriodo=0.5; }

  $salario_periodo = $salario_mensual * $factorPeriodo;

  // Extras en rango
  $id_extras_mes = ensure_header($con,$idempleado,$mes);
  $totales = compute_totals_extras_range($con,$id_extras_mes,$desde,$hasta);

  $rules_fb = get_legal_rules_fallback($con);
  $fOver = $rules_fb['factor_overtime'] ?? 1.5;
  $fHol  = $rules_fb['factor_holiday']  ?? 2.0;
  $fCus  = $rules_fb['factor_custom']   ?? 3.0;

  $m15 = ((float)$totales['h15']) * $sal_h * $fOver;
  $m20 = ((float)$totales['h20']) * $sal_h * $fHol;
  $m30 = ((float)$totales['h30']) * $sal_h * $fCus;
  $montos_extras = $inc_extras ? ($m15 + $m20 + $m30) : 0.0;

  // Ausencias/retrasos en rango
  $horas_ausencia=0.0; $horas_retraso=0.0;
  $idaus = ensure_abs_header($con,$idempleado,$mes);
  if ($idaus){
    $sql="SELECT fecha, horas, UPPER(TRIM(COALESCE(tipo,''))) tipo
          FROM ausencias_detalle
          WHERE id_ausencias_mes=? AND fecha BETWEEN ? AND ?";
    if($st=mysqli_prepare($con,$sql)){
      mysqli_stmt_bind_param($st,'iss',$idaus,$desde,$hasta);
      mysqli_stmt_execute($st); $rs=mysqli_stmt_get_result($st);
      while($row=mysqli_fetch_assoc($rs)){
        $t=$row['tipo']; $h=(float)$row['horas'];
        if ($t==='R' || $t==='RETRASO') $horas_retraso += $h; else $horas_ausencia += $h;
      }
      mysqli_stmt_close($st);
    }
  } else {
    if ($rs = mysqli_query($con,"SHOW COLUMNS FROM ausencias_detalle LIKE 'idempleado'")) {
      if (mysqli_fetch_row($rs)) {
        $sql="SELECT fecha, horas, UPPER(TRIM(COALESCE(tipo,''))) tipo
              FROM ausencias_detalle
              WHERE idempleado=? AND fecha BETWEEN ? AND ?";
        if($st=mysqli_prepare($con,$sql)){
          mysqli_stmt_bind_param($st,'iss',$idempleado,$desde,$hasta);
          mysqli_stmt_execute($st); $rs=mysqli_stmt_get_result($st);
          while($row=mysqli_fetch_assoc($rs)){
            $t=$row['tipo']; $h=(float)$row['horas'];
            if ($t==='R' || $t==='RETRASO') $horas_retraso += $h; else $horas_ausencia += $h;
          }
          mysqli_stmt_close($st);
        }
      }
      mysqli_free_result($rs);
    }
  }
  $horas_aus_tot = $inc_aus ? ($horas_ausencia + $horas_retraso) : 0.0;
  $descuento_aus = $horas_aus_tot * $sal_h;

  // Base gravable
  $base_gravable = $salario_periodo + $montos_extras - $descuento_aus;
  if ($base_gravable < 0) $base_gravable = 0.0;

  // Tasas EMPLEADO y tramos ISR del set activo
  $legal_set  = get_active_legal_set($con);
  $core_rates = pick_core_employee_rates($legal_set);

  // Deducciones empleado (IVM, EM/SEM, Banco Popular)
  $ded_empleado_ss = 0.0; $detalle_deducciones=[];
  foreach ($core_rates as $cr){
    $m = $base_gravable * ((float)$cr['tasa']/100.0);
    $ded_empleado_ss += $m;
    $detalle_deducciones[] = ['nombre'=>$cr['nombre'],'tasa'=>$cr['tasa'],'monto'=>$m];
  }

  // ISR progresivo (base después de cargas, ajustando tramos si es quincena)
  $base_para_isr = max(0.0, $base_gravable - $ded_empleado_ss);
  $factorTramos  = $factorPeriodo;
  $isrData = $legal_set ? compute_isr_progressive_scaled_with_breakdown($base_para_isr, $legal_set['tramos'], $factorTramos) : ['monto'=>0.0,'aplicados'=>[],'top'=>null];
  $isr = (float)$isrData['monto'];

  // Neto
  $deducciones_totales = $ded_empleado_ss + $isr;
  $neto_pagar = max(0.0, $base_gravable - $deducciones_totales);

  return [
    'ok'=>true,
    'empleado'=>$emp,
    'pref'=>$pref,
    'sal_h'=>$sal_h,
    'labelPeriodo'=>$labelPeriodo,
    'desde'=>$desde,'hasta'=>$hasta,
    'salario_periodo'=>$salario_periodo,
    'h15'=>$totales['h15'],'h20'=>$totales['h20'],'h30'=>$totales['h30'],
    'mExtras'=>$inc_extras?($montos_extras):0.0,
    'horasAus'=>$horas_ausencia,'horasRet'=>$horas_retraso,
    'horasAusTot'=>$horas_aus_tot,'descAus'=>$descuento_aus,
    'baseGrav'=>$base_gravable,
    'dedetalle'=>$detalle_deducciones,
    'baseISR'=>$base_para_isr,
    'isr'=>$isr,
    'isr_aplicados'=>$isrData['aplicados'],
    'isr_top'=>$isrData['top'],
    'dedTot'=>$deducciones_totales,
    'neto'=>$neto_pagar,
    'setLegal'=>($legal_set? $legal_set['nombre'] : '—')
  ];
}

/* ===================== Parámetros y posible AJAX ===================== */
$idempleado = isset($_GET['idempleado']) ? (int)$_GET['idempleado'] : 0;
$mes        = isset($_GET['mes']) ? $_GET['mes'] : date('Y-m');
$modo       = isset($_GET['modo']) ? strtolower(trim((string)$_GET['modo'])) : 'mes';   // mes|q1|q2
$inc_extras = isset($_GET['inc_extras']) ? (int)$_GET['inc_extras'] : 1;               // 1/0
$inc_aus    = isset($_GET['inc_aus'])    ? (int)$_GET['inc_aus']    : 1;               // 1/0
$ajax       = isset($_GET['ajax'])       ? (int)$_GET['ajax']       : 0;

if ($ajax === 1) {
  header('Content-Type: application/json; charset=utf-8');
  if ($idempleado<=0 || !preg_match('~^\d{4}-\d{2}$~',$mes)) { echo json_encode(['ok'=>false,'msg'=>'Parámetros inválidos.']); exit; }
  echo json_encode(do_calc($con,$idempleado,$mes,$modo,$inc_extras,$inc_aus), JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
  exit;
}

/* =========================== Cálculo inicial =========================== */
if ($idempleado<=0 || !preg_match('~^\d{4}-\d{2}$~',$mes)) { die('Parámetros inválidos.'); }
$calc = do_calc($con,$idempleado,$mes,$modo,$inc_extras,$inc_aus);
if (!$calc['ok']) { die('No se pudo calcular.'); }

$MES_NOMBRE = strftime('%B %Y', strtotime($mes.'-01'));

/* ============================== UI ============================== */
if(!$ES_SUPER) include __DIR__.'/view/header.php'; else include __DIR__.'/view/headeradmin.html';
?>
<script>
  $('#payroll_control').addClass("treeview active");
  $('#payroll_control_2').addClass("treeview active");
</script>

<style>
  .polilla{background:#fff;border:1px solid #e5e7eb;border-radius:12px;padding:16px}
  .polilla h3{margin:6px 0 12px 0;font-weight:800}
  .kv{margin-bottom:4px}
  .kv b{display:inline-block;min-width:160px;color:#334155}
  .tbl{width:100%;border-collapse:collapse;margin-top:10px;font-size:13px}
  .tbl th,.tbl td{border:1px solid #e5e7eb;padding:8px}
  .tbl th{background:#f8fafc;font-weight:700}
  .totals{display:grid;grid-template-columns:repeat(4,1fr);gap:8px;margin-top:12px}
  .totals2{display:grid;grid-template-columns:repeat(4,1fr);gap:8px;margin-top:8px}
  .filterbar{border:1px solid #e5e7eb;border-radius:10px;background:#f8fafc;padding:10px;margin-bottom:10px}
  @media print{ .box-header,.main-header,.main-sidebar,.btn,.filterbar,.content-header{display:none!important} body{background:#fff} .polilla{border:none} }
</style>

<div class="box">
  <div class="box-header with-border">
    <h3 class="box-title">Polilla – <?= htmlspecialchars(ucfirst($MES_NOMBRE)) ?> · <span id="lblPeriodo"><?= htmlspecialchars($calc['labelPeriodo']) ?></span></h3>
    <div class="box-tools pull-right">
      <a href="javascript:window.print()" class="btn btn-default btn-sm"><i class="fa fa-print"></i> Imprimir</a>
    </div>
  </div>

  <div class="box-body">
    <!-- Barra de filtros -->
    <form class="filterbar" id="polillaForm" method="get" action="polilla_html.php">
      <input type="hidden" name="idempleado" id="idempleado" value="<?= (int)$idempleado ?>">
      <input type="month" name="mes" id="mes" value="<?= htmlspecialchars($mes) ?>" style="margin-right:12px">
      <label style="margin-right:10px"><input type="radio" name="modo" value="mes" <?= $modo==='mes'?'checked':''; ?>> Mes</label>
      <label style="margin-right:10px"><input type="radio" name="modo" value="q1"  <?= $modo==='q1'?'checked':''; ?>> Quincena 1</label>
      <label style="margin-right:14px"><input type="radio" name="modo" value="q2"  <?= $modo==='q2'?'checked':''; ?>> Quincena 2</label>
      <label style="margin-right:10px"><input type="checkbox" name="inc_extras" id="inc_extras" value="1" <?= $inc_extras? 'checked':''; ?>> Incluir extras</label>
      <label style="margin-right:14px"><input type="checkbox" name="inc_aus" id="inc_aus" value="1" <?= $inc_aus? 'checked':''; ?>> Incluir ausencias</label>
      <button class="btn btn-primary btn-sm" type="submit">Aplicar</button>
    </form>

    <div class="polilla">
      <h3>Resumen del colaborador</h3>
      <div class="kv"><b>Colaborador:</b> <span id="empNombre"><?= htmlspecialchars($calc['empleado']['nombre_completo']) ?></span></div>
      <div class="kv"><b>Cédula:</b> <?= htmlspecialchars($calc['empleado']['cedula']) ?></div>
      <div class="kv"><b>Categoría:</b> <?= htmlspecialchars($calc['empleado']['categoria']) ?></div>
      <div class="kv"><b>Período:</b> <span id="rangoDesde"><?= htmlspecialchars($calc['desde']) ?></span> a <span id="rangoHasta"><?= htmlspecialchars($calc['hasta']) ?></span></div>
      <div class="kv"><b>Salario base del período:</b> ₡<span id="salPeriodo"><?= number_format($calc['salario_periodo'],2,'.',',') ?></span></div>
      <div class="kv"><b>Salario hora (<?= (float)$calc['pref']['horas_dia'] ?>h × <?= (float)$calc['pref']['dias_mes'] ?>d):</b>
        ₡<span id="salHora"><?= number_format($calc['sal_h'],2,'.',',') ?></span></div>

      <h3 style="margin-top:16px">Ajustes del período</h3>
      <div class="totals2">
        <div><b>Horas extra 1.5x:</b> <span id="h15"><?= number_format((float)$calc['h15'],2,'.',',') ?></span></div>
        <div><b>Horas extra 2.0x:</b> <span id="h20"><?= number_format((float)$calc['h20'],2,'.',',') ?></span></div>
        <div><b>Horas extra 3.0x:</b> <span id="h30"><?= number_format((float)$calc['h30'],2,'.',',') ?></span></div>
        <div><b>Monto por extras<?= $inc_extras?'':' (no aplicado)'; ?>:</b> ₡<span id="mExtras"><?= number_format($calc['mExtras'],2,'.',',') ?></span></div>
      </div>
      <div class="totals2" style="margin-top:6px">
        <div><b>Horas de ausencia:</b> <span id="hAus"><?= number_format($calc['horasAus'],2,'.',',') ?></span></div>
        <div><b>Horas de retraso:</b> <span id="hRet"><?= number_format($calc['horasRet'],2,'.',',') ?></span></div>
        <div><b>Horas ausentes totales:</b> <span id="hAusTot"><?= number_format($calc['horasAusTot'],2,'.',',') ?></span></div>
        <div><b>Descuento por ausencias/retrasos:</b> ₡<span id="mAus"><?= number_format($calc['descAus'],2,'.',',') ?></span></div>
      </div>

      <div class="totals2" style="margin-top:6px">
        <div><b>Base gravable (salario ± ajustes):</b> ₡<span id="baseGrav"><?= number_format($calc['baseGrav'],2,'.',',') ?></span></div>
        <div></div><div></div><div></div>
      </div>

      <h3 style="margin-top:16px">Deducciones legales</h3>
      <div class="kv"><b>Set legal:</b> <span id="setLegal"><?= htmlspecialchars($calc['setLegal']) ?></span></div>
      <table class="tbl">
        <thead>
          <tr><th>Concepto</th><th style="width:120px;text-align:right">Tasa</th><th style="width:160px;text-align:right">Monto</th></tr>
        </thead>
        <tbody id="dedBody">
          <?php foreach ($calc['dedetalle'] as $d): ?>
            <tr>
              <td><?= htmlspecialchars($d['nombre']) ?></td>
              <td style="text-align:right"><?= number_format((float)$d['tasa'],2,'.',',') ?>%</td>
              <td style="text-align:right">₡<?= number_format((float)$d['monto'],2,'.',',') ?></td>
            </tr>
          <?php endforeach; ?>
          <tr>
            <td><b>Base para ISR (después de cargas)</b></td>
            <td></td>
            <td style="text-align:right"><b>₡<span id="baseISR"><?= number_format($calc['baseISR'],2,'.',',') ?></span></b></td>
          </tr>
          <tr>
            <td>
              <b>Impuesto sobre la Renta (ISR)</b><br>
              <small id="isrNote">
                <?php
                  if (!empty($calc['isr_aplicados'])) {
                    $top=$calc['isr_top'];
                    $topTxt = $top ? (number_format((float)$top['porcentaje'],2).'%') : '—';
                    $pieces=[];
                    foreach ($calc['isr_aplicados'] as $ap) {
                      $desdeTxt = '₡'.number_format($ap['desde'],0,'.',',');
                      $hastaTxt = is_null($ap['hasta']) ? '∞' : '₡'.number_format($ap['hasta'],0,'.',',');
                      $pieces[] = $desdeTxt.'–'.$hastaTxt.' @ '.number_format((float)$ap['pct'],2).'% → ₡'.number_format((float)$ap['monto'],2,'.',',');
                    }
                    echo 'Tramo superior aplicado: '.$topTxt.'; Detalle: '.implode(' | ',$pieces);
                  } else {
                    echo 'Sin ISR en el período.';
                  }
                ?>
              </small>
            </td>
            <td style="text-align:right">—</td>
            <td style="text-align:right"><b>₡<span id="isrVal"><?= number_format($calc['isr'],2,'.',',') ?></span></b></td>
          </tr>
          <tr>
            <td><b>Total deducciones</b></td>
            <td></td>
            <td style="text-align:right"><b>₡<span id="dedTot"><?= number_format($calc['dedTot'],2,'.',',') ?></span></b></td>
          </tr>
        </tbody>
      </table>

      <div class="totals2" style="margin-top:10px">
        <div style="grid-column: 1 / span 4; border:1px solid #e5e7eb; border-radius:10px; padding:10px; background:#f8fafc;">
          <b>Neto a pagar al colaborador (<span id="lblPeriodo2"><?= htmlspecialchars($calc['labelPeriodo']) ?></span>):</b>
          <span style="float:right">₡<span id="netoVal"><?= number_format($calc['neto'], 2, '.', ',') ?></span></span>
          <div style="clear:both; font-size:12px; color:#64748b; margin-top:4px">
            * Neto = Salario del período ± (extras − ausencias/retrasos) − (IVM + EM/SEM + Banco Popular + ISR).
          </div>
        </div>
      </div>

      <?php
      // Info (mes) de vacaciones/incapacidades:
      $vac=[]; $inc=[];
      $y=(int)substr($mes,0,4); $m=(int)substr($mes,5,2);
      $desdeMes = sprintf('%04d-%02d-01',$y,$m);
      $hastaMes = date('Y-m-t', strtotime($desdeMes));
      $qv="SELECT id, fecha_desde, fecha_hasta, motivo, cantidad_dias
           FROM ingreso_vacaciones
           WHERE idempleado=$idempleado AND status=1 AND NOT (fecha_hasta < '$desdeMes' OR fecha_desde > '$hastaMes')
           ORDER BY fecha_desde";
      if($rs=mysqli_query($con,$qv)){ while($r=mysqli_fetch_assoc($rs)) $vac[]=$r; mysqli_free_result($rs); }
      $qi="SELECT id, fecha_inicio, fecha_fin, motivo
           FROM ingreso_incapacidades
           WHERE idempleado=$idempleado AND NOT (fecha_fin < '$desdeMes' OR fecha_inicio > '$hastaMes')
           ORDER BY fecha_inicio";
      if($rs=mysqli_query($con,$qi)){ while($r=mysqli_fetch_assoc($rs)) $inc[]=$r; mysqli_free_result($rs); }
      ?>

      <?php if(count($vac)): ?>
        <h3 style="margin-top:18px">Vacaciones activas (mes)</h3>
        <table class="tbl">
          <thead><tr><th>Desde</th><th>Hasta</th><th>Días</th><th>Motivo</th></tr></thead>
          <tbody>
            <?php foreach($vac as $v): ?>
              <tr>
                <td><?= htmlspecialchars($v['fecha_desde']) ?></td>
                <td><?= htmlspecialchars($v['fecha_hasta']) ?></td>
                <td style="text-align:right"><?= number_format((float)($v['cantidad_dias']??0),2,'.',',') ?></td>
                <td><?= htmlspecialchars($v['motivo']??'') ?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      <?php endif; ?>

      <?php if(count($inc)): ?>
        <h3 style="margin-top:18px">Incapacidades en el mes</h3>
        <table class="tbl">
          <thead><tr><th>Desde</th><th>Hasta</th><th>Motivo</th></tr></thead>
          <tbody>
            <?php foreach($inc as $i): ?>
              <tr>
                <td><?= htmlspecialchars($i['fecha_inicio']) ?></td>
                <td><?= htmlspecialchars($i['fecha_fin']) ?></td>
                <td><?= htmlspecialchars($i['motivo']??'') ?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      <?php endif; ?>

    </div>
  </div>
</div>

<script>
// --- AJAX recalc (sin recargar) ---
function fmt(n){ return (Number(n||0)).toLocaleString('en-US',{minimumFractionDigits:2, maximumFractionDigits:2}); }
function fmt0(n){ return (Number(n||0)).toLocaleString('en-US',{maximumFractionDigits:0}); }

function gatherParams(){
  const f = document.getElementById('polillaForm');
  const params = new URLSearchParams();
  params.set('ajax','1');
  params.set('idempleado', document.getElementById('idempleado').value);
  params.set('mes', document.getElementById('mes').value);
  params.set('modo', (new FormData(f)).get('modo') || 'mes');
  params.set('inc_extras', document.getElementById('inc_extras').checked ? '1':'0');
  params.set('inc_aus',    document.getElementById('inc_aus').checked    ? '1':'0');
  return params;
}

async function recalc(){
  const p = gatherParams();
  const url = 'polilla_html.php?'+p.toString();
  const res = await fetch(url, {headers:{'Accept':'application/json'}});
  const r = await res.json();
  if(!r || !r.ok) return;

  // Encabezado
  document.getElementById('lblPeriodo').textContent = r.labelPeriodo;
  document.getElementById('lblPeriodo2').textContent = r.labelPeriodo;
  document.getElementById('rangoDesde').textContent = r.desde;
  document.getElementById('rangoHasta').textContent = r.hasta;
  document.getElementById('salPeriodo').textContent = fmt(r.salario_periodo);
  document.getElementById('salHora').textContent    = fmt(r.sal_h);

  // Extras / Ausencias
  document.getElementById('h15').textContent    = fmt(r.h15);
  document.getElementById('h20').textContent    = fmt(r.h20);
  document.getElementById('h30').textContent    = fmt(r.h30);
  document.getElementById('mExtras').textContent= fmt(r.mExtras);

  document.getElementById('hAus').textContent   = fmt(r.horasAus);
  document.getElementById('hRet').textContent   = fmt(r.horasRet);
  document.getElementById('hAusTot').textContent= fmt(r.horasAusTot);
  document.getElementById('mAus').textContent   = fmt(r.descAus);

  // Base
  document.getElementById('baseGrav').textContent= fmt(r.baseGrav);
  document.getElementById('setLegal').textContent= r.setLegal || '—';

  // Deducciones tabla
  const tbody = document.getElementById('dedBody');
  let rows = '';
  (r.dedetalle||[]).forEach(d=>{
    rows += `<tr>
      <td>${d.nombre}</td>
      <td style="text-align:right">${fmt(d.tasa)}%</td>
      <td style="text-align:right">₡${fmt(d.monto)}</td>
    </tr>`;
  });
  rows += `<tr>
    <td><b>Base para ISR (después de cargas)</b></td>
    <td></td>
    <td style="text-align:right"><b>₡<span id="baseISR">${fmt(r.baseISR)}</span></b></td>
  </tr>`;

  // Nota ISR (tramo usado y desglose)
  let isrNote = 'Sin ISR en el período.';
  if ((r.isr_aplicados||[]).length){
    const top = r.isr_top ? (Number(r.isr_top.pct).toFixed(2)+'%') : '—';
    const pieces = (r.isr_aplicados||[]).map(ap=>{
      const desdeTxt = '₡'+fmt0(ap.desde);
      const hastaTxt = (ap.hasta===null || ap.hasta===undefined) ? '∞' : ('₡'+fmt0(ap.hasta));
      return `${desdeTxt}–${hastaTxt} @ ${fmt(ap.pct)}% → ₡${fmt(ap.monto)}`;
    });
    isrNote = `Tramo superior aplicado: ${top}; Detalle: ${pieces.join(' | ')}`;
  }
  rows += `<tr>
    <td><b>Impuesto sobre la Renta (ISR)</b><br><small id="isrNote">${isrNote}</small></td>
    <td style="text-align:right">—</td>
    <td style="text-align:right"><b>₡<span id="isrVal">${fmt(r.isr)}</span></b></td>
  </tr>`;
  rows += `<tr>
    <td><b>Total deducciones</b></td>
    <td></td>
    <td style="text-align:right"><b>₡<span id="dedTot">${fmt(r.dedTot)}</span></b></td>
  </tr>`;
  tbody.innerHTML = rows;

  // Neto
  document.getElementById('netoVal').textContent = fmt(r.neto);
}

// Listeners: radio/checkbox/mes
document.querySelectorAll('#polillaForm input[name="modo"]').forEach(r=>{
  r.addEventListener('change', recalc);
});
document.getElementById('inc_extras').addEventListener('change', recalc);
document.getElementById('inc_aus').addEventListener('change', recalc);
document.getElementById('mes').addEventListener('change', recalc);
</script>

<?php include __DIR__.'/view/footer.html';
