<?php
// model/HomeRRHH.php
// Archivo limpio (ASCII).

// ---- Carga de conexión (tolerante a distintas rutas/nombres) ----
$conexion_cargada = false;
$posibles = array(
    __DIR__ . '/Conexion.php',
    __DIR__ . '/conexion.php',
    __DIR__ . '/../Conexion.php',
    __DIR__ . '/../conexion.php',
    __DIR__ . '/../dbcon.php'
);
foreach ($posibles as $p) {
    if (file_exists($p)) {
        require_once $p;
        $conexion_cargada = true;
        break;
    }
}
if (!$conexion_cargada) {
    die('No se encontró archivo de conexión (Conexion.php / conexion.php / dbcon.php).');
}

// Normaliza el handler de base de datos a $db (mysqli)
$db = null;
if (isset($conexion) && $conexion instanceof mysqli) {
    $db = $conexion;
} elseif (isset($con) && $con instanceof mysqli) {
    $db = $con;
} else {
    die('No hay conexión mysqli disponible en $conexion o $con.');
}

// Compatibilidad para intdiv en PHP < 7
if (!function_exists('intdiv')) {
    function intdiv($a, $b) {
        return ($a - ($a % $b)) / $b;
    }
}

class HomeRRHH {

    /**
     * Vacaciones (y vacaciones extra) que se traslapan con un rango.
     * IMPORTANTE: ahora devuelve también `cantidad_dias` EXACTO desde BD.
     */
    public function VacacionesEnRango($inicio, $fin) {
        global $db;

        $inicio = $db->real_escape_string($inicio);
        $fin    = $db->real_escape_string($fin);

        // ingreso_vacaciones: incluye cantidad_dias
        $sql1 = "
            SELECT
                v.id,
                v.idempleado,
                e.nombre_completo,
                e.categoria,
                v.fecha_desde,
                v.fecha_hasta,
                v.cantidad_dias,
                'vacaciones' AS tipo
            FROM ingreso_vacaciones v
            INNER JOIN empleados_planilla e ON e.id = v.idempleado
            WHERE v.status='1' AND v.fecha_desde <= '$fin' AND v.fecha_hasta >= '$inicio'
        ";

        // ingreso_vacaciones_extra: incluye cantidad_dias
        $sql2 = "
            SELECT
                vx.id,
                vx.idempleado,
                e.nombre_completo,
                e.categoria,
                vx.fecha_desde,
                vx.fecha_hasta,
                vx.cantidad_dias,
                'vacaciones_extra' AS tipo
            FROM ingreso_vacaciones_extra vx
            INNER JOIN empleados_planilla e ON e.id = vx.idempleado
            WHERE vx.fecha_desde <= '$fin' AND vx.fecha_hasta >= '$inicio'
        ";

        $sql = $sql1 . " UNION ALL " . $sql2 . " ORDER BY fecha_desde ASC";
        return $db->query($sql);
    }

    /**
     * Próximos hitos de 50 semanas (350 días) dentro de una ventana en días.
     */
    public function ProximosCincuentaSemanas($limiteDias = 90) {
        global $db;

        $hoy = new DateTime(date('Y-m-d'));
        $hoy->setTime(0, 0, 0);
        $hasta = clone $hoy;
        $hasta->modify('+' . (int)$limiteDias . ' day');

        $res = $db->query("
            SELECT id, nombre_completo, categoria, year_ingreso, mes_ingreso, dia_ingreso
            FROM empleados_planilla
        ");

        $out = array();
        if ($res) {
            while ($r = $res->fetch_object()) {
                $anio = (int)$r->year_ingreso;
                $mes  = (int)$r->mes_ingreso;
                $dia  = (int)$r->dia_ingreso;

                $ingreso = DateTime::createFromFormat('Y-n-j', sprintf('%d-%d-%d', $anio, $mes, $dia));
                if (!$ingreso) { continue; }
                $ingreso->setTime(0, 0, 0);

                // 50 semanas = 350 días
                $diffDias = (int)$ingreso->diff($hoy)->days;
                $ciclosCompletos = intdiv($diffDias, 350);

                $proximoHito = clone $ingreso;
                $proximoHito->modify('+' . (350 * ($ciclosCompletos + 1)) . ' day');

                if ($proximoHito >= $hoy && $proximoHito <= $hasta) {
                    $faltan = (int)$hoy->diff($proximoHito)->days;
                    $out[] = array(
                        'idempleado' => (int)$r->id,
                        'nombre'     => $r->nombre_completo,
                        'categoria'  => $r->categoria,
                        'fecha_hito' => $proximoHito->format('Y-m-d'),
                        'faltan_dias'=> $faltan
                    );
                }
            }
        }

        // Orden ascendente por días faltantes
        usort($out, function($a, $b) {
            if ($a['faltan_dias'] == $b['faltan_dias']) return 0;
            return ($a['faltan_dias'] < $b['faltan_dias']) ? -1 : 1;
        });

        return $out;
    }

    /**
     * Top N de empleados con mayor saldo de vacaciones.
     */
    public function TopSaldoVacaciones($topN = 10) {
        global $db;

        $hoy = new DateTime(date('Y-m-d'));
        $hoy->setTime(0, 0, 0);

        $sql = "
            SELECT
              c.*,
              c.id AS idempleado,
              IFNULL((SELECT SUM(a.cantidad_dias) FROM ingreso_vacaciones a WHERE a.status='1' AND a.idempleado = c.id), 0) AS dias_solicitados,
              IFNULL((SELECT SUM(e.dias_calculo)   FROM ingreso_inscapacidad e WHERE e.idempleado = c.id), 0) AS dias_inscapacidad_cal,
              IFNULL((SELECT SUM(d.cantidad_dias) FROM ingreso_vacaciones_extra d WHERE d.idempleado = c.id), 0) AS dias_gastados
            FROM empleados_planilla c
        ";
        $res = $db->query($sql);

        $out = array();
        if ($res) {
            while ($reg = $res->fetch_object()) {
                $ing = DateTime::createFromFormat(
                    'Y-n-j',
                    sprintf('%d-%d-%d', (int)$reg->year_ingreso, (int)$reg->mes_ingreso, (int)$reg->dia_ingreso)
                );
                if (!$ing) { continue; }
                $ing->setTime(0, 0, 0);

                $diasTrab = (int)$ing->diff($hoy)->days;
                $diasVacBase = $diasTrab / 35.0;

                // Si dias_calculo son días calendario, convertir a unidades de vacaciones:
                $diasIns = ((float)$reg->dias_inscapacidad_cal) / 35.0;
                // Si tu campo ya viene convertido a unidades, reemplaza la línea anterior por:
                // $diasIns = (float)$reg->dias_inscapacidad_cal;

                $total_vac = $diasVacBase + (float)$reg->sumar - (float)$reg->restar;
                $saldo1 = $total_vac + (float)$reg->dias_gastados - (float)$reg->dias_solicitados - $diasIns;
                $saldo = floor($saldo1);
                if ($saldo < 0) { $saldo = 0.0; }

                $out[] = array(
                    'idempleado' => (int)$reg->idempleado,
                    'nombre'     => $reg->nombre_completo,
                    'categoria'  => $reg->categoria,
                    'saldo'      => round($saldo, 2)
                );
            }
        }

        usort($out, function($a, $b) {
            if ($a['saldo'] == $b['saldo']) return 0;
            return ($a['saldo'] > $b['saldo']) ? -1 : 1; // desc
        });

        return array_slice($out, 0, (int)$topN);
    }
}

