<?php
/**
 * json_utf8_safe_strict.php
 * - No borra caracteres; los "control chars" ASCII 0x00-0x1F se ESCAPAN como \u00XX
 * - Convierte a UTF-8 cuando detecta ISO-8859-1/Windows-1252 (no pierde tildes/ñ)
 * - Usa JSON_INVALID_UTF8_SUBSTITUTE para que bytes inválidos se sustituyan por U+FFFD (�) sin romper el JSON
 * - Exporta json_utf8_strict($data, $flags = JSON_UNESCAPED_UNICODE|JSON_INVALID_UTF8_SUBSTITUTE)
 */

declare(strict_types=1);

if (!function_exists('utf8_force')) {
  function utf8_force($value) {
    if (is_array($value)) {
      foreach ($value as $k=>$v) $value[$k] = utf8_force($v);
      return $value;
    }
    if (is_object($value)) {
      foreach ($value as $k=>$v) $value->$k = utf8_force($v);
      return $value;
    }
    if (!is_string($value)) return $value;

    // Normaliza saltos de línea
    $value = str_replace(["\r\n","\r"], "\n", $value);

    // Si no es UTF-8 válido, intenta convertir desde ISO-8859-1/Windows-1252
    if (!mb_check_encoding($value, 'UTF-8')) {
      $value = mb_convert_encoding($value, 'UTF-8', 'UTF-8, ISO-8859-1, Windows-1252');
    }
    return $value;
  }
}

if (!function_exists('escape_control_chars')) {
  function escape_control_chars(string $s): string {
    // Escapa 0x00-0x1F en forma \u00XX; mantiene \n y \t si deseas mostrarlos textual
    // Si prefieres también escaparlos, quita las dos condiciones de \n y \t
    $out = '';
    $len = strlen($s);
    for ($i=0; $i<$len; $i++) {
      $ord = ord($s[$i]);
      if ($ord < 32) {
        if ($s[$i] === "\n") { $out .= "\\n"; continue; }
        if ($s[$i] === "\t") { $out .= "\\t"; continue; }
        $out .= sprintf("\\u%04X", $ord);
      } else {
        // Escapa backslash para no romper el JSON más adelante
        if ($s[$i] === "\\") { $out .= "\\\\"; }
        else { $out .= $s[$i]; }
      }
    }
    return $out;
  }
}

if (!function_exists('utf8_escape_controls_recursive')) {
  function utf8_escape_controls_recursive($value) {
    if (is_array($value)) {
      foreach ($value as $k=>$v) $value[$k] = utf8_escape_controls_recursive($v);
      return $value;
    }
    if (is_object($value)) {
      foreach ($value as $k=>$v) $value->$k = utf8_escape_controls_recursive($v);
      return $value;
    }
    if (!is_string($value)) return $value;
    return escape_control_chars($value);
  }
}

if (!function_exists('json_utf8_strict')) {
  function json_utf8_strict($data, int $flags = (JSON_UNESCAPED_UNICODE|JSON_INVALID_UTF8_SUBSTITUTE)) : string {
    $clean = utf8_force($data);
    $clean = utf8_escape_controls_recursive($clean);
    $json = json_encode($clean, $flags);
    if ($json === false) {
      error_log('JSON strict error: '.json_last_error_msg());
      http_response_code(500);
      return '{"ok":false,"error":"json_encode failed"}';
    }
    return $json;
  }
}
