<?php
/**
 * json_utf8_safe.php — Emite JSON seguro con tildes y ñ en PHP 8.1+
 * - Limpia caracteres de control invisibles que rompen json_encode
 * - Normaliza a UTF-8 (convierte desde ISO-8859-1/Windows-1252 si hace falta)
 * - Proporciona json_utf8($data) para output
 */
declare(strict_types=1);

if (!function_exists('utf8_sanitize')) {
  function utf8_sanitize($value) {
    if (is_array($value)) {
      foreach ($value as $k => $v) $value[$k] = utf8_sanitize($v);
      return $value;
    }
    if (is_object($value)) {
      foreach ($value as $k => $v) $value->$k = utf8_sanitize($v);
      return $value;
    }
    if (is_string($value)) {
      // Quitar caracteres de control 0x00–0x1F (excepto \n si quieres conservarlo)
      $value = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F]/u', ' ', $value);
      // Normalizar saltos de línea
      $value = str_replace(["\r\n", "\r"], "\n", $value);
      // Asegurar UTF-8 válido
      if (!mb_check_encoding($value, 'UTF-8')) {
        $value = mb_convert_encoding($value, 'UTF-8', 'UTF-8, ISO-8859-1, Windows-1252');
      }
    }
    return $value;
  }
}

if (!function_exists('iconv_recursive')) {
  function iconv_recursive($val, string $to, string $from) {
    if (is_array($val))  { foreach ($val as $k=>$v) $val[$k]  = iconv_recursive($v,$to,$from); return $val; }
    if (is_object($val)) { foreach ($val as $k=>$v) $val->$k = iconv_recursive($v,$to,$from); return $val; }
    if (is_string($val)) { $r = @iconv($from, $to, $val); return ($r === false) ? $val : $r; }
    return $val;
  }
}

if (!function_exists('json_utf8')) {
  function json_utf8($data, int $flags = JSON_UNESCAPED_UNICODE) : string {
    $clean = utf8_sanitize($data);
    $json  = json_encode($clean, $flags);
    if ($json === false) {
      // Último recurso: forzar conversión
      $clean = iconv_recursive($clean, 'UTF-8//TRANSLIT', 'UTF-8');
      $json  = json_encode($clean, $flags);
    }
    if ($json === false) {
      error_log('JSON error: '.json_last_error_msg());
      http_response_code(500);
      return '{"ok":false,"error":"json_encode failed"}';
    }
    return $json;
  }
}
