<?php
// rrhh_home/lib/payroll_legal.php
// Helpers para cargar el conjunto activo y aplicar tasas a salario/horas extra

declare(strict_types=1);

function payroll_get_active_set(mysqli $con): ?array {
  $q = "SELECT id, nombre FROM payroll_legal_set WHERE activo=1 LIMIT 1";
  if ($rs = mysqli_query($con,$q)) {
    $row = mysqli_fetch_assoc($rs);
    mysqli_free_result($rs);
    return $row ?: null;
  }
  return null;
}

function payroll_get_rates(mysqli $con, int $set_id): array {
  $out = ['EMPLOYEE'=>[], 'EMPLOYER'=>[]];
  $q = "SELECT id, code, nombre, side, porcentaje, base, obligatorio
        FROM payroll_legal_rate
        WHERE id_set=? ORDER BY side, id";
  if ($st = mysqli_prepare($con,$q)) {
    mysqli_stmt_bind_param($st,'i',$set_id);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    while($r = mysqli_fetch_assoc($rs)){
      $out[$r['side']][] = $r;
    }
    mysqli_stmt_close($st);
  }
  return $out;
}

/**
 * Calcula los montos de cargas para salario base y horas extra.
 * $bases:
 *  - 'salario' => salario bruto del mes (sin extras)
 *  - 'extras' => monto de horas extra del mes (bruto extras)
 * Devuelve:
 *  - 'employee' => total y desglose
 *  - 'employer' => total y desglose
 */
function payroll_apply_rates(array $rates, float $salario, float $extras): array {
  $calc = [
    'employee'=>['total'=>0.0,'items'=>[]],
    'employer'=>['total'=>0.0,'items'=>[]],
  ];
  foreach (['EMPLOYEE'=>'employee','EMPLOYER'=>'employer'] as $sideDb => $sideOut) {
    foreach ($rates[$sideDb] as $r) {
      $pct = (float)$r['porcentaje'] / 100.0;
      $base = 0.0;
      switch ($r['base']) {
        case 'SALARIO': $base = $salario; break;
        case 'EXTRAS' : $base = $extras;  break;
        default       : $base = $salario + $extras; break; // AMBOS
      }
      $monto = round($base * $pct, 2);
      $calc[$sideOut]['items'][] = [
        'code'=>$r['code'],
        'nombre'=>$r['nombre'],
        'porcentaje'=>(float)$r['porcentaje'],
        'base'=>$r['base'],
        'monto'=>$monto,
      ];
      $calc[$sideOut]['total'] += $monto;
    }
    // redondeo final
    $calc[$sideOut]['total'] = round($calc[$sideOut]['total'],2);
  }
  return $calc;
}

/**
 * ISR mensual (método general por tramos). Devuelve monto ISR.
 * Si no hay tramos cargados, retorna 0.
 */
function payroll_isr_mensual(mysqli $con, int $set_id, float $base_mensual): float {
  $q = "SELECT base_min, base_max, tasa, deduccion FROM payroll_isr_tramo
        WHERE id_set=? ORDER BY orden ASC";
  $isr = 0.0;
  if ($st = mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'i',$set_id);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    while ($r = mysqli_fetch_assoc($rs)) {
      $min = (float)$r['base_min'];
      $max = is_null($r['base_max']) ? INF : (float)$r['base_max'];
      if ($base_mensual >= $min && $base_mensual <= $max) {
        $isr = round(($base_mensual * ((float)$r['tasa']/100.0)) - (float)$r['deduccion'], 2);
        if ($isr < 0) $isr = 0.0;
        break;
      }
    }
    mysqli_stmt_close($st);
  }
  return $isr;
}
