<?php
// index.php
declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');

// Si ya está logueado, manda a acceder.php
if (isset($_SESSION['idusuario']) && (int)$_SESSION['idusuario'] > 0) {
  header('Location: acceder.php'); exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta http-equiv="X-UA-Compatible" content="IE=edge">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>SISTEMA POM V1.1</title>

  <!-- CSS -->
  <link rel="stylesheet" href="public/css/bootstrap.min.css">
  <link rel="stylesheet" href="public/css/font-awesome.min.css">
  <link rel="stylesheet" href="public/css/form-elements.css">
  <link rel="stylesheet" href="public/css/style.css">

  <!-- Favicon and touch icons -->
  <link rel="shortcut icon" href="ico/favicon.png">
  <link rel="apple-touch-icon-precomposed" sizes="144x144" href="ico/apple-touch-icon-144-precomposed.png">
  <link rel="apple-touch-icon-precomposed" sizes="114x114" href="ico/apple-touch-icon-114-precomposed.png">
  <link rel="apple-touch-icon-precomposed" sizes="72x72" href="ico/apple-touch-icon-72-precomposed.png">
  <link rel="apple-touch-icon-precomposed" href="ico/apple-touch-icon-57-precomposed.png">

  <style>
    .cam-wrap { text-align:center; }
    #videoCam { width: 100%; max-width: 420px; border-radius: 12px; background: #111; }
    #snapImg  { width: 100%; max-width: 420px; border-radius: 12px; display:none; }
    .gps-pill { display:inline-block; padding:6px 10px; border-radius:999px; background:#f2f2f2; margin-top:8px; font-size:12px;}
    .evi-status { margin-top:10px; font-size:13px; }
    .evi-status .ok { color:#1e7e34; font-weight:600; }
    .evi-status .bad{ color:#b21f2d; font-weight:600; }
    .evi-detail pre{
      text-align:left;
      max-height:200px;
      overflow:auto;
      background:#f7f7f7;
      border:1px solid #ddd;
      padding:8px;
      border-radius:8px;
      font-size:12px;
      margin-top:8px;
    }
  </style>
</head>

<body>

<!-- Top content -->
<div class="top-content">
  <div class="inner-bg">
    <div class="container">
      <div class="row">
        <div class="col-sm-8 col-sm-offset-2 text"></div>
      </div>

      <div class="row">
        <div class="col-sm-6 col-sm-offset-3 form-box">
          <div class="form-top">
            <div class="form-top-left">
              <img src="assets/img/backgrounds/logo.png" alt="Imagen">
              <h3>Acceder a tu cuenta POM Administrativo)</h3>
              <p style="color:#FFF">Ingresa usuario y clave:</p>
            </div>
            <div class="form-top-right">
              <i class="fa fa-lock"></i>
            </div>
          </div>

          <div class="form-bottom">
            <form role="form" id="frmIngresar" name="frmIngresar" method="post" class="login-form" autocomplete="off">
              <div class="form-group">
                <label class="sr-only" for="form-username">Usuario</label>
                <input type="text" name="form-username" placeholder="Escribe tu usuario" class="form-username form-control" id="form-username">
              </div>
              <div class="form-group">
                <label class="sr-only" for="form-password">Clave</label>
                <input type="password" name="form-password" placeholder="Escribe la clave" class="form-password form-control" id="form-password">
              </div>
              <button type="submit" class="btn" id="btnIngresar">Ingresar</button>

              <div style="margin-top:10px; color:#fff; font-size:12px; opacity:.9;">
                * Al ingresar se solicitará <b>ubicación</b> y <b>foto</b> para registrar evidencia.
              </div>
            </form>
          </div>

        </div>
      </div>
    </div>
  </div>
</div>

<!-- MODAL CAMARA/GPS -->
<div class="modal fade" id="modalEvidencia" tabindex="-1" role="dialog" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">

      <div class="modal-header">
        <button type="button" class="close" data-dismiss="modal" style="display:none;"><span>&times;</span></button>
        <h4 class="modal-title">Evidencia de ingreso</h4>
      </div>

      <div class="modal-body">
        <div class="cam-wrap">
          <video id="videoCam" playsinline autoplay></video>
          <img id="snapImg" alt="Foto tomada">
          <canvas id="snapCanvas" style="display:none;"></canvas>

          <div class="gps-pill" id="gpsStatus">GPS: pendiente…</div>

          <div class="evi-status" id="eviStatus">Preparando permisos…</div>
          <div class="evi-detail" id="eviDetail"></div>
        </div>

        <hr>
        <div class="alert alert-info" style="margin-bottom:0;">
          Se realizará el proceso automáticamente (GPS + Foto + Login).<br>
          Si el navegador bloquea permisos, presiona <b>Reintentar</b>.
        </div>
      </div>

      <div class="modal-footer">
        <button type="button" class="btn btn-default" id="btnReintentar">Reintentar</button>
      </div>

    </div>
  </div>
</div>

<!-- Javascript -->
<script type="text/javascript" src="public/js/jquery.min.js"></script>
<script type="text/javascript" src="public/js/bootstrap.min.js"></script>
<script type="text/javascript" src="public/js/jquery.backstretch.min.js"></script>
<script type="text/javascript" src="public/js/scripts.js"></script>
<script type="text/javascript" src="public/js/bootbox.min.js"></script>

<script>
jQuery(function($){

  // Endpoints
  var LOGIN_URL  = "/ajax/UsuarioAjax.php";
  var EVID_URL   = "/ajax/LoginEvidenciaAjax.php";
  var LOGOUT_URL = "/ajax/logout.php";

  var pendingUser = null;
  var pendingPass = null;

  var stream = null;
  var gps = { lat:null, lng:null, accuracy:null };
  var photoBlob = null;
  var autoRunning = false;

  function setGpsStatus(txt){ $("#gpsStatus").text(txt); }
  function setEviStatus(html){ $("#eviStatus").html(html); }
  function setEviDetail(html){ $("#eviDetail").html(html || ""); }

  function resetState(){
    gps = { lat:null, lng:null, accuracy:null };
    photoBlob = null;
    autoRunning = false;
    setGpsStatus("GPS: pendiente…");
    setEviStatus("Preparando permisos…");
    setEviDetail("");
    $("#snapImg").hide();
    $("#videoCam").show();
  }

  function stopCamera(){
    if(stream){
      try { stream.getTracks().forEach(t=>t.stop()); } catch(e){}
      stream = null;
    }
  }

  async function startCamera(){
    try{
      stopCamera();
      stream = await navigator.mediaDevices.getUserMedia({ video: { facingMode: "user" }, audio:false });
      var video = document.getElementById('videoCam');
      video.srcObject = stream;
      return true;
    }catch(e){
      return false;
    }
  }

  function waitForVideoReady(timeoutMs){
    timeoutMs = timeoutMs || 8000;
    return new Promise(function(resolve){
      var video = document.getElementById('videoCam');
      var t0 = Date.now();

      if (video.readyState >= 2 && video.videoWidth > 0) return resolve(true);

      var timer = setInterval(function(){
        if (video.readyState >= 2 && video.videoWidth > 0){
          clearInterval(timer);
          resolve(true);
        } else if (Date.now() - t0 > timeoutMs){
          clearInterval(timer);
          resolve(false);
        }
      }, 120);
    });
  }

  function getGps(){
    return new Promise(function(resolve){
      if(!navigator.geolocation){
        setGpsStatus("GPS: no soportado");
        return resolve(false);
      }
      setGpsStatus("GPS: solicitando permiso…");
      navigator.geolocation.getCurrentPosition(function(pos){
        gps.lat = pos.coords.latitude;
        gps.lng = pos.coords.longitude;
        gps.accuracy = Math.round(pos.coords.accuracy || 0);
        setGpsStatus("GPS: OK ("+gps.accuracy+"m aprox.)");
        resolve(true);
      }, function(){
        setGpsStatus("GPS: denegado / error");
        resolve(false);
      }, { enableHighAccuracy:true, timeout:12000, maximumAge:0 });
    });
  }

  function takeSnapshotAsync(){
    return new Promise(function(resolve){
      var video = document.getElementById('videoCam');
      var canvas = document.getElementById('snapCanvas');
      var img = document.getElementById('snapImg');

      var w = video.videoWidth || 640;
      var h = video.videoHeight || 480;

      canvas.width = w;
      canvas.height = h;

      var ctx = canvas.getContext('2d');
      ctx.drawImage(video, 0, 0, w, h);

      canvas.toBlob(function(blob){
        if(!blob) return resolve(null);

        photoBlob = blob;
        var url = URL.createObjectURL(blob);
        img.src = url;

        $("#videoCam").hide();
        $("#snapImg").show();

        resolve(blob);
      }, "image/jpeg", 0.85);
    });
  }

  function doLogin(){
    return $.ajax({
      url: LOGIN_URL,
      type: "GET",
      dataType: "json",
      data: { op: "IngresarSistema", user: pendingUser, pass: pendingPass }
    });
  }

  function saveEvidence(){
    var fd = new FormData();
    fd.append("lat", gps.lat);
    fd.append("lng", gps.lng);
    fd.append("accuracy", gps.accuracy || "");
    fd.append("photo", photoBlob, "login.jpg");

    return $.ajax({
      url: EVID_URL,
      type: "POST",
      data: fd,
      processData: false,
      contentType: false,
      dataType: "json"
    });
  }

  async function forceLogoutAndBack(msgHtml, detailHtml){
    try{
      await $.ajax({ url: LOGOUT_URL, type: "POST", dataType: "json" });
    }catch(e){ /* no importa */ }

    stopCamera();
    autoRunning = false;

    setEviStatus(msgHtml || "<span class='bad'>Error.</span> Cerrando sesión…");
    setEviDetail(detailHtml || "");

    setTimeout(function(){
      window.location.href = "index.php";
    }, 1000);
  }

  async function runAutoFlow(){
    if(autoRunning) return;
    autoRunning = true;

    setEviStatus("Solicitando permisos de <b>cámara</b> y <b>ubicación</b>…");
    setEviDetail("");

    var camOk = await startCamera();
    var gpsOk = await getGps();

    if(!camOk || !gpsOk){
      autoRunning = false;
      var msg = "";
      if(!camOk) msg += "❌ Cámara no disponible / permisos denegados.<br>";
      if(!gpsOk) msg += "❌ GPS no disponible / permisos denegados.<br>";
      msg += "<span class='bad'>No puedo continuar automático.</span> Presiona <b>Reintentar</b> y acepta los permisos.";
      setEviStatus(msg);
      stopCamera();
      return;
    }

    setEviStatus("Permisos OK. Preparando captura automática…");

    var videoReady = await waitForVideoReady(8000);
    if(!videoReady){
      autoRunning = false;
      setEviStatus("<span class='bad'>No se pudo inicializar video.</span> Presiona <b>Reintentar</b>.");
      stopCamera();
      return;
    }

    await new Promise(r => setTimeout(r, 900));

    setEviStatus("📸 Tomando foto automáticamente…");
    var blob = await takeSnapshotAsync();
    if(!blob){
      autoRunning = false;
      setEviStatus("<span class='bad'>No se pudo tomar la foto.</span> Presiona <b>Reintentar</b>.");
      stopCamera();
      return;
    }

    setEviStatus("✅ Foto lista. Iniciando sesión…");

    // 1) LOGIN
    try{
      var r = await doLogin();
      if(!r){
        autoRunning = false;
        stopCamera();
        setEviStatus("<span class='bad'>Usuario o clave incorrecta.</span> Corrige y vuelve a intentar.");
        setEviDetail("");
        return;
      }
    }catch(xhr){
      autoRunning = false;
      stopCamera();
      var detail = "<pre>"+(xhr.responseText || ("HTTP " + (xhr.status || '')))+"</pre>";
      setEviStatus("<span class='bad'>No se pudo contactar el servidor (login).</span>");
      setEviDetail(detail);
      return;
    }

    // 2) GUARDAR EVIDENCIA (SI FALLA => LOGOUT)
    setEviStatus("Guardando evidencia (GPS + foto)…");

    var resp = null;
    try{
      resp = await saveEvidence();
    }catch(xhr){
      var detail = "";
      try{
        detail = "<pre>HTTP " + (xhr.status || '') + "\n" + (xhr.responseText || '') + "</pre>";
      }catch(e){}

      await forceLogoutAndBack(
        "<span class='bad'>Error guardando evidencia.</span><br>Se cerrará la sesión por seguridad…",
        detail
      );
      return;
    }

    if(resp && resp.ok){
      setEviStatus("<span class='ok'>Evidencia guardada. Entrando al sistema…</span>");
      stopCamera();
      $("#modalEvidencia").modal("hide");
      window.location.href = "acceder.php";
      return;
    } else {
      await forceLogoutAndBack(
        "<span class='bad'>Login OK, pero evidencia falló.</span><br>Se cerrará la sesión por seguridad…",
        (resp && resp.msg) ? ("<pre>"+resp.msg+"</pre>") : ""
      );
      return;
    }
  }

  async function openModalAndRun(){
    resetState();
    $("#modalEvidencia").modal({ backdrop: 'static', keyboard: false });
    await runAutoFlow();
  }

  $("#frmIngresar").on("submit", async function(e){
    e.preventDefault();

    var user = $("#form-username").val().trim();
    var pass = $("#form-password").val();

    if(!user || !pass){
      bootbox.alert("Debes ingresar usuario y clave.");
      return;
    }

    pendingUser = user;
    pendingPass = pass;

    await openModalAndRun();
  });

  $("#btnReintentar").on("click", async function(){
    await openModalAndRun();
  });

  $("#modalEvidencia").on("hidden.bs.modal", function(){
    stopCamera();
    autoRunning = false;
  });

});
</script>

</body>
</html>
