<?php
// rrhh_home/horario_laboral.php
// Configuración global de horario laboral + cafés/almuerzo (horas BRUTAS).
// Valida 48 horas semanales (bloquea guardar si excede).
// PHP 8.1.33 – UTF-8
// Source - https://stackoverflow.com/q
// Posted by Abs, modified by community. See post 'Timeline' for change history
// Retrieved 2026-01-29, License - CC BY-SA 4.0

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');

if (!isset($_SESSION["idusuario"])) { header("Location:index.html"); exit; }
$ES_SUPER = isset($_SESSION["superadmin"]) && $_SESSION["superadmin"] === "S";

require_once __DIR__ . '/dbcon.php';
mysqli_set_charset($con,'utf8mb4');

function h(string $s): string { return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }

function dow_label(int $n): string {
  return match($n){
    1=>'Lunes',2=>'Martes',3=>'Miércoles',4=>'Jueves',5=>'Viernes',6=>'Sábado',7=>'Domingo',
    default=>'Día'
  };
}

function norm_time(?string $s): ?string {
  $s = trim((string)$s);
  if ($s === '') return null;

  // acepta HH:MM o HH:MM:SS
  if (preg_match('/^\d{2}:\d{2}$/', $s)) $s .= ':00';
  if (!preg_match('/^\d{2}:\d{2}:\d{2}$/', $s)) return null;

  $dt = DateTime::createFromFormat('H:i:s', $s);
  if (!$dt) return null;
  return $dt->format('H:i:s');
}

function minutes_between(?string $start, ?string $end): int {
  if (!$start || !$end) return 0;
  $base = '2000-01-01 ';
  $a = strtotime($base.$start);
  $b = strtotime($base.$end);
  if ($a === false || $b === false) return 0;
  $diff = (int)floor(($b - $a) / 60);
  return max(0, $diff);
}

function within_range(?string $t, ?string $start, ?string $end): bool {
  if (!$t || !$start || !$end) return false;
  $base = '2000-01-01 ';
  $x = strtotime($base.$t);
  $a = strtotime($base.$start);
  $b = strtotime($base.$end);
  if ($x===false || $a===false || $b===false) return false;
  return ($x >= $a && $x <= $b);
}

// Asegura tabla (idempotente)
@mysqli_query($con, "
CREATE TABLE IF NOT EXISTS work_schedule (
  day_of_week TINYINT NOT NULL PRIMARY KEY,
  active TINYINT NOT NULL DEFAULT 1,
  start_time TIME NULL,
  end_time TIME NULL,
  late_grace_minutes INT NOT NULL DEFAULT 15,
  early_grace_minutes INT NOT NULL DEFAULT 5,
  lunch_start TIME NULL,
  lunch_end TIME NULL,
  coffee1_start TIME NULL,
  coffee1_end TIME NULL,
  coffee2_start TIME NULL,
  coffee2_end TIME NULL,
  break_window_before_min INT NOT NULL DEFAULT 10,
  break_window_after_min INT NOT NULL DEFAULT 10,
  updated_at DATETIME NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
");

// Carga actual
$rows = [];
$q = "SELECT * FROM work_schedule ORDER BY day_of_week ASC";
if ($rs=mysqli_query($con,$q)) {
  while($r=mysqli_fetch_assoc($rs)) $rows[(int)$r['day_of_week']] = $r;
  mysqli_free_result($rs);
}

// defaults si faltan días
for($d=1;$d<=7;$d++){
  if (!isset($rows[$d])) {
    $rows[$d] = [
      'day_of_week'=>$d,
      'active'=> ($d<=5 ? 1 : 0),
      'start_time'=> ($d<=5 ? '08:00:00' : null),
      'end_time'=> ($d<=5 ? '17:30:00' : null),
      'late_grace_minutes'=>15,
      'early_grace_minutes'=>5,
      'lunch_start'=>'12:00:00',
      'lunch_end'=>'13:00:00',
      'coffee1_start'=>'09:00:00',
      'coffee1_end'=>'09:15:00',
      'coffee2_start'=>'15:00:00',
      'coffee2_end'=>'15:15:00',
      'break_window_before_min'=>10,
      'break_window_after_min'=>10,
    ];
  }
}

$err = '';
$ok  = '';

if ($_SERVER['REQUEST_METHOD']==='POST') {

  $tmp = [];
  $weeklyMinutes = 0;

  $beforeMin = (int)($_POST['break_window_before_min'] ?? 10);
  $afterMin  = (int)($_POST['break_window_after_min'] ?? 10);
  if ($beforeMin < 0) $beforeMin = 0;
  if ($afterMin < 0)  $afterMin  = 0;

  for($d=1;$d<=7;$d++){
    $active = (int)($_POST["active_$d"] ?? 0);

    $start = norm_time($_POST["start_$d"] ?? '');
    $end   = norm_time($_POST["end_$d"] ?? '');

    $late  = (int)($_POST["late_$d"] ?? 15);
    $early = (int)($_POST["early_$d"] ?? 5);

    $lS = norm_time($_POST["lunch_s_$d"] ?? '');
    $lE = norm_time($_POST["lunch_e_$d"] ?? '');

    $c1S = norm_time($_POST["c1_s_$d"] ?? '');
    $c1E = norm_time($_POST["c1_e_$d"] ?? '');

    $c2S = norm_time($_POST["c2_s_$d"] ?? '');
    $c2E = norm_time($_POST["c2_e_$d"] ?? '');

    if ($active === 1) {
      if (!$start || !$end) { $err = "Falta hora de entrada/salida en ".dow_label($d)."."; break; }
      if (minutes_between($start,$end) <= 0) { $err = "La salida debe ser mayor que la entrada en ".dow_label($d)."."; break; }

      // Validar que descansos estén dentro del rango si se llenan
      $dayStart = $start; $dayEnd = $end;

      $pairs = [
        ['Almuerzo', $lS, $lE],
        ['Café 1', $c1S, $c1E],
        ['Café 2', $c2S, $c2E],
      ];
      foreach($pairs as [$label,$s,$e]){
        if (($s && !$e) || (!$s && $e)) { $err = "Complete inicio/fin de $label en ".dow_label($d)." (o déjelo vacío)."; break 2; }
        if ($s && $e) {
          if (minutes_between($s,$e) <= 0) { $err = "$label: fin debe ser mayor que inicio en ".dow_label($d)."."; break 2; }
          if (!within_range($s,$dayStart,$dayEnd) || !within_range($e,$dayStart,$dayEnd)) {
            $err = "$label debe estar dentro del horario del día en ".dow_label($d)."."; break 2;
          }
        }
      }

      $weeklyMinutes += minutes_between($start,$end); // ✅ HORAS BRUTAS (no restamos breaks)
    } else {
      // inactivo: permite vacío
      $start = $end = null;
    }

    $tmp[$d] = [
      'day_of_week'=>$d,
      'active'=>$active,
      'start_time'=>$start,
      'end_time'=>$end,
      'late_grace_minutes'=>max(0,$late),
      'early_grace_minutes'=>max(0,$early),
      'lunch_start'=>$lS, 'lunch_end'=>$lE,
      'coffee1_start'=>$c1S, 'coffee1_end'=>$c1E,
      'coffee2_start'=>$c2S, 'coffee2_end'=>$c2E,
    ];
  }

  // ✅ Validación 48h semanales (Código de Trabajo: jornada ordinaria y su límite semanal)
  $weeklyHours = $weeklyMinutes / 60.0;
  if (!$err && $weeklyHours > 48.0 + 1e-9) {
    $err = "No permitido: el horario suma ".number_format($weeklyHours,2)." horas brutas semanales y excede 48 horas.";
  }

  if (!$err) {
    // Guardar (REPLACE por día)
    $sql = "REPLACE INTO work_schedule
            (day_of_week, active, start_time, end_time, late_grace_minutes, early_grace_minutes,
             lunch_start, lunch_end, coffee1_start, coffee1_end, coffee2_start, coffee2_end,
             break_window_before_min, break_window_after_min, updated_at)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,NOW())";
    $st = mysqli_prepare($con, $sql);
    if (!$st) {
      $err = "Error prepare: ".mysqli_error($con);
    } else {
      foreach($tmp as $d=>$r){
        mysqli_stmt_bind_param(
          $st, 'iissiissssssii',
          $r['day_of_week'],
          $r['active'],
          $r['start_time'],
          $r['end_time'],
          $r['late_grace_minutes'],
          $r['early_grace_minutes'],
          $r['lunch_start'], $r['lunch_end'],
          $r['coffee1_start'], $r['coffee1_end'],
          $r['coffee2_start'], $r['coffee2_end'],
          $beforeMin, $afterMin
        );
        if (!mysqli_stmt_execute($st)) { $err = "Error al guardar ".dow_label((int)$d).": ".mysqli_error($con); break; }
      }
      mysqli_stmt_close($st);
    }

    if (!$err) {
      $ok = "Horario guardado. Total semanal: ".number_format($weeklyHours,2)." horas brutas.";
      // recargar para mostrar lo guardado
      $rows = [];
      if ($rs=mysqli_query($con,"SELECT * FROM work_schedule ORDER BY day_of_week ASC")) {
        while($r=mysqli_fetch_assoc($rs)) $rows[(int)$r['day_of_week']] = $r;
        mysqli_free_result($rs);
      }
      for($d=1;$d<=7;$d++){ if(!isset($rows[$d])) $rows[$d] = $tmp[$d]; }
    } else {
      // mantener valores del post
      for($d=1;$d<=7;$d++){
        $rows[$d] = $tmp[$d] + $rows[$d];
        $rows[$d]['break_window_before_min'] = $beforeMin;
        $rows[$d]['break_window_after_min']  = $afterMin;
      }
    }
  } else {
    // mantener valores del post
    for($d=1;$d<=7;$d++){
      if(isset($tmp[$d])) $rows[$d] = $tmp[$d] + $rows[$d];
      $rows[$d]['break_window_before_min'] = $beforeMin;
      $rows[$d]['break_window_after_min']  = $afterMin;
    }
  }
}

// Header
if(!$ES_SUPER) include __DIR__.'/view/header.php';
else           include __DIR__.'/view/headeradmin.html';
?>
<script>
  $('#payroll_control').addClass("treeview active");
</script>

<div class="box">
  <div class="box-header with-border">
    <h3>Horario Laboral (Global) • Asistencia</h3>
    <p class="text-muted" style="margin:6px 0 0">
      Este horario lo consumen: <code>/rrhh_home/asistencia/jornada_api.php</code> (escaneo QR) y luego el módulo de Horas Extra.
    </p>
  </div>

  <div class="box-body">
    <?php if($err): ?><div class="alert alert-danger"><?= h($err) ?></div><?php endif; ?>
    <?php if($ok): ?><div class="alert alert-success"><?= h($ok) ?></div><?php endif; ?>

    <form method="post" id="frmSchedule">
      <div class="row">
        <div class="col-md-6">
          <div class="form-group">
            <label>Ventana de detección para Cafés/Almuerzo (minutos)</label>
            <div class="row">
              <div class="col-xs-6">
                <label class="text-muted">Antes (min)</label>
                <input class="form-control" type="number" min="0" name="break_window_before_min"
                       id="bw_before"
                       value="<?= (int)($rows[1]['break_window_before_min'] ?? 10) ?>">
              </div>
              <div class="col-xs-6">
                <label class="text-muted">Después (min)</label>
                <input class="form-control" type="number" min="0" name="break_window_after_min"
                       id="bw_after"
                       value="<?= (int)($rows[1]['break_window_after_min'] ?? 10) ?>">
              </div>
            </div>
            <p class="help-block">
              Ejemplo: 10/10 → el escaneo detecta salida a café desde 10 min antes del inicio y 10 min después.
            </p>
          </div>
        </div>

        <div class="col-md-6">
          <div class="alert alert-info" style="margin-top:24px">
            <b>Horas BRUTAS:</b> el total semanal se calcula como <i>(salida - entrada)</i> por día.
            <br><small class="text-muted">Por ahora NO se restan cafés/almuerzo del total (solo se marcan como eventos).</small>
          </div>
        </div>
      </div>

      <div class="table-responsive">
        <table class="table table-bordered table-hover" id="tblSchedule">
          <thead>
            <tr>
              <th style="width:110px">Día</th>
              <th style="width:70px">Activo</th>
              <th>Entrada</th>
              <th>Salida</th>
              <th>Almuerzo (ini)</th>
              <th>Almuerzo (fin)</th>
              <th>Café 1 (ini)</th>
              <th>Café 1 (fin)</th>
              <th>Café 2 (ini)</th>
              <th>Café 2 (fin)</th>
              <th style="width:90px">Gracia late</th>
              <th style="width:90px">Gracia early</th>
              <th style="width:120px">Horas/día (brutas)</th>
            </tr>
          </thead>
          <tbody>
          <?php for($d=1;$d<=7;$d++):
            $r = $rows[$d];
            $active = (int)($r['active'] ?? 0);
          ?>
            <tr>
              <td><b><?= h(dow_label($d)) ?></b></td>
              <td class="text-center">
                <input type="checkbox" name="active_<?= $d ?>" id="active_<?= $d ?>" value="1" <?= $active===1?'checked':'' ?>>
              </td>
              <td><input class="form-control t" type="time" step="60" name="start_<?= $d ?>" id="start_<?= $d ?>" value="<?= h(substr((string)($r['start_time'] ?? ''),0,5)) ?>"></td>
              <td><input class="form-control t" type="time" step="60" name="end_<?= $d ?>"   id="end_<?= $d ?>"   value="<?= h(substr((string)($r['end_time'] ?? ''),0,5)) ?>"></td>

              <td><input class="form-control t" type="time" step="60" name="lunch_s_<?= $d ?>" id="lunch_s_<?= $d ?>" value="<?= h(substr((string)($r['lunch_start'] ?? ''),0,5)) ?>"></td>
              <td><input class="form-control t" type="time" step="60" name="lunch_e_<?= $d ?>" id="lunch_e_<?= $d ?>" value="<?= h(substr((string)($r['lunch_end'] ?? ''),0,5)) ?>"></td>

              <td><input class="form-control t" type="time" step="60" name="c1_s_<?= $d ?>" id="c1_s_<?= $d ?>" value="<?= h(substr((string)($r['coffee1_start'] ?? ''),0,5)) ?>"></td>
              <td><input class="form-control t" type="time" step="60" name="c1_e_<?= $d ?>" id="c1_e_<?= $d ?>" value="<?= h(substr((string)($r['coffee1_end'] ?? ''),0,5)) ?>"></td>

              <td><input class="form-control t" type="time" step="60" name="c2_s_<?= $d ?>" id="c2_s_<?= $d ?>" value="<?= h(substr((string)($r['coffee2_start'] ?? ''),0,5)) ?>"></td>
              <td><input class="form-control t" type="time" step="60" name="c2_e_<?= $d ?>" id="c2_e_<?= $d ?>" value="<?= h(substr((string)($r['coffee2_end'] ?? ''),0,5)) ?>"></td>

              <td><input class="form-control n" type="number" min="0" name="late_<?= $d ?>" id="late_<?= $d ?>" value="<?= (int)($r['late_grace_minutes'] ?? 15) ?>"></td>
              <td><input class="form-control n" type="number" min="0" name="early_<?= $d ?>" id="early_<?= $d ?>" value="<?= (int)($r['early_grace_minutes'] ?? 5) ?>"></td>

              <td class="text-center">
                <span class="label label-default" id="dayHours_<?= $d ?>">0.00</span>
              </td>
            </tr>
          <?php endfor; ?>
          </tbody>
        </table>
      </div>

      <div class="row" style="margin-top:10px">
        <div class="col-md-6">
          <div class="alert" id="sumBox" style="margin-bottom:10px">
            <b>Total semanal (horas brutas):</b> <span id="weekHours">0.00</span> / 48.00
            <div class="text-muted"><small id="sumNote">Edita el horario para ver el cálculo.</small></div>
          </div>
        </div>
        <div class="col-md-6 text-right">
          <a class="btn btn-default" href="HomeRRHH.php">Volver</a>
          <button class="btn btn-primary" id="btnSave"><i class="fa fa-save"></i> Guardar</button>
        </div>
      </div>

      <div class="alert alert-warning" style="margin-top:10px">
        <b>Referencia legal (Costa Rica):</b><br>
        “La jornada ordinaria de trabajo efectivo no podrá ser mayor … de cuarenta y ocho horas por semana.”<br>
        <small class="text-muted">
          Nota: la norma también indica límites diarios (según tipo de jornada) y permite excepciones en ciertos casos.
        </small>
      </div>
    </form>
  </div>
</div>

<script>
(function(){
  const LIMIT = 48.0;

  function toMin(t){
    if(!t) return 0;
    const p = String(t).split(':');
    if(p.length < 2) return 0;
    const h = parseInt(p[0],10)||0;
    const m = parseInt(p[1],10)||0;
    return h*60+m;
  }
  function fmt(n){ return (Math.round(n*100)/100).toFixed(2); }

  function recalc(){
    let weekMin = 0;

    for(let d=1; d<=7; d++){
      const active = document.getElementById('active_'+d)?.checked;
      const s = document.getElementById('start_'+d)?.value || '';
      const e = document.getElementById('end_'+d)?.value || '';

      let dayMin = 0;
      if(active && s && e){
        const ms = toMin(s), me = toMin(e);
        dayMin = Math.max(0, me - ms);
        weekMin += dayMin;
      }
      document.getElementById('dayHours_'+d).textContent = fmt(dayMin/60);
    }

    const weekH = weekMin/60;
    document.getElementById('weekHours').textContent = fmt(weekH);

    const box = document.getElementById('sumBox');
    const note = document.getElementById('sumNote');
    const btn = document.getElementById('btnSave');

    if(weekH > LIMIT + 1e-9){
      box.className = 'alert alert-danger';
      note.textContent = 'No permitido: excede 48 horas semanales. Ajusta entradas/salidas o desactiva días.';
      btn.disabled = true;
    } else {
      box.className = 'alert alert-success';
      note.textContent = 'Cumple el límite semanal configurado.';
      btn.disabled = false;
    }
  }

  document.querySelectorAll('#frmSchedule input').forEach(el=>{
    el.addEventListener('input', recalc);
    el.addEventListener('change', recalc);
  });

  recalc();
})();
</script>

<?php include __DIR__.'/view/footer.html'; ?>
