<?php
// rrhh_home/extras_dashboard_api.php
// API: KPIs por mes o rango y Top 5 colaboradores por horas extra (filtro opcional por empleado)
// PHP 8.1.33 – UTF-8

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');
header('Content-Type: application/json; charset=utf-8');

if (!isset($_SESSION["idusuario"])) {
  echo json_encode(['ok' => false, 'msg' => 'Sesión expirada']); exit;
}

require_once __DIR__ . '/dbcon.php';
if (!isset($con) || !($con instanceof mysqli)) {
  echo json_encode(['ok' => false, 'msg' => 'Sin conexión a BD']); exit;
}
mysqli_set_charset($con, 'utf8mb4');

$action = $_POST['action'] ?? $_GET['action'] ?? '';
if ($action !== 'stats_month') {
  echo json_encode(['ok' => false, 'msg' => 'Acción no válida']); exit;
}

// Soportar MES o RANGO
$mes    = $_POST['mes']   ?? $_GET['mes']   ?? '';
$desde  = $_POST['desde'] ?? $_GET['desde'] ?? '';
$hasta  = $_POST['hasta'] ?? $_GET['hasta'] ?? '';

$use_range = false;
if ($desde && $hasta && preg_match('/^\d{4}\-\d{2}\-\d{2}$/',$desde) && preg_match('/^\d{4}\-\d{2}\-\d{2}$/',$hasta) && $hasta >= $desde) {
  $use_range = true;
} else {
  if (!preg_match('/^\d{4}\-\d{2}$/', $mes)) { $mes = date('Y-m'); }
}

$idempleado = isset($_POST['idempleado']) ? (int)$_POST['idempleado'] : (isset($_GET['idempleado']) ? (int)$_GET['idempleado'] : 0);
$scope = $idempleado > 0 ? 'empleado' : 'general';

/* ===== Preferencias para salario-hora ===== */
$pref_horas = 8.0; $pref_dias = 30.0;
if ($rs = mysqli_query($con, "SELECT horas_dia, dias_mes FROM payroll_pref WHERE activo=1 ORDER BY id DESC LIMIT 1")) {
  if ($r = mysqli_fetch_assoc($rs)) {
    $pref_horas = max(0.0, (float)$r['horas_dia']);
    $pref_dias  = max(0.0, (float)$r['dias_mes']);
  }
  mysqli_free_result($rs);
}
$den = ($pref_horas * $pref_dias);
if ($den <= 0) $den = 240.0; // fallback

/* ===== WHERE dinámico (mes o rango) ===== */
$bind_types = '';
$bind_vals  = [];
$periodo    = $use_range ? 'range' : 'month';

if ($use_range) {
  $where = " d.fecha BETWEEN ? AND ? ";
  $bind_types .= 'ss';
  $bind_vals[] = $desde; $bind_vals[] = $hasta;
} else {
  $where = " m.mes = ? ";
  $bind_types .= 's';
  $bind_vals[] = $mes;
}

if ($idempleado > 0) {
  $where .= " AND m.idempleado = ? ";
  $bind_types .= 'i';
  $bind_vals[] = $idempleado;
}

/* ===== Totales por factor (filtrados) ===== */
$sqlTot = "
  SELECT
    COALESCE(SUM(CASE WHEN d.factor = 1.5 THEN d.horas ELSE 0 END),0) AS h15,
    COALESCE(SUM(CASE WHEN d.factor = 2.0 THEN d.horas ELSE 0 END),0) AS h20,
    COALESCE(SUM(CASE WHEN d.factor = 3.0 THEN d.horas ELSE 0 END),0) AS h30,
    COALESCE(SUM(d.horas),0) AS hTot
  FROM extras_detalle d
  INNER JOIN extras_mes m ON m.id = d.id_extras_mes
  WHERE $where
";
$tot = ['h15'=>0.0,'h20'=>0.0,'h30'=>0.0,'hTot'=>0.0];
if ($stmt = mysqli_prepare($con, $sqlTot)) {
  mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_vals);
  mysqli_stmt_execute($stmt);
  mysqli_stmt_bind_result($stmt, $a, $b, $c, $t);
  if (mysqli_stmt_fetch($stmt)) {
    $tot = ['h15'=>(float)$a, 'h20'=>(float)$b, 'h30'=>(float)$c, 'hTot'=>(float)$t];
  }
  mysqli_stmt_close($stmt);
}

/* ===== Top 5 colaboradores ===== */
$sqlTop = "
  SELECT ep.nombre_completo, COALESCE(SUM(d.horas),0) AS horas
  FROM extras_detalle d
  INNER JOIN extras_mes m ON m.id = d.id_extras_mes
  INNER JOIN empleados_planilla ep ON ep.id = m.idempleado
  WHERE $where
  GROUP BY ep.id, ep.nombre_completo
  ORDER BY horas DESC
  LIMIT 5
";
$top = [];
if ($stmt = mysqli_prepare($con, $sqlTop)) {
  mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_vals);
  mysqli_stmt_execute($stmt);
  mysqli_stmt_bind_result($stmt, $n, $h);
  while (mysqli_stmt_fetch($stmt)) {
    $top[] = ['nombre' => $n, 'horas' => (float)$h];
  }
  mysqli_stmt_close($stmt);
}

/* ===== Salario-hora base para montos ===== */
$salario_hora = 0.0;
$empleado_nombre = '';

if ($idempleado > 0) {
  if ($st = mysqli_prepare($con, "SELECT nombre_completo, COALESCE(salario_bruto,0) FROM empleados_planilla WHERE id=?")) {
    mysqli_stmt_bind_param($st, 'i', $idempleado);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st, $nn, $sb);
    if (mysqli_stmt_fetch($st)) {
      $empleado_nombre = $nn ?: '';
      $salario_hora = ((float)$sb) / $den;
    }
    mysqli_stmt_close($st);
  }
} else {
  // Promedio de quienes tuvieron horas en el período
  if ($use_range) {
    $sqlAvg = "
      SELECT AVG(COALESCE(ep.salario_bruto,0)/?) AS prom
      FROM empleados_planilla ep
      INNER JOIN extras_mes m ON m.idempleado = ep.id
      INNER JOIN extras_detalle d ON d.id_extras_mes = m.id
      WHERE d.fecha BETWEEN ? AND ?
      GROUP BY 1
    ";
    if ($st = mysqli_prepare($con, $sqlAvg)) {
      mysqli_stmt_bind_param($st, 'dss', $den, $desde, $hasta);
      mysqli_stmt_execute($st);
      mysqli_stmt_bind_result($st, $prom);
      if (mysqli_stmt_fetch($st)) { $salario_hora = (float)$prom; }
      mysqli_stmt_close($st);
    }
  } else {
    $sqlAvg = "
      SELECT AVG(COALESCE(ep.salario_bruto,0)/?) AS prom
      FROM empleados_planilla ep
      INNER JOIN extras_mes m ON m.idempleado = ep.id
      INNER JOIN extras_detalle d ON d.id_extras_mes = m.id
      WHERE m.mes = ?
      GROUP BY 1
    ";
    if ($st = mysqli_prepare($con, $sqlAvg)) {
      mysqli_stmt_bind_param($st, 'ds', $den, $mes);
      mysqli_stmt_execute($st);
      mysqli_stmt_bind_result($st, $prom);
      if (mysqli_stmt_fetch($st)) { $salario_hora = (float)$prom; }
      mysqli_stmt_close($st);
    }
  }
}

/* ===== Montos (en colones) ===== */
$h15 = (float)$tot['h15']; $h20 = (float)$tot['h20']; $h30 = (float)$tot['h30'];
$m15 = $h15 * $salario_hora * 1.5;
$m20 = $h20 * $salario_hora * 2.0;
$m30 = $h30 * $salario_hora * 3.0;
$montos = ['m15'=>$m15, 'm20'=>$m20, 'm30'=>$m30, 'mTot'=>($m15+$m20+$m30)];

echo json_encode([
  'ok' => true,
  'scope' => $scope,
  'periodo' => $periodo,
  'mes'   => $use_range ? null : $mes,
  'desde' => $use_range ? $desde : null,
  'hasta' => $use_range ? $hasta : null,
  'idempleado' => $idempleado,
  'empleado_nombre' => $empleado_nombre,
  'salario_hora' => round($salario_hora, 4),
  'tot' => $tot,
  'montos' => $montos,
  'top' => $top
], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
