<?php
// rrhh_home/extras_api.php
// API v1: asegurar cabecera (empleado+mes), devolver mes/rango, upsert/borrado diario,
// totales, montos, polilla_data, helpers de UX y bitácora.
// + NUEVO: Soporte QR overtime_qr (get/save/delete por fecha) para corregir bloqueos desde editor.
// PHP 8.1.33 – UTF-8

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');
header('Content-Type: application/json; charset=utf-8');

if (!isset($_SESSION["idusuario"])) { echo json_encode(['ok'=>false,'msg'=>'Sesión expirada']); exit; }

require_once __DIR__ . '/dbcon.php';
if (!isset($con) || !($con instanceof mysqli)) { echo json_encode(['ok'=>false,'msg'=>'Sin conexión a BD']); exit; }
mysqli_set_charset($con,'utf8mb4');

/* ===== Permisos ===== */
$READ_ONLY = false;
if (isset($_SESSION['rol']) && $_SESSION['rol'] === 'viewer') $READ_ONLY = true;

$action = $_POST['action'] ?? $_GET['action'] ?? '';

function valid_mes(string $mes): bool { return (bool)preg_match('/^\d{4}\-\d{2}$/',$mes); }
function valid_ymd(string $d): bool { return (bool)preg_match('/^\d{4}\-\d{2}\-\d{2}$/',$d); }
function respond($arr){ echo json_encode($arr, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }

function norm_time(?string $s): ?string {
  $s = trim((string)$s);
  if ($s === '') return null;
  if (preg_match('/^\d{2}:\d{2}$/', $s)) $s .= ':00';
  if (!preg_match('/^\d{2}:\d{2}:\d{2}$/', $s)) return null;
  $dt = DateTime::createFromFormat('H:i:s', $s);
  return $dt ? $dt->format('H:i:s') : null;
}
function fmt_hhmm(?string $t): ?string {
  if (!$t) return null;
  return substr((string)$t,0,5);
}
function minutes_between(string $start, string $end): int {
  $base = '2000-01-01 ';
  $a = strtotime($base.$start);
  $b = strtotime($base.$end);
  if ($a===false || $b===false) return 0;
  return (int)floor(($b-$a)/60);
}
function round_up_half_hours(int $minutes): float {
  if ($minutes <= 0) return 0.0;
  $blocks = (int)ceil($minutes / 30);
  return $blocks * 0.5;
}

/* ===== asegurar tabla overtime_qr aquí también (para el editor) ===== */
@mysqli_query($con, "
CREATE TABLE IF NOT EXISTS overtime_qr (
  id INT AUTO_INCREMENT PRIMARY KEY,
  idempleado INT NOT NULL,
  gestor_id INT NOT NULL DEFAULT 0,
  fecha DATE NOT NULL,
  schedule_end TIME NOT NULL,
  planned_end TIME NULL,
  planned_at DATETIME NULL,
  actual_end TIME NULL,
  closed_at DATETIME NULL,
  overtime_minutes INT NOT NULL DEFAULT 0,
  factor DECIMAL(3,1) NOT NULL DEFAULT 1.5,
  note VARCHAR(255) NULL,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NULL,
  UNIQUE KEY uq_emp_fecha (idempleado, fecha),
  KEY idx_gestor (gestor_id),
  KEY idx_fecha (fecha)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
");

switch($action){

  /* ================== MES ================== */
  case 'get_month': {
    $idempleado = (int)($_POST['idempleado'] ?? 0);
    $mes = $_POST['mes'] ?? date('Y-m');
    if($idempleado<=0 || !valid_mes($mes)) respond(['ok'=>false,'msg'=>'Parámetros inválidos']);

    $id_extras_mes = ensure_header($con, $idempleado, $mes);

    $detalle = [];
    $sql = "SELECT id, fecha, horas, factor, descripcion
            FROM extras_detalle
            WHERE id_extras_mes=?
            ORDER BY fecha ASC";
    if($st = mysqli_prepare($con,$sql)){
      mysqli_stmt_bind_param($st,'i',$id_extras_mes);
      mysqli_stmt_execute($st);
      $rs = mysqli_stmt_get_result($st);
      while($r = mysqli_fetch_assoc($rs)){
        $detalle[] = [
          'id'=>(int)$r['id'],
          'fecha'=>$r['fecha'],
          'horas'=>(float)$r['horas'],
          'factor'=> (string)$r['factor'],
          'descripcion'=>$r['descripcion'] ?? ''
        ];
      }
      mysqli_stmt_close($st);
    }

    $totales = compute_totals($con, $id_extras_mes);

    $vac = [];
    $sqlv = "SELECT id, fecha_desde AS fecha_inicio, fecha_hasta AS fecha_fin, motivo
             FROM ingreso_vacaciones
             WHERE idempleado=? AND status=1";
    if($st = mysqli_prepare($con,$sqlv)){
      mysqli_stmt_bind_param($st,'i',$idempleado);
      mysqli_stmt_execute($st);
      $rs = mysqli_stmt_get_result($st);
      while($r=mysqli_fetch_assoc($rs)){
        $vac[] = ['id'=>(int)$r['id'],'fecha_inicio'=>$r['fecha_inicio'],'fecha_fin'=>$r['fecha_fin'],'motivo'=>$r['motivo'] ?? ''];
      }
      mysqli_stmt_close($st);
    }

    [$y,$m] = explode('-', $mes);
    $iniMes = "$y-$m-01";
    $finMes = date('Y-m-t', strtotime($iniMes));
    $inc = get_incapacidades_rango($con, $idempleado, $iniMes, $finMes);

    $pref = get_pref($con);
    $sb = get_salario_bruto($con, $idempleado);
    $salario_hora = ($pref['horas_dia']>0 && $pref['dias_mes']>0) ? ($sb/($pref['horas_dia']*$pref['dias_mes'])) : 0.0;

    $rules  = get_legal_rules($con);
    $montos = compute_amounts_param($totales, $salario_hora, $rules);

    respond([
      'ok'=>true,
      'modo'=>'month',
      'detalle'=>$detalle,
      'totales'=>$totales,
      'vacaciones'=>$vac,
      'incapacidades'=>$inc,
      'pref'=>$pref,
      'salario_hora'=>$salario_hora,
      'rules'=>$rules,
      'montos'=>$montos,
      'readonly'=>$READ_ONLY
    ]);
  } break;

  /* ================== RANGO ================== */
  case 'get_range': {
    $idempleado = (int)($_POST['idempleado'] ?? 0);
    $desde = $_POST['desde'] ?? '';
    $hasta = $_POST['hasta'] ?? '';
    if ($idempleado<=0 || !valid_ymd($desde) || !valid_ymd($hasta) || $hasta < $desde) {
      respond(['ok'=>false,'msg'=>'Parámetros inválidos (rango)']);
    }

    $detalle = [];
    $qd = "SELECT d.id, d.fecha, d.horas, d.factor, d.descripcion
           FROM extras_detalle d
           INNER JOIN extras_mes m ON m.id = d.id_extras_mes
           WHERE m.idempleado=? AND d.fecha BETWEEN ? AND ?
           ORDER BY d.fecha ASC";
    if($st = mysqli_prepare($con,$qd)){
      mysqli_stmt_bind_param($st,'iss',$idempleado,$desde,$hasta);
      mysqli_stmt_execute($st);
      $rs = mysqli_stmt_get_result($st);
      while($r=mysqli_fetch_assoc($rs)){
        $detalle[] = [
          'id'=>(int)$r['id'],
          'fecha'=>$r['fecha'],
          'horas'=>(float)$r['horas'],
          'factor'=>$r['factor'],
          'descripcion'=>$r['descripcion'] ?? ''
        ];
      }
      mysqli_stmt_close($st);
    }

    $totales = ['h15'=>0.0,'h20'=>0.0,'h30'=>0.0,'hTot'=>0.0];
    $qt = "SELECT
              COALESCE(SUM(CASE WHEN d.factor=1.5 THEN d.horas ELSE 0 END),0) AS h15,
              COALESCE(SUM(CASE WHEN d.factor=2.0 THEN d.horas ELSE 0 END),0) AS h20,
              COALESCE(SUM(CASE WHEN d.factor=3.0 THEN d.horas ELSE 0 END),0) AS h30,
              COALESCE(SUM(d.horas),0) AS hTot
           FROM extras_detalle d
           INNER JOIN extras_mes m ON m.id=d.id_extras_mes
           WHERE m.idempleado=? AND d.fecha BETWEEN ? AND ?";
    if($st = mysqli_prepare($con,$qt)){
      mysqli_stmt_bind_param($st,'iss',$idempleado,$desde,$hasta);
      mysqli_stmt_execute($st);
      mysqli_stmt_bind_result($st,$a,$b,$c,$t);
      if(mysqli_stmt_fetch($st)) $totales = ['h15'=>(float)$a,'h20'=>(float)$b,'h30'=>(float)$c,'hTot'=>(float)$t];
      mysqli_stmt_close($st);
    }

    $vac = [];
    $sqlv = "SELECT id, fecha_desde AS fecha_inicio, fecha_hasta AS fecha_fin, motivo
             FROM ingreso_vacaciones
             WHERE idempleado=? AND status=1
               AND NOT (fecha_hasta < ? OR fecha_desde > ?)";
    if($st=mysqli_prepare($con,$sqlv)){
      mysqli_stmt_bind_param($st,'iss',$idempleado,$desde,$hasta);
      mysqli_stmt_execute($st);
      $rs = mysqli_stmt_get_result($st);
      while($r=mysqli_fetch_assoc($rs)){
        $vac[] = ['id'=>(int)$r['id'],'fecha_inicio'=>$r['fecha_inicio'],'fecha_fin'=>$r['fecha_fin'],'motivo'=>$r['motivo'] ?? ''];
      }
      mysqli_stmt_close($st);
    }

    $inc = get_incapacidades_rango($con, $idempleado, $desde, $hasta);

    $pref = get_pref($con);
    $sb = get_salario_bruto($con, $idempleado);
    $salario_hora = ($pref['horas_dia']>0 && $pref['dias_mes']>0) ? ($sb/($pref['horas_dia']*$pref['dias_mes'])) : 0.0;

    $rules  = get_legal_rules($con);
    $montos = compute_amounts_param($totales, $salario_hora, $rules);

    respond([
      'ok'=>true,
      'modo'=>'range',
      'desde'=>$desde,'hasta'=>$hasta,
      'detalle'=>$detalle,
      'totales'=>$totales,
      'vacaciones'=>$vac,
      'incapacidades'=>$inc,
      'pref'=>$pref,
      'salario_hora'=>$salario_hora,
      'rules'=>$rules,
      'montos'=>$montos,
      'readonly'=>$READ_ONLY
    ]);
  } break;

  /* ================== Upsert día (0 = eliminar) ================== */
  case 'upsert_day': {
    if ($READ_ONLY) respond(['ok'=>false,'msg'=>'Sin permiso para modificar']);

    $idempleado = (int)($_POST['idempleado'] ?? 0);
    $mes = $_POST['mes'] ?? '';
    $fecha = $_POST['fecha'] ?? '';
    $horas = isset($_POST['horas']) ? (float)$_POST['horas'] : 0.0;
    $factor = (string)($_POST['factor'] ?? '1.5');
    $desc = trim((string)($_POST['descripcion'] ?? ''));

    if($idempleado<=0 || !valid_mes($mes) || !valid_ymd($fecha)){
      respond(['ok'=>false,'msg'=>'Parámetros inválidos']);
    }
    if($horas<0) respond(['ok'=>false,'msg'=>'Horas inválidas']);
    if(!in_array($factor, ['1.5','2.0','3.0'], true)) respond(['ok'=>false,'msg'=>'Factor inválido']);
    if(strlen($desc)>255) $desc = substr($desc,0,255);
    if(strpos($fecha, $mes)!==0) respond(['ok'=>false,'msg'=>'La fecha no pertenece al mes']);

    $id_extras_mes = ensure_header($con, $idempleado, $mes);
    $uid = (int)($_SESSION['idusuario'] ?? 0);

    if ($horas <= 0) {
      delete_day($con, $id_extras_mes, $fecha, $uid);
      $totales = compute_totals($con, $id_extras_mes);
      respond(['ok'=>true,'totales'=>$totales,'deleted'=>true]);
    }

    $old = ['horas'=>null,'factor'=>null,'descripcion'=>null];
    if($st0=mysqli_prepare($con,"SELECT horas, factor, descripcion FROM extras_detalle WHERE id_extras_mes=? AND fecha=? LIMIT 1")){
      mysqli_stmt_bind_param($st0,'is',$id_extras_mes,$fecha);
      mysqli_stmt_execute($st0);
      mysqli_stmt_bind_result($st0,$oh,$of,$od);
      if(mysqli_stmt_fetch($st0)) $old=['horas'=>$oh,'factor'=>$of,'descripcion'=>$od];
      mysqli_stmt_close($st0);
    }

    $sql = "INSERT INTO extras_detalle (id_extras_mes, fecha, horas, factor, descripcion, creado_por)
            VALUES (?,?,?,?,?,?)
            ON DUPLICATE KEY UPDATE horas=VALUES(horas), factor=VALUES(factor), descripcion=VALUES(descripcion), updated_by=VALUES(creado_por), updated_at=NOW()";
    if($st = mysqli_prepare($con,$sql)){
      mysqli_stmt_bind_param($st,'issdsi',$id_extras_mes,$fecha,$horas,$factor,$desc,$uid);
      mysqli_stmt_execute($st);
      mysqli_stmt_close($st);
    } else {
      respond(['ok'=>false,'msg'=>mysqli_error($con)]);
    }

    log_change($con,$id_extras_mes,$fecha,$old['horas'],$old['factor'],$old['descripcion'],$horas,$factor,$desc,$uid);

    $totales = compute_totals($con, $id_extras_mes);
    respond(['ok'=>true,'totales'=>$totales]);
  } break;

  /* ================== Eliminar día ================== */
  case 'delete_day': {
    if ($READ_ONLY) respond(['ok'=>false,'msg'=>'Sin permiso para modificar']);

    $idempleado = (int)($_POST['idempleado'] ?? 0);
    $mes        = $_POST['mes'] ?? '';
    $fecha      = $_POST['fecha'] ?? '';

    if($idempleado<=0 || !valid_mes($mes) || !valid_ymd($fecha)){
      respond(['ok'=>false,'msg'=>'Parámetros inválidos']);
    }
    if(strpos($fecha, $mes)!==0) respond(['ok'=>false,'msg'=>'La fecha no pertenece al mes']);

    $id_extras_mes = ensure_header($con, $idempleado, $mes);
    $uid = (int)($_SESSION['idusuario'] ?? 0);

    $ok = delete_day($con, $id_extras_mes, $fecha, $uid);
    $totales = compute_totals($con, $id_extras_mes);
    respond(['ok'=>$ok, 'totales'=>$totales]);
  } break;

  /* ================== QR: obtener por fecha ================== */
  case 'qr_get': {
    $idempleado = (int)($_POST['idempleado'] ?? 0);
    $fecha = (string)($_POST['fecha'] ?? '');
    if($idempleado<=0 || !valid_ymd($fecha)) respond(['ok'=>false,'msg'=>'Parámetros inválidos']);

    $row = overtime_get($con, $idempleado, $fecha);
    if(!$row){
      respond(['ok'=>true,'has'=>false]);
    }

    $state = (!empty($row['actual_end'])) ? 'CLOSED' : (!empty($row['planned_end']) ? 'PLANNED' : 'OPEN');
    respond([
      'ok'=>true,
      'has'=>true,
      'state'=>$state,
      'fecha'=>$fecha,
      'schedule_end'=>fmt_hhmm($row['schedule_end']),
      'planned_end'=>($row['planned_end']?fmt_hhmm($row['planned_end']):null),
      'actual_end'=>($row['actual_end']?fmt_hhmm($row['actual_end']):null),
      'minutes'=>(int)($row['overtime_minutes'] ?? 0),
      'factor'=>(float)($row['factor'] ?? 1.5),
      'note'=>$row['note'] ?? ''
    ]);
  } break;

  /* ================== QR: guardar ajustes (planned/actual/schedule) ================== */
  case 'qr_save': {
    if ($READ_ONLY) respond(['ok'=>false,'msg'=>'Sin permiso para modificar']);

    $idempleado = (int)($_POST['idempleado'] ?? 0);
    $fecha = (string)($_POST['fecha'] ?? '');
    $schedule_end = norm_time($_POST['schedule_end'] ?? '');
    $planned_end  = norm_time($_POST['planned_end'] ?? '');
    $actual_end   = norm_time($_POST['actual_end'] ?? '');
    $note = trim((string)($_POST['note'] ?? ''));

    if($idempleado<=0 || !valid_ymd($fecha) || !$schedule_end){
      respond(['ok'=>false,'msg'=>'Parámetros inválidos (fecha / salida oficial)']);
    }
    if(strlen($note)>255) $note = substr($note,0,255);

    $mins = 0;
    $hours = 0.0;

    $actual_for_calc = $actual_end ?: null;
    if ($actual_for_calc) {
      $mins = minutes_between($schedule_end, $actual_for_calc);
      if ($mins < 0) $mins = 0;
      $hours = round_up_half_hours($mins);
    }

    $uid = (int)($_SESSION['idusuario'] ?? 0);

    // Upsert overtime_qr
    $sql = "INSERT INTO overtime_qr
              (idempleado, gestor_id, fecha, schedule_end, planned_end, planned_at, actual_end, closed_at, overtime_minutes, factor, note, created_at, updated_at)
            VALUES
              (?,?,?,?,?,IF(? IS NULL, NULL, NOW()), ?, IF(? IS NULL, NULL, NOW()), ?, 1.5, ?, NOW(), NOW())
            ON DUPLICATE KEY UPDATE
              schedule_end=VALUES(schedule_end),
              planned_end=VALUES(planned_end),
              planned_at=IF(VALUES(planned_end) IS NULL, NULL, NOW()),
              actual_end=VALUES(actual_end),
              closed_at=IF(VALUES(actual_end) IS NULL, NULL, NOW()),
              overtime_minutes=VALUES(overtime_minutes),
              factor=1.5,
              note=VALUES(note),
              updated_at=NOW()";

    if(!$st=mysqli_prepare($con,$sql)) respond(['ok'=>false,'msg'=>mysqli_error($con)]);
    $planned_or_null = $planned_end ?: null;
    $actual_or_null  = $actual_end ?: null;

    // (idempleado, gestor_id=0, fecha, schedule_end, planned_end, planned_end? , actual_end, actual_end?, minutes, note)
    $gestor_id = 0;
    mysqli_stmt_bind_param(
      $st,'iissssssiss',
      $idempleado, $gestor_id, $fecha, $schedule_end,
      $planned_or_null, $planned_or_null,
      $actual_or_null, $actual_or_null,
      $mins,
      $note
    );
    mysqli_stmt_execute($st);
    mysqli_stmt_close($st);

    // Sin cierre real => no inyecta, y si antes fue QR, lo quitamos (solo si era QR)
    $mes = substr($fecha,0,7);
    $id_extras_mes = ensure_header($con, $idempleado, $mes);

    if (!$actual_or_null) {
      // borrar extras_detalle solo si era un QR Extra
      $q = "SELECT descripcion FROM extras_detalle WHERE id_extras_mes=? AND fecha=? LIMIT 1";
      $descOld = '';
      if($st=mysqli_prepare($con,$q)){
        mysqli_stmt_bind_param($st,'is',$id_extras_mes,$fecha);
        mysqli_stmt_execute($st);
        mysqli_stmt_bind_result($st,$tmp);
        if(mysqli_stmt_fetch($st)) $descOld = (string)$tmp;
        mysqli_stmt_close($st);
      }
      if ($descOld !== '' && stripos($descOld, 'QR Extra |') === 0) {
        delete_day($con, $id_extras_mes, $fecha, $uid);
      }

      $state = ($planned_or_null ? 'PLANNED' : 'OPEN');
      respond([
        'ok'=>true,
        'state'=>$state,
        'fecha'=>$fecha,
        'schedule_end'=>fmt_hhmm($schedule_end),
        'planned_end'=>($planned_or_null?fmt_hhmm($planned_or_null):null),
        'actual_end'=>null,
        'minutes'=>0,
        'hours'=>0,
        'msg'=>'QR actualizado (sin cierre real).'
      ]);
    }

    // Con cierre real => inyectar/actualizar extras_detalle con horas redondeadas 0.5
    $desc = "QR Extra | Salida oficial: ".fmt_hhmm($schedule_end)
          ." | Plan: ".($planned_or_null?fmt_hhmm($planned_or_null):'-')
          ." | Real: ".fmt_hhmm($actual_or_null)
          ." | (Horas en 0.5)";

    // reutilizamos el mismo upsert del API (factor 1.5)
    $sql2 = "INSERT INTO extras_detalle (id_extras_mes, fecha, horas, factor, descripcion, creado_por)
             VALUES (?,?,?,?,?,?)
             ON DUPLICATE KEY UPDATE
               horas=VALUES(horas),
               factor=VALUES(factor),
               descripcion=VALUES(descripcion),
               updated_by=VALUES(creado_por),
               updated_at=NOW()";
    if(!$st2=mysqli_prepare($con,$sql2)) respond(['ok'=>false,'msg'=>mysqli_error($con)]);
    $factor = '1.5';
    mysqli_stmt_bind_param($st2,'issdsi',$id_extras_mes,$fecha,$hours,$factor,$desc,$uid);
    mysqli_stmt_execute($st2);
    mysqli_stmt_close($st2);

    respond([
      'ok'=>true,
      'state'=>'CLOSED',
      'fecha'=>$fecha,
      'schedule_end'=>fmt_hhmm($schedule_end),
      'planned_end'=>($planned_or_null?fmt_hhmm($planned_or_null):null),
      'actual_end'=>fmt_hhmm($actual_or_null),
      'minutes'=>$mins,
      'hours'=>$hours,
      'msg'=>'QR actualizado e inyectado en horas extra (0.5 en 0.5).'
    ]);
  } break;

  /* ================== QR: eliminar registro del día ================== */
  case 'qr_delete': {
    if ($READ_ONLY) respond(['ok'=>false,'msg'=>'Sin permiso para modificar']);

    $idempleado = (int)($_POST['idempleado'] ?? 0);
    $fecha = (string)($_POST['fecha'] ?? '');
    if($idempleado<=0 || !valid_ymd($fecha)) respond(['ok'=>false,'msg'=>'Parámetros inválidos']);

    // borrar overtime_qr
    $d="DELETE FROM overtime_qr WHERE idempleado=? AND fecha=? LIMIT 1";
    if($st=mysqli_prepare($con,$d)){
      mysqli_stmt_bind_param($st,'is',$idempleado,$fecha);
      mysqli_stmt_execute($st);
      mysqli_stmt_close($st);
    }

    // borrar extras_detalle solo si era QR
    $mes = substr($fecha,0,7);
    $id_extras_mes = ensure_header($con, $idempleado, $mes);
    $uid = (int)($_SESSION['idusuario'] ?? 0);

    $descOld = '';
    if($st=mysqli_prepare($con,"SELECT descripcion FROM extras_detalle WHERE id_extras_mes=? AND fecha=? LIMIT 1")){
      mysqli_stmt_bind_param($st,'is',$id_extras_mes,$fecha);
      mysqli_stmt_execute($st);
      mysqli_stmt_bind_result($st,$tmp);
      if(mysqli_stmt_fetch($st)) $descOld = (string)$tmp;
      mysqli_stmt_close($st);
    }
    if ($descOld !== '' && stripos($descOld, 'QR Extra |') === 0) {
      delete_day($con, $id_extras_mes, $fecha, $uid);
    }

    respond(['ok'=>true,'msg'=>'Se eliminó el QR de ese día.']);
  } break;

  default:
    respond(['ok'=>false,'msg'=>'Acción no válida']);
}

/* ================= Helpers ================= */

function overtime_get(mysqli $con, int $idempleado, string $fecha): ?array {
  $q="SELECT id, schedule_end, planned_end, actual_end, overtime_minutes, factor, note
      FROM overtime_qr WHERE idempleado=? AND fecha=? LIMIT 1";
  if($st=mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'is',$idempleado,$fecha);
    mysqli_stmt_execute($st);
    $rs=mysqli_stmt_get_result($st);
    $row=$rs?mysqli_fetch_assoc($rs):null;
    mysqli_stmt_close($st);
    return $row ?: null;
  }
  return null;
}

function ensure_header(mysqli $con, int $idempleado, string $mes): int {
  $id = 0;
  $q = "SELECT id FROM extras_mes WHERE idempleado=? AND mes=? LIMIT 1";
  if($st = mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'is',$idempleado,$mes);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$idtmp);
    if(mysqli_stmt_fetch($st)) $id = (int)$idtmp;
    mysqli_stmt_close($st);
  }
  if($id>0) return $id;

  $uid = (int)($_SESSION['idusuario'] ?? 0);
  $ins = "INSERT INTO extras_mes (idempleado, mes, creado_por) VALUES (?,?,?)";
  if($st = mysqli_prepare($con,$ins)){
    mysqli_stmt_bind_param($st,'isi',$idempleado,$mes,$uid);
    mysqli_stmt_execute($st);
    $id = mysqli_insert_id($con);
    mysqli_stmt_close($st);
  }
  return (int)$id;
}

function compute_totals(mysqli $con, int $id_extras_mes): array {
  $out = ['h15'=>0.0,'h20'=>0.0,'h30'=>0.0,'hTot'=>0.0];
  $q = "SELECT
          COALESCE(SUM(CASE WHEN factor=1.5 THEN horas ELSE 0 END),0) AS h15,
          COALESCE(SUM(CASE WHEN factor=2.0 THEN horas ELSE 0 END),0) AS h20,
          COALESCE(SUM(CASE WHEN factor=3.0 THEN horas ELSE 0 END),0) AS h30,
          COALESCE(SUM(horas),0) AS hTot
        FROM extras_detalle
        WHERE id_extras_mes=?";
  if($st = mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'i',$id_extras_mes);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$a,$b,$c,$t);
    if(mysqli_stmt_fetch($st)) $out = ['h15'=>(float)$a,'h20'=>(float)$b,'h30'=>(float)$c,'hTot'=>(float)$t];
    mysqli_stmt_close($st);
  }
  return $out;
}

function diff_days_inclusive(string $a, string $b): int {
  $t1 = strtotime($a.' 00:00:00'); $t2 = strtotime($b.' 00:00:00');
  if($t2 < $t1) return 0;
  return (int)round(($t2 - $t1)/86400) + 1;
}

function get_pref(mysqli $con): array {
  $pref=['horas_dia'=>8.0,'dias_mes'=>30.0];
  $qp="SELECT horas_dia, dias_mes FROM payroll_pref WHERE activo=1 ORDER BY id DESC LIMIT 1";
  if($rs=mysqli_query($con,$qp)){
    if($r=mysqli_fetch_assoc($rs)){
      $pref=['horas_dia'=>(float)$r['horas_dia'],'dias_mes'=>(float)$r['dias_mes']];
    }
    mysqli_free_result($rs);
  }
  return $pref;
}

function get_salario_bruto(mysqli $con, int $idempleado): float {
  $sb=0.0;
  $q="SELECT COALESCE(salario_bruto,0) sb FROM empleados_planilla WHERE id=?";
  if($st=mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'i',$idempleado);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$tmp);
    if(mysqli_stmt_fetch($st)) $sb=(float)$tmp;
    mysqli_stmt_close($st);
  }
  return $sb;
}

function get_salario_hora(float $salario_bruto, array $pref): float {
  $h=($pref['horas_dia']??0)*($pref['dias_mes']??0);
  return $h>0?($salario_bruto/$h):0.0;
}

function log_change(mysqli $con, int $id_extras_mes, string $fecha,
  $old_horas, $old_factor, $old_desc, $new_horas, $new_factor, $new_desc, int $uid): void {
  $sql="INSERT INTO extras_detalle_log (id_extras_mes, fecha, old_horas, old_factor, old_desc, new_horas, new_factor, new_desc, changed_by)
        VALUES (?,?,?,?,?,?,?,?,?)";
  if($st=mysqli_prepare($con,$sql)){
    mysqli_stmt_bind_param($st,'issdssssi',
      $id_extras_mes, $fecha,
      $old_horas, $old_factor, $old_desc,
      $new_horas, $new_factor, $new_desc,
      $uid
    );
    mysqli_stmt_execute($st);
    mysqli_stmt_close($st);
  }
}

function get_legal_rules(mysqli $con): array {
  $r = [
    'factor_regular'       => 1.00,
    'factor_overtime'      => 1.50,
    'factor_holiday'       => 2.00,
    'factor_custom'        => 3.00,
    'max_extra_por_dia'    => 4.00,
    'max_extra_por_semana' => 12.00
  ];
  $sql = "SELECT factor_regular,factor_overtime,factor_holiday,factor_custom,
                 max_extra_por_dia,max_extra_por_semana
          FROM payroll_legal_rules
          WHERE activo=1
          ORDER BY id DESC
          LIMIT 1";
  if ($rs = mysqli_query($con, $sql)) {
    if ($row = mysqli_fetch_assoc($rs)) {
      $r['factor_regular']       = (float)$row['factor_regular'];
      $r['factor_overtime']      = (float)$row['factor_overtime'];
      $r['factor_holiday']       = (float)$row['factor_holiday'];
      $r['factor_custom']        = (float)$row['factor_custom'];
      $r['max_extra_por_dia']    = (float)$row['max_extra_por_dia'];
      $r['max_extra_por_semana'] = (float)$row['max_extra_por_semana'];
    }
    mysqli_free_result($rs);
  }
  return $r;
}

function compute_amounts_param(array $tot, float $sal_hora, array $rules): array {
  $h15 = (float)($tot['h15'] ?? 0.0);
  $h20 = (float)($tot['h20'] ?? 0.0);
  $h30 = (float)($tot['h30'] ?? 0.0);
  $m15 = $h15 * $sal_hora * (float)($rules['factor_overtime'] ?? 1.5);
  $m20 = $h20 * $sal_hora * (float)($rules['factor_holiday']  ?? 2.0);
  $m30 = $h30 * $sal_hora * (float)($rules['factor_custom']   ?? 3.0);
  return ['m15'=>$m15,'m20'=>$m20,'m30'=>$m30,'mTot'=>($m15+$m20+$m30)];
}

function get_incapacidades_rango(mysqli $con, int $idempleado, string $desde, string $hasta): array {
  $inc=[];
  $sqli = "SELECT id, fecha_inicio, fecha_fin, motivo
           FROM ingreso_incapacidades
           WHERE idempleado=? AND NOT (fecha_fin < ? OR fecha_inicio > ?)";
  if($st = mysqli_prepare($con,$sqli)){
    mysqli_stmt_bind_param($st,'iss',$idempleado,$desde,$hasta);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    while($r=mysqli_fetch_assoc($rs)){
      $inc[]=['id'=>(int)$r['id'],'fecha_inicio'=>$r['fecha_inicio'],'fecha_fin'=>$r['fecha_fin'],'motivo'=>$r['motivo'] ?? '','dias'=> diff_days_inclusive($r['fecha_inicio'],$r['fecha_fin'])];
    }
    mysqli_stmt_close($st);
  }
  return $inc;
}

function delete_day(mysqli $con, int $id_extras_mes, string $fecha, int $uid=0): bool {
  $old = ['horas'=>null,'factor'=>null,'descripcion'=>null];
  if($st0=mysqli_prepare($con,"SELECT horas, factor, descripcion FROM extras_detalle WHERE id_extras_mes=? AND fecha=? LIMIT 1")){
    mysqli_stmt_bind_param($st0,'is',$id_extras_mes,$fecha);
    mysqli_stmt_execute($st0);
    mysqli_stmt_bind_result($st0,$oh,$of,$od);
    if(mysqli_stmt_fetch($st0)) $old=['horas'=>$oh,'factor'=>$of,'descripcion'=>$od];
    mysqli_stmt_close($st0);
  }
  if($st=mysqli_prepare($con,"DELETE FROM extras_detalle WHERE id_extras_mes=? AND fecha=?")){
    mysqli_stmt_bind_param($st,'is',$id_extras_mes,$fecha);
    $ok = mysqli_stmt_execute($st);
    mysqli_stmt_close($st);
    log_change($con,$id_extras_mes,$fecha,$old['horas'],$old['factor'],$old['descripcion'],null,null,null,$uid);
    return $ok;
  }
  return false;
}
