<?php
// rrhh_home/export_polilla_pdf.php
// Exporta la Polilla (por empleado y mes) a PDF usando FPDF.
// Si FPDF no está disponible, hace fallback a HTML imprimible.

declare(strict_types=1);
session_start();
@date_default_timezone_set('America/Costa_Rica');
set_time_limit(20);

if (!isset($_SESSION["idusuario"])) { http_response_code(401); exit('Sesión expirada'); }
require_once __DIR__ . '/dbcon.php';
if (!isset($con) || !($con instanceof mysqli)) { http_response_code(500); exit('Sin conexión BD'); }
mysqli_set_charset($con,'utf8mb4');

$idempleado = isset($_GET['idempleado']) ? (int)$_GET['idempleado'] : 0;
$mes        = $_GET['mes'] ?? date('Y-m');
if ($idempleado<=0 || !preg_match('/^\d{4}\-\d{2}$/',$mes)) { http_response_code(400); exit('Parámetros inválidos'); }

/* ===== Helpers ===== */
function ensure_header(mysqli $con, int $idempleado, string $mes): int {
  $id=0;
  if ($st=mysqli_prepare($con,"SELECT id FROM extras_mes WHERE idempleado=? AND mes=? LIMIT 1")){
    mysqli_stmt_bind_param($st,'is',$idempleado,$mes);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$tmp);
    if (mysqli_stmt_fetch($st)) $id=(int)$tmp;
    mysqli_stmt_close($st);
  }
  if ($id>0) return $id;
  $uid=(int)($_SESSION['idusuario']??0);
  if ($st=mysqli_prepare($con,"INSERT INTO extras_mes (idempleado, mes, creado_por) VALUES (?,?,?)")){
    mysqli_stmt_bind_param($st,'isi',$idempleado,$mes,$uid);
    mysqli_stmt_execute($st);
    $id=mysqli_insert_id($con);
    mysqli_stmt_close($st);
  }
  return (int)$id;
}
function get_pref(mysqli $con): array {
  $pref=['horas_dia'=>8.0,'dias_mes'=>30.0];
  if ($rs=mysqli_query($con,"SELECT horas_dia, dias_mes FROM payroll_pref WHERE activo=1 ORDER BY id DESC LIMIT 1")){
    if ($r=mysqli_fetch_assoc($rs)) $pref=['horas_dia'=>(float)$r['horas_dia'],'dias_mes'=>(float)$r['dias_mes']];
    mysqli_free_result($rs);
  }
  return $pref;
}
function get_empleado(mysqli $con, int $id): array {
  $e=['nombre'=>'','cedula'=>'','categoria'=>'','salario_bruto'=>0.0];
  if ($st=mysqli_prepare($con,"SELECT nombre_completo, cedula, categoria, COALESCE(salario_bruto,0) FROM empleados_planilla WHERE id=?")){
    mysqli_stmt_bind_param($st,'i',$id);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$n,$c,$cat,$sb);
    if (mysqli_stmt_fetch($st)) $e=['nombre'=>$n,'cedula'=>$c,'categoria'=>$cat,'salario_bruto'=>(float)$sb];
    mysqli_stmt_close($st);
  }
  return $e;
}
function salario_hora(float $sb, array $pref): float {
  $h=($pref['horas_dia']??0)*($pref['dias_mes']??0);
  return $h>0 ? ($sb/$h) : 0.0;
}

/* ===== Data ===== */
$id_extras_mes = ensure_header($con, $idempleado, $mes);
$emp  = get_empleado($con, $idempleado);
$pref = get_pref($con);
$sal_hora = salario_hora((float)$emp['salario_bruto'], $pref);

$detalle=[];
if ($st=mysqli_prepare($con,"SELECT fecha, horas, factor, descripcion FROM extras_detalle WHERE id_extras_mes=? ORDER BY fecha ASC")){
  mysqli_stmt_bind_param($st,'i',$id_extras_mes);
  mysqli_stmt_execute($st);
  $rs=mysqli_stmt_get_result($st);
  while($r=mysqli_fetch_assoc($rs)){
    $r['horas']=(float)$r['horas'];
    $r['monto'] = $sal_hora * $r['horas'] * (float)$r['factor'];
    $detalle[]=$r;
  }
  mysqli_stmt_close($st);
}
$tot_h15=$tot_h20=$tot_h30=$tot_all=0.0;
$m15=$m20=$m30=$mTot=0.0;
foreach($detalle as $d){
  $h=$d['horas']; $f=(float)$d['factor']; $monto=$d['monto'];
  if ($f==1.5) $tot_h15+=$h;
  elseif ($f==2.0) $tot_h20+=$h;
  elseif ($f==3.0) $tot_h30+=$h;
  $tot_all += $h;
  if ($f==1.5) $m15 += $monto;
  if ($f==2.0) $m20 += $monto;
  if ($f==3.0) $m30 += $monto;
}
$mTot=$m15+$m20+$m30;

/* ===== Intento FPDF ===== */
$fpdf_loaded = false;
$paths = [
  __DIR__ . '/lib/fpdf.php',            // si lo subes a rrhh_home/lib/fpdf.php
  __DIR__ . '/fpdf/fpdf.php',           // o rrhh_home/fpdf/fpdf.php
  __DIR__ . '/vendor/fpdf/fpdf.php'     // o vía composer
];
foreach($paths as $p){ if (is_file($p)) { require_once $p; $fpdf_loaded = class_exists('FPDF'); break; } }

if ($fpdf_loaded) {
  // ---------- PDF con FPDF (muy liviano) ----------
  $pdf = new FPDF('P','mm','Letter');
  $pdf->SetTitle('Polilla '.$mes);
  $pdf->SetAuthor('RRHH');
  $pdf->AddPage();
  $pdf->SetFont('Arial','B',14);
  $pdf->Cell(0,8,utf8_decode('Polilla de pago'),0,1,'L');
  $pdf->SetFont('Arial','',10);
  $pdf->Cell(0,6,utf8_decode("Empleado: {$emp['nombre']}  |  Cédula: {$emp['cedula']}"),0,1,'L');
  $pdf->Cell(0,6,utf8_decode("Categoría: {$emp['categoria']}  |  Mes: $mes"),0,1,'L');
  $pdf->Cell(0,6,utf8_decode("Salario bruto: ₡".number_format((float)$emp['salario_bruto'],2,'.',',')."  |  Salario hora: ₡".number_format($sal_hora,2,'.',',')),0,1,'L');

  $pdf->Ln(2);
  // Header tabla
  $pdf->SetFont('Arial','B',10);
  $pdf->Cell(28,8,utf8_decode('Fecha'),1,0,'C');
  $pdf->Cell(20,8,utf8_decode('Horas'),1,0,'C');
  $pdf->Cell(18,8,utf8_decode('Factor'),1,0,'C');
  $pdf->Cell(90,8,utf8_decode('Descripción'),1,0,'C');
  $pdf->Cell(30,8,utf8_decode('Monto'),1,1,'C');

  $pdf->SetFont('Arial','',10);
  foreach($detalle as $d){
    $pdf->Cell(28,7,utf8_decode($d['fecha']),1,0,'C');
    $pdf->Cell(20,7,number_format($d['horas'],2,'.',','),1,0,'R');
    $pdf->Cell(18,7,utf8_decode($d['factor'].'x'),1,0,'C');
    // Descripción truncada a una línea
    $desc = trim((string)($d['descripcion'] ?? ''));
    if (mb_strlen($desc)>60) $desc = mb_substr($desc,0,57).'...';
    $pdf->Cell(90,7,utf8_decode($desc),1,0,'L');
    $pdf->Cell(30,7,number_format($d['monto'],2,'.',','),1,1,'R');
  }

  $pdf->Ln(2);
  $pdf->SetFont('Arial','B',10);
  $pdf->Cell(0,7,utf8_decode('Totales de horas'),0,1,'L');
  $pdf->SetFont('Arial','',10);
  $pdf->Cell(0,6,utf8_decode('1.5x: '.number_format($tot_h15,2,'.',',').'  |  2.0x: '.number_format($tot_h20,2,'.',',').'  |  3.0x: '.number_format($tot_h30,2,'.',',').'  |  Total: '.number_format($tot_all,2,'.',',')),0,1,'L');

  $pdf->Ln(1);
  $pdf->SetFont('Arial','B',10);
  $pdf->Cell(0,7,utf8_decode('Montos'),0,1,'L');
  $pdf->SetFont('Arial','',10);
  $pdf->Cell(0,6,utf8_decode('1.5x: ₡'.number_format($m15,2,'.',',').'  |  2.0x: ₡'.number_format($m20,2,'.',',').'  |  3.0x: ₡'.number_format($m30,2,'.',',').'  |  Total a pagar: ₡'.number_format($mTot,2,'.',',')),0,1,'L');

  $pdf->Output('I', 'Polilla_'.$mes.'.pdf');
  exit;
}

/* ===== Fallback HTML imprimible ===== */
header('Content-Type: text/html; charset=utf-8');
?><!doctype html>
<html lang="es"><head>
<meta charset="utf-8">
<title>Polilla <?= htmlspecialchars($mes) ?></title>
<meta name="viewport" content="width=device-width,initial-scale=1">
<style>
  body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial,sans-serif;margin:20px;color:#111827}
  h1{font-size:20px;margin:0 0 8px}
  .sub{color:#374151;margin-bottom:10px}
  table{width:100%;border-collapse:collapse;margin-top:8px;font-size:12px}
  th,td{border:1px solid #e5e7eb;padding:6px}
  th{background:#f9fafb;text-align:left}
  .tot{margin-top:10px}
  @media print{ .noprint{display:none} body{margin:0} }
</style>
</head><body>
  <div class="noprint" style="text-align:right;margin-bottom:8px">
    <button onclick="window.print()">Imprimir / Guardar como PDF</button>
  </div>
  <h1>Polilla de pago</h1>
  <div class="sub">
    <div><b>Empleado:</b> <?= htmlspecialchars($emp['nombre']) ?> &nbsp; | &nbsp; <b>Cédula:</b> <?= htmlspecialchars($emp['cedula']) ?></div>
    <div><b>Categoría:</b> <?= htmlspecialchars($emp['categoria']) ?> &nbsp; | &nbsp; <b>Mes:</b> <?= htmlspecialchars($mes) ?></div>
    <div><b>Salario bruto:</b> ₡<?= number_format((float)$emp['salario_bruto'],2,'.',',') ?> &nbsp; | &nbsp;
        <b>Salario hora:</b> ₡<?= number_format($sal_hora,2,'.',',') ?></div>
  </div>

  <table>
    <thead><tr><th>Fecha</th><th style="width:90px">Horas</th><th style="width:80px">Factor</th><th>Descripción</th><th style="width:120px">Monto</th></tr></thead>
    <tbody>
      <?php if(!$detalle): ?>
        <tr><td colspan="5" style="text-align:center;color:#6b7280">Sin registros</td></tr>
      <?php else:
        foreach($detalle as $d): ?>
          <tr>
            <td><?= htmlspecialchars($d['fecha']) ?></td>
            <td style="text-align:right"><?= number_format((float)$d['horas'],2,'.',',') ?></td>
            <td style="text-align:center"><?= htmlspecialchars($d['factor']) ?>x</td>
            <td><?= htmlspecialchars($d['descripcion'] ?? '') ?></td>
            <td style="text-align:right">₡<?= number_format((float)$d['monto'],2,'.',',') ?></td>
          </tr>
      <?php endforeach; endif; ?>
    </tbody>
  </table>

  <div class="tot">
    <p><b>Totales de horas:</b>
      1.5x: <?= number_format($tot_h15,2,'.',',') ?> ·
      2.0x: <?= number_format($tot_h20,2,'.',',') ?> ·
      3.0x: <?= number_format($tot_h30,2,'.',',') ?> ·
      Total: <?= number_format($tot_all,2,'.',',') ?></p>
    <p><b>Montos:</b>
      1.5x: ₡<?= number_format($m15,2,'.',',') ?> ·
      2.0x: ₡<?= number_format($m20,2,'.',',') ?> ·
      3.0x: ₡<?= number_format($m30,2,'.',',') ?> ·
      <b>Total a pagar: ₡<?= number_format($mTot,2,'.',',') ?></b></p>
  </div>
</body></html>
