<?php
// rrhh_home/export_polilla_excel.php
// Exporta la Polilla (por empleado y mes) en CSV compatible con Excel (sin dependencias).

declare(strict_types=1);
session_start();
@date_default_timezone_set('America/Costa_Rica');
set_time_limit(20);

if (!isset($_SESSION["idusuario"])) { http_response_code(401); exit('Sesión expirada'); }
require_once __DIR__ . '/dbcon.php';
if (!isset($con) || !($con instanceof mysqli)) { http_response_code(500); exit('Sin conexión BD'); }
mysqli_set_charset($con,'utf8mb4');

$idempleado = isset($_GET['idempleado']) ? (int)$_GET['idempleado'] : 0;
$mes        = $_GET['mes'] ?? date('Y-m');
if ($idempleado<=0 || !preg_match('/^\d{4}\-\d{2}$/',$mes)) { http_response_code(400); exit('Parámetros inválidos'); }

/* ===== Helpers ===== */
function ensure_header(mysqli $con, int $idempleado, string $mes): int {
  $id=0;
  if ($st=mysqli_prepare($con,"SELECT id FROM extras_mes WHERE idempleado=? AND mes=? LIMIT 1")){
    mysqli_stmt_bind_param($st,'is',$idempleado,$mes);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$tmp);
    if (mysqli_stmt_fetch($st)) $id=(int)$tmp;
    mysqli_stmt_close($st);
  }
  if ($id>0) return $id;
  $uid=(int)($_SESSION['idusuario']??0);
  if ($st=mysqli_prepare($con,"INSERT INTO extras_mes (idempleado, mes, creado_por) VALUES (?,?,?)")){
    mysqli_stmt_bind_param($st,'isi',$idempleado,$mes,$uid);
    mysqli_stmt_execute($st);
    $id=mysqli_insert_id($con);
    mysqli_stmt_close($st);
  }
  return (int)$id;
}
function get_pref(mysqli $con): array {
  $pref=['horas_dia'=>8.0,'dias_mes'=>30.0];
  if ($rs=mysqli_query($con,"SELECT horas_dia, dias_mes FROM payroll_pref WHERE activo=1 ORDER BY id DESC LIMIT 1")){
    if ($r=mysqli_fetch_assoc($rs)) $pref=['horas_dia'=>(float)$r['horas_dia'],'dias_mes'=>(float)$r['dias_mes']];
    mysqli_free_result($rs);
  }
  return $pref;
}
function get_empleado(mysqli $con, int $id): array {
  $e=['nombre'=>'','cedula'=>'','categoria'=>'','salario_bruto'=>0.0];
  if ($st=mysqli_prepare($con,"SELECT nombre_completo, cedula, categoria, COALESCE(salario_bruto,0) FROM empleados_planilla WHERE id=?")){
    mysqli_stmt_bind_param($st,'i',$id);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$n,$c,$cat,$sb);
    if (mysqli_stmt_fetch($st)) $e=['nombre'=>$n,'cedula'=>$c,'categoria'=>$cat,'salario_bruto'=>(float)$sb];
    mysqli_stmt_close($st);
  }
  return $e;
}
function salario_hora(float $sb, array $pref): float {
  $h=($pref['horas_dia']??0)*($pref['dias_mes']??0);
  return $h>0 ? ($sb/$h) : 0.0;
}

/* ===== Data ===== */
$id_extras_mes = ensure_header($con, $idempleado, $mes);
$emp = get_empleado($con, $idempleado);
$pref = get_pref($con);
$sal_hora = salario_hora((float)$emp['salario_bruto'], $pref);

$detalle=[];
if ($st=mysqli_prepare($con,"SELECT fecha, horas, factor, descripcion FROM extras_detalle WHERE id_extras_mes=? ORDER BY fecha ASC")){
  mysqli_stmt_bind_param($st,'i',$id_extras_mes);
  mysqli_stmt_execute($st);
  $rs=mysqli_stmt_get_result($st);
  while($r=mysqli_fetch_assoc($rs)){
    $r['horas']=(float)$r['horas'];
    $r['monto'] = $sal_hora * $r['horas'] * (float)$r['factor'];
    $detalle[]=$r;
  }
  mysqli_stmt_close($st);
}

$tot_h15=$tot_h20=$tot_h30=$tot_all=0.0;
$m15=$m20=$m30=$mTot=0.0;
foreach($detalle as $d){
  $h=$d['horas']; $f=(float)$d['factor']; $monto=$d['monto'];
  if ($f==1.5) $tot_h15+=$h;
  elseif ($f==2.0) $tot_h20+=$h;
  elseif ($f==3.0) $tot_h30+=$h;
  $tot_all += $h;
  if ($f==1.5) $m15 += $monto;
  if ($f==2.0) $m20 += $monto;
  if ($f==3.0) $m30 += $monto;
}
$mTot = $m15+$m20+$m30;

/* ===== CSV Output ===== */
$filename = sprintf('Polilla_%s_%s.csv', preg_replace('/\D+/','',$emp['cedula']??'EMP'), $mes);
header('Content-Type: text/csv; charset=utf-8');
header('Content-Disposition: attachment; filename="'.$filename.'"');
header('Pragma: no-cache');
header('Expires: 0');

$out = fopen('php://output', 'w');
// BOM para Excel en Windows
fwrite($out, chr(0xEF).chr(0xBB).chr(0xBF));

fputcsv($out, ['Polilla de pago']);
fputcsv($out, ['Empleado', $emp['nombre']]);
fputcsv($out, ['Cédula', $emp['cedula']]);
fputcsv($out, ['Categoría', $emp['categoria']]);
fputcsv($out, ['Mes', $mes]);
fputcsv($out, ['Salario bruto', number_format((float)$emp['salario_bruto'],2,'.',',')]);
fputcsv($out, ['Salario hora', number_format($sal_hora,2,'.',',')]);
fputcsv($out, []); // línea vacía

fputcsv($out, ['Fecha','Horas','Factor','Descripción','Monto']);
foreach($detalle as $d){
  fputcsv($out, [
    $d['fecha'],
    number_format($d['horas'],2,'.',''),
    $d['factor'].'x',
    $d['descripcion'] ?? '',
    number_format($d['monto'],2,'.','')
  ]);
}
fputcsv($out, []);
fputcsv($out, ['Totales Horas','1.5x','2.0x','3.0x','Total']);
fputcsv($out, [
  '',
  number_format($tot_h15,2,'.',''),
  number_format($tot_h20,2,'.',''),
  number_format($tot_h30,2,'.',''),
  number_format($tot_all,2,'.','')
]);
fputcsv($out, []);
fputcsv($out, ['Montos','1.5x','2.0x','3.0x','Total a pagar']);
fputcsv($out, [
  '',
  number_format($m15,2,'.',''),
  number_format($m20,2,'.',''),
  number_format($m30,2,'.',''),
  number_format($mTot,2,'.','')
]);

fclose($out);
