<?php
// rrhh_home/export_dashboard_pdf.php
// Exporta Dashboard a PDF (FPDF). Si FPDF no existe, cae a HTML imprimible.

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');

if (!isset($_SESSION["idusuario"])) { http_response_code(401); exit('Sesión expirada'); }

require_once __DIR__ . '/dbcon.php';
if (!isset($con) || !($con instanceof mysqli)) { http_response_code(500); exit('Sin conexión BD'); }
mysqli_set_charset($con, 'utf8mb4');

$mes = $_GET['mes'] ?? date('Y-m');
if (!preg_match('/^\d{4}\-\d{2}$/', $mes)) $mes = date('Y-m');
$idempleado = isset($_GET['idempleado']) && $_GET['idempleado'] !== '' ? (int)$_GET['idempleado'] : null;

// --- helpers ---
function get_pref(mysqli $con): array {
  $pref=['horas_dia'=>8.0,'dias_mes'=>30.0];
  $qp="SELECT horas_dia, dias_mes FROM payroll_pref WHERE activo=1 ORDER BY id DESC LIMIT 1";
  if($rs=mysqli_query($con,$qp)){
    if($r=mysqli_fetch_assoc($rs)){
      $pref=['horas_dia'=>(float)$r['horas_dia'],'dias_mes'=>(float)$r['dias_mes']];
    }
    mysqli_free_result($rs);
  }
  return $pref;
}
function get_emp(mysqli $con, int $id): array {
  $out=['nombre'=>'','salario_bruto'=>0.0];
  if($st=mysqli_prepare($con,"SELECT nombre_completo, COALESCE(salario_bruto,0) FROM empleados_planilla WHERE id=?")){
    mysqli_stmt_bind_param($st,'i',$id);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$n,$sb);
    if(mysqli_stmt_fetch($st)) $out=['nombre'=>$n,'salario_bruto'=>(float)$sb];
    mysqli_stmt_close($st);
  }
  return $out;
}
function salario_hora(float $sb, array $pref): float {
  $h=($pref['horas_dia']??0)*($pref['dias_mes']??0);
  return $h>0?($sb/$h):0.0;
}

// --- datos (mismos del dashboard) ---
$where = "WHERE m.mes = ?";
$params = [$mes]; $types = 's';
if ($idempleado) { $where .= " AND m.idempleado = ?"; $params[] = $idempleado; $types .= 'i'; }

$sqlTot = "
  SELECT
    COALESCE(SUM(CASE WHEN d.factor = 1.5 THEN d.horas ELSE 0 END),0) AS h15,
    COALESCE(SUM(CASE WHEN d.factor = 2.0 THEN d.horas ELSE 0 END),0) AS h20,
    COALESCE(SUM(CASE WHEN d.factor = 3.0 THEN d.horas ELSE 0 END),0) AS h30,
    COALESCE(SUM(d.horas),0) AS hTot
  FROM extras_detalle d
  INNER JOIN extras_mes m ON m.id = d.id_extras_mes
  $where
";
$tot=['h15'=>0.0,'h20'=>0.0,'h30'=>0.0,'hTot'=>0.0];
if ($st=mysqli_prepare($con,$sqlTot)){
  mysqli_stmt_bind_param($st,$types,...$params);
  mysqli_stmt_execute($st);
  mysqli_stmt_bind_result($st,$a,$b,$c,$t);
  if (mysqli_stmt_fetch($st)) $tot=['h15'=>(float)$a,'h20'=>(float)$b,'h30'=>(float)$c,'hTot'=>(float)$t];
  mysqli_stmt_close($st);
}

$sqlTop = "
  SELECT ep.nombre_completo, COALESCE(SUM(d.horas),0) AS horas
  FROM extras_detalle d
  INNER JOIN extras_mes m ON m.id = d.id_extras_mes
  INNER JOIN empleados_planilla ep ON ep.id = m.idempleado
  $where
  GROUP BY ep.id, ep.nombre_completo
  ORDER BY horas DESC
  LIMIT 5
";
$top=[];
if ($st=mysqli_prepare($con,$sqlTop)){
  mysqli_stmt_bind_param($st,$types,...$params);
  mysqli_stmt_execute($st);
  mysqli_stmt_bind_result($st,$n,$h);
  while(mysqli_stmt_fetch($st)) $top[]=['nombre'=>$n,'horas'=>(float)$h];
  mysqli_stmt_close($st);
}

$pref = get_pref($con);
$scope_desc = 'General del mes';
$sal_h = 0.0;
$emp_nombre = '';
if ($idempleado) {
  $emp = get_emp($con, $idempleado);
  $emp_nombre = $emp['nombre'];
  $sal_h = salario_hora((float)$emp['salario_bruto'], $pref);
  $scope_desc = 'Empleado: ' . ($emp_nombre ?: ('ID '.$idempleado));
}
$m15 = $tot['h15']*$sal_h*1.5;
$m20 = $tot['h20']*$sal_h*2.0;
$m30 = $tot['h30']*$sal_h*3.0;
$mTot= $m15+$m20+$m30;

// --- intentar FPDF ---
$fpdf_path = __DIR__ . '/lib/fpdf/fpdf.php';
if (is_file($fpdf_path)) {
  require_once $fpdf_path;

  class PDF extends FPDF {
    function Header(){
      $this->SetFont('Arial','B',14);
      $this->Cell(0,10,utf8_decode('Dashboard de Horas Extra'),0,1,'C');
      $this->Ln(3);
    }
    function Footer(){
      $this->SetY(-15);
      $this->SetFont('Arial','I',8);
      $this->Cell(0,10,utf8_decode('Página '.$this->PageNo()),0,0,'C');
    }
    function RowKV($k,$v){
      $this->SetFont('Arial','',10);
      $this->Cell(50,7,utf8_decode($k),0,0);
      $this->SetFont('Arial','B',10);
      $this->Cell(0,7,utf8_decode($v),0,1);
    }
    function Table2x2($a,$b,$c,$d){
      $this->SetFont('Arial','',10);
      $w=48; $h=8;
      $this->Cell($w,$h,utf8_decode($a[0]),1,0); $this->Cell($w,$h,$a[1],1,0,'R');
      $this->Cell($w,$h,utf8_decode($b[0]),1,0); $this->Cell($w,$h,$b[1],1,1,'R');
      $this->Cell($w,$h,utf8_decode($c[0]),1,0); $this->Cell($w,$h,$c[1],1,0,'R');
      $this->Cell($w,$h,utf8_decode($d[0]),1,1,'R');
    }
    function TableTop($rows){
      $this->SetFont('Arial','B',10);
      $this->Cell(120,8,utf8_decode('Colaborador'),1,0);
      $this->Cell(60,8,utf8_decode('Horas'),1,1,'R');
      $this->SetFont('Arial','',10);
      foreach($rows as $r){
        $this->Cell(120,8,utf8_decode($r['nombre']),1,0);
        $this->Cell(60,8,number_format($r['horas'],2,'.',','),1,1,'R');
      }
    }
  }

  $pdf = new PDF('P','mm','Letter');
  $pdf->SetMargins(15,15,15);
  $pdf->AddPage();

  // Meta
  $pdf->RowKV('Mes:', $mes);
  $pdf->RowKV('Ámbito:', $scope_desc);
  $pdf->RowKV('Horas/día:', (string)$pref['horas_dia'].'   Días/mes: '.(string)$pref['dias_mes']);
  if ($idempleado) $pdf->RowKV('Salario hora (prorrateo):', '₡'.number_format($sal_h,2,'.',','));
  $pdf->Ln(3);

  // KPIs HORAS
  $pdf->SetFont('Arial','B',12);
  $pdf->Cell(0,8,utf8_decode('KPIs – Horas'),0,1);
  $pdf->Table2x2(
    ['1.5x', number_format($tot['h15'],2,'.',',')],
    ['2.0x', number_format($tot['h20'],2,'.',',')],
    ['3.0x', number_format($tot['h30'],2,'.',',')],
    ['Total', number_format($tot['hTot'],2,'.',',')]
  );
  $pdf->Ln(3);

  // KPIs MONTOS
  $pdf->SetFont('Arial','B',12);
  $pdf->Cell(0,8,utf8_decode('KPIs – Montos'),0,1);
  $pdf->Table2x2(
    ['Monto 1.5x', '₡'.number_format($m15,2,'.',',')],
    ['Monto 2.0x', '₡'.number_format($m20,2,'.',',')],
    ['Monto 3.0x', '₡'.number_format($m30,2,'.',',')],
    ['Total a pagar', '₡'.number_format($mTot,2,'.',',')]
  );
  $pdf->Ln(5);

  // TOP 5
  $pdf->SetFont('Arial','B',12);
  $pdf->Cell(0,8,utf8_decode('Top 5 colaboradores'),0,1);
  $pdf->TableTop($top);

  $fname = 'dashboard_'.$mes.($idempleado?('_emp'.$idempleado):'').'.pdf';
  header('Content-Type: application/pdf');
  header('Content-Disposition: attachment; filename="'.$fname.'"');
  $pdf->Output('I', $fname);
  exit;
}

// --- fallback HTML imprimible (sin dependencias) ---
$fmt = fn($n)=>number_format((float)$n,2,'.',',');
?>
<!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Dashboard <?= htmlspecialchars($mes) ?></title>
  <style>
    body{font-family:Arial,Helvetica,sans-serif; margin:24px; color:#111}
    h1{margin:0 0 6px 0}
    h2{margin:16px 0 6px 0}
    table{border-collapse:collapse; width:100%; margin:8px 0}
    th,td{border:1px solid #ddd; padding:8px; font-size:13px}
    th{background:#f5f5f5; text-align:left}
    .kv{margin:2px 0}
    .muted{color:#666; font-size:12px}
    @media print { .hint{display:none} }
  </style>
</head>
<body>
  <h1>Dashboard de Horas Extra</h1>
  <div class="kv"><b>Mes:</b> <?= htmlspecialchars($mes) ?></div>
  <div class="kv"><b>Ámbito:</b> <?= htmlspecialchars($scope_desc) ?></div>
  <div class="kv"><b>Horas/día:</b> <?= $pref['horas_dia'] ?> &nbsp; <b>Días/mes:</b> <?= $pref['dias_mes'] ?></div>
  <?php if ($idempleado): ?>
    <div class="kv"><b>Salario hora (prorrateo):</b> ₡<?= $fmt($sal_h) ?></div>
  <?php endif; ?>

  <h2>KPIs – Horas</h2>
  <table>
    <tr><th>1.5x</th><th>2.0x</th><th>3.0x</th><th>Total</th></tr>
    <tr>
      <td><?= $fmt($tot['h15']) ?></td>
      <td><?= $fmt($tot['h20']) ?></td>
      <td><?= $fmt($tot['h30']) ?></td>
      <td><?= $fmt($tot['hTot']) ?></td>
    </tr>
  </table>

  <h2>KPIs – Montos</h2>
  <table>
    <tr><th>Monto 1.5x</th><th>Monto 2.0x</th><th>Monto 3.0x</th><th>Total a pagar</th></tr>
    <tr>
      <td>₡<?= $fmt($m15) ?></td>
      <td>₡<?= $fmt($m20) ?></td>
      <td>₡<?= $fmt($m30) ?></td>
      <td>₡<?= $fmt($mTot) ?></td>
    </tr>
  </table>

  <h2>Top 5 colaboradores</h2>
  <table>
    <tr><th>Colaborador</th><th>Horas</th></tr>
    <?php foreach($top as $r): ?>
      <tr><td><?= htmlspecialchars($r['nombre']) ?></td><td style="text-align:right"><?= $fmt($r['horas']) ?></td></tr>
    <?php endforeach; ?>
  </table>

  <p class="muted hint">Imprime esta página y elige “Guardar como PDF” si no tienes FPDF instalado.</p>
</body>
</html>
