<?php
// rrhh_home/export_dashboard_excel.php
// Exporta Dashboard a CSV (Excel-friendly) con filtro por mes e idempleado (opcional)

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');

if (!isset($_SESSION["idusuario"])) { http_response_code(401); exit('Sesión expirada'); }

require_once __DIR__ . '/dbcon.php';
if (!isset($con) || !($con instanceof mysqli)) { http_response_code(500); exit('Sin conexión BD'); }
mysqli_set_charset($con, 'utf8mb4');

$mes = $_GET['mes'] ?? date('Y-m');
if (!preg_match('/^\d{4}\-\d{2}$/', $mes)) $mes = date('Y-m');
$idempleado = isset($_GET['idempleado']) && $_GET['idempleado'] !== '' ? (int)$_GET['idempleado'] : null;

// --- helpers ligeros ---
function get_pref(mysqli $con): array {
  $pref=['horas_dia'=>8.0,'dias_mes'=>30.0];
  $qp="SELECT horas_dia, dias_mes FROM payroll_pref WHERE activo=1 ORDER BY id DESC LIMIT 1";
  if($rs=mysqli_query($con,$qp)){
    if($r=mysqli_fetch_assoc($rs)){
      $pref=['horas_dia'=>(float)$r['horas_dia'],'dias_mes'=>(float)$r['dias_mes']];
    }
    mysqli_free_result($rs);
  }
  return $pref;
}
function get_emp(mysqli $con, int $id): array {
  $out=['nombre'=>'','salario_bruto'=>0.0];
  if($st=mysqli_prepare($con,"SELECT nombre_completo, COALESCE(salario_bruto,0) FROM empleados_planilla WHERE id=?")){
    mysqli_stmt_bind_param($st,'i',$id);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$n,$sb);
    if(mysqli_stmt_fetch($st)) $out=['nombre'=>$n,'salario_bruto'=>(float)$sb];
    mysqli_stmt_close($st);
  }
  return $out;
}
function salario_hora(float $sb, array $pref): float {
  $h=($pref['horas_dia']??0)*($pref['dias_mes']??0);
  return $h>0?($sb/$h):0.0;
}

// --- totales por factor (filtrable) ---
$where = "WHERE m.mes = ?";
$params = [$mes]; $types = 's';
if ($idempleado) { $where .= " AND m.idempleado = ?"; $params[] = $idempleado; $types .= 'i'; }

$sqlTot = "
  SELECT
    COALESCE(SUM(CASE WHEN d.factor = 1.5 THEN d.horas ELSE 0 END),0) AS h15,
    COALESCE(SUM(CASE WHEN d.factor = 2.0 THEN d.horas ELSE 0 END),0) AS h20,
    COALESCE(SUM(CASE WHEN d.factor = 3.0 THEN d.horas ELSE 0 END),0) AS h30,
    COALESCE(SUM(d.horas),0) AS hTot
  FROM extras_detalle d
  INNER JOIN extras_mes m ON m.id = d.id_extras_mes
  $where
";
$tot = ['h15'=>0.0,'h20'=>0.0,'h30'=>0.0,'hTot'=>0.0];
if ($st = mysqli_prepare($con, $sqlTot)) {
  mysqli_stmt_bind_param($st, $types, ...$params);
  mysqli_stmt_execute($st);
  mysqli_stmt_bind_result($st,$a,$b,$c,$t);
  if (mysqli_stmt_fetch($st)) $tot=['h15'=>(float)$a,'h20'=>(float)$b,'h30'=>(float)$c,'hTot'=>(float)$t];
  mysqli_stmt_close($st);
}

// --- top 5 (filtrable; si hay empleado, será máx 1) ---
$sqlTop = "
  SELECT ep.nombre_completo, COALESCE(SUM(d.horas),0) AS horas
  FROM extras_detalle d
  INNER JOIN extras_mes m ON m.id = d.id_extras_mes
  INNER JOIN empleados_planilla ep ON ep.id = m.idempleado
  $where
  GROUP BY ep.id, ep.nombre_completo
  ORDER BY horas DESC
  LIMIT 5
";
$top = [];
if ($st = mysqli_prepare($con, $sqlTop)) {
  mysqli_stmt_bind_param($st, $types, ...$params);
  mysqli_stmt_execute($st);
  mysqli_stmt_bind_result($st,$n,$h);
  while (mysqli_stmt_fetch($st)) $top[] = ['nombre'=>$n,'horas'=>(float)$h];
  mysqli_stmt_close($st);
}

// --- montos (necesitamos salario_hora) ---
$pref = get_pref($con);
$sal_h = 0.0;
$scope_desc = 'General del mes';
if ($idempleado) {
  $emp = get_emp($con, $idempleado);
  $sal_h = salario_hora((float)$emp['salario_bruto'], $pref);
  $scope_desc = 'Empleado: ' . ($emp['nombre'] ?: ('ID '.$idempleado));
} else {
  // Promedio ponderado NO solicitado; mantenemos 0 para “general”
  $sal_h = 0.0;
}
$m15 = $tot['h15']*$sal_h*1.5;
$m20 = $tot['h20']*$sal_h*2.0;
$m30 = $tot['h30']*$sal_h*3.0;
$mTot= $m15+$m20+$m30;

// --- CSV STREAMING ---
$fname = 'dashboard_' . $mes . ($idempleado ? ('_emp'.$idempleado) : '') . '.csv';
header('Content-Type: text/csv; charset=utf-8');
header('Content-Disposition: attachment; filename="'.$fname.'"');
// BOM para Excel (UTF-8)
echo "\xEF\xBB\xBF";

$fp = fopen('php://output', 'w');

// Encabezado informativo
fputcsv($fp, ['Dashboard – Colegio', 'Mes', $mes]);
fputcsv($fp, ['Ámbito', $scope_desc]);
fputcsv($fp, ['Horas/día', $pref['horas_dia'], 'Días/mes', $pref['dias_mes']]);
fputcsv($fp, ['Salario hora (si aplica)', number_format($sal_h,2,'.','')]);
fputcsv($fp, []);

// Totales por factor
fputcsv($fp, ['KPIs HORAS']);
fputcsv($fp, ['Horas 1.5x','Horas 2.0x','Horas 3.0x','Total horas']);
fputcsv($fp, [
  number_format($tot['h15'],2,'.',''),
  number_format($tot['h20'],2,'.',''),
  number_format($tot['h30'],2,'.',''),
  number_format($tot['hTot'],2,'.','')
]);
fputcsv($fp, []);

// Montos (si hay salario hora)
fputcsv($fp, ['KPIs MONTOS (calculados con salario-hora)']);
fputcsv($fp, ['Monto 1.5x','Monto 2.0x','Monto 3.0x','Total a pagar']);
fputcsv($fp, [
  number_format($m15,2,'.',''),
  number_format($m20,2,'.',''),
  number_format($m30,2,'.',''),
  number_format($mTot,2,'.','')
]);
fputcsv($fp, []);

// Top 5
fputcsv($fp, ['Top 5 colaboradores por horas']);
fputcsv($fp, ['Colaborador','Horas']);
foreach($top as $row){
  fputcsv($fp, [$row['nombre'], number_format($row['horas'],2,'.','')]);
}
fclose($fp);
exit;
