<?php
// rrhh_home/empleado_panel_api.php
// API: cálculo integral por empleado+mes (bruto, extras, cargas social EMPLOYEE/EMPLOYER, ISR, neto, costo patrono)
// PHP 8.1.33 – UTF-8

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');
header('Content-Type: application/json; charset=utf-8');

if (!isset($_SESSION["idusuario"])) { echo json_encode(['ok'=>false,'msg'=>'Sesión expirada']); exit; }

require_once __DIR__ . '/dbcon.php';
if (!isset($con) || !($con instanceof mysqli)) { echo json_encode(['ok'=>false,'msg'=>'Sin conexión a BD']); exit; }
mysqli_set_charset($con,'utf8mb4');

$action = $_POST['action'] ?? $_GET['action'] ?? '';
function respond($arr){ echo json_encode($arr, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }
function valid_mes(string $mes): bool { return (bool)preg_match('/^\d{4}\-\d{2}$/',$mes); }

if ($action !== 'calc') { respond(['ok'=>false,'msg'=>'Acción no válida']); }

$idempleado = (int)($_POST['idempleado'] ?? 0);
$mes        = (string)($_POST['mes'] ?? date('Y-m'));
if ($idempleado<=0 || !valid_mes($mes)) respond(['ok'=>false,'msg'=>'Parámetros inválidos']);

/* ===== Datos del empleado ===== */
$emp=['id'=>$idempleado,'nombre'=>'','cedula'=>'','categoria'=>'','salario_bruto'=>0.0];
$qe = "SELECT nombre_completo, cedula, categoria, COALESCE(salario_bruto,0) AS salario_bruto FROM empleados_planilla WHERE id=?";
if($st = mysqli_prepare($con,$qe)){
  mysqli_stmt_bind_param($st,'i',$idempleado);
  mysqli_stmt_execute($st);
  mysqli_stmt_bind_result($st,$n,$c,$cat,$sb);
  if(mysqli_stmt_fetch($st)) $emp = ['id'=>$idempleado,'nombre'=>$n,'cedula'=>$c,'categoria'=>$cat,'salario_bruto'=>(float)$sb];
  mysqli_stmt_close($st);
}
if (!$emp['nombre']) respond(['ok'=>false,'msg'=>'Empleado no existe']);

/* ===== Preferencias (salario hora) ===== */
$pref=['horas_dia'=>8.0,'dias_mes'=>30.0];
$rs = mysqli_query($con,"SELECT horas_dia, dias_mes FROM payroll_pref WHERE activo=1 ORDER BY id DESC LIMIT 1");
if($rs){ if($r=mysqli_fetch_assoc($rs)){ $pref=['horas_dia'=>(float)$r['horas_dia'],'dias_mes'=>(float)$r['dias_mes']]; } mysqli_free_result($rs); }
$salario_hora = ($pref['horas_dia']>0 && $pref['dias_mes']>0) ? ($emp['salario_bruto']/($pref['horas_dia']*$pref['dias_mes'])) : 0.0;

/* ===== Reglas de horas extra (factores) ===== */
$rules = [
  'factor_regular'=>1.0,'factor_overtime'=>1.5,'factor_holiday'=>2.0,'factor_custom'=>3.0,
  'max_extra_por_dia'=>4.0,'max_extra_por_semana'=>12.0
];
$rs = mysqli_query($con,"SELECT factor_regular,factor_overtime,factor_holiday,factor_custom,max_extra_por_dia,max_extra_por_semana
                         FROM payroll_legal_rules WHERE activo=1 ORDER BY id DESC LIMIT 1");
if($rs && $r=mysqli_fetch_assoc($rs)){
  $rules['factor_regular']=(float)$r['factor_regular'];
  $rules['factor_overtime']=(float)$r['factor_overtime'];
  $rules['factor_holiday']=(float)$r['factor_holiday'];
  $rules['factor_custom']=(float)$r['factor_custom'];
  $rules['max_extra_por_dia']=(float)$r['max_extra_por_dia'];
  $rules['max_extra_por_semana']=(float)$r['max_extra_por_semana'];
  mysqli_free_result($rs);
}

/* ===== Totales de horas por factor en el mes ===== */
$tot=['h15'=>0.0,'h20'=>0.0,'h30'=>0.0,'hTot'=>0.0];
$sql = "SELECT
          COALESCE(SUM(CASE WHEN d.factor=1.5 THEN d.horas ELSE 0 END),0) AS h15,
          COALESCE(SUM(CASE WHEN d.factor=2.0 THEN d.horas ELSE 0 END),0) AS h20,
          COALESCE(SUM(CASE WHEN d.factor=3.0 THEN d.horas ELSE 0 END),0) AS h30,
          COALESCE(SUM(d.horas),0) AS hTot
        FROM extras_detalle d
        INNER JOIN extras_mes m ON m.id=d.id_extras_mes
        WHERE m.idempleado=? AND m.mes=?";
if($st=mysqli_prepare($con,$sql)){
  mysqli_stmt_bind_param($st,'is',$idempleado,$mes);
  mysqli_stmt_execute($st);
  mysqli_stmt_bind_result($st,$a,$b,$c,$t);
  if(mysqli_stmt_fetch($st)) $tot=['h15'=>(float)$a,'h20'=>(float)$b,'h30'=>(float)$c,'hTot'=>(float)$t];
  mysqli_stmt_close($st);
}

/* ===== Montos de extras con factores paramétricos ===== */
$m15 = $tot['h15'] * $salario_hora * (float)($rules['factor_overtime'] ?? 1.5);
$m20 = $tot['h20'] * $salario_hora * (float)($rules['factor_holiday']  ?? 2.0);
$m30 = $tot['h30'] * $salario_hora * (float)($rules['factor_custom']   ?? 3.0);
$extras_total = $m15 + $m20 + $m30;

/* ===== Conjunto legal activo + tasas ===== */
$set = null;
$rs = mysqli_query($con,"SELECT id, nombre, vigente_desde FROM payroll_legal_set WHERE activo=1 ORDER BY id DESC LIMIT 1");
if($rs && $r=mysqli_fetch_assoc($rs)){ $set=['id'=>(int)$r['id'],'nombre'=>$r['nombre'],'vigente_desde'=>$r['vigente_desde']]; mysqli_free_result($rs); }

$rates = ['EMPLOYEE'=>[],'EMPLOYER'=>[]];
if ($set){
  $st = mysqli_prepare($con,"SELECT side, code, nombre, pct FROM payroll_legal_rate WHERE set_id=? ORDER BY id ASC");
  mysqli_stmt_bind_param($st,'i',$set['id']);
  mysqli_stmt_execute($st);
  $rs = mysqli_stmt_get_result($st);
  while($row=mysqli_fetch_assoc($rs)){
    $side = $row['side'] === 'EMPLOYER' ? 'EMPLOYER' : 'EMPLOYEE';
    $rates[$side][] = ['code'=>$row['code'],'nombre'=>$row['nombre'],'pct'=>(float)$row['pct']];
  }
  mysqli_stmt_close($st);
}

/* ===== Aplicar cargas sociales ===== */
$base_general = $emp['salario_bruto'] + $extras_total;

$cargas_emp_items=[]; $tot_emp=0.0;
foreach($rates['EMPLOYEE'] as $it){
  $m = $base_general * ((float)$it['pct']/100.0);
  $cargas_emp_items[] = ['code'=>$it['code'],'nombre'=>$it['nombre'],'pct'=>$it['pct'],'monto'=>$m];
  $tot_emp += $m;
}

$cargas_pat_items=[]; $tot_pat=0.0;
foreach($rates['EMPLOYER'] as $it){
  $m = $base_general * ((float)$it['pct']/100.0);
  $cargas_pat_items[] = ['code'=>$it['code'],'nombre'=>$it['nombre'],'pct'=>$it['pct'],'monto'=>$m];
  $tot_pat += $m;
}

/* ===== ISR mensual (tramos progresivos) ===== */
$isr_base = max(0.0, $base_general - $tot_emp); // (bruto+extras) - cargas employee
$isr_monto = 0.0;
if ($set){
  $trs=[];
  $st = mysqli_prepare($con,"SELECT desde, hasta, pct FROM payroll_isr_tramo WHERE set_id=? ORDER BY desde ASC");
  mysqli_stmt_bind_param($st,'i',$set['id']);
  mysqli_stmt_execute($st);
  $rs = mysqli_stmt_get_result($st);
  while($row=mysqli_fetch_assoc($rs)){
    $trs[]=['desde'=>(float)$row['desde'],'hasta'=>$row['hasta']!==null?(float)$row['hasta']:null,'pct'=>(float)$row['pct']];
  }
  mysqli_stmt_close($st);

  $rem = $isr_base;
  foreach($trs as $tr){
    if ($rem<=0) break;
    $limInf = $tr['desde'];
    $limSup = $tr['hasta'] ?? $isr_base; // infinito
    if ($isr_base <= $limInf) continue;
    $montoRango = min($rem, max(0.0, ($limSup - max($limInf,0.0))));
    if ($tr['hasta'] === null) $montoRango = $rem;
    if ($montoRango > 0){
      $isr_monto += $montoRango * ($tr['pct']/100.0);
      $rem -= $montoRango;
    }
  }
}

/* ===== Neto y costo patrono ===== */
$neto_estimado = ($base_general - $tot_emp - $isr_monto);
$costo_patrono = ($base_general + $tot_pat);

/* ===== Responder ===== */
respond([
  'ok' => true,
  'empleado' => $emp,
  'pref' => $pref,
  'salario_hora' => round($salario_hora,4),
  'legal_set' => $set ?: (object)[],
  'rules' => $rules,
  'totales' => $tot,
  'montos' => ['m15'=>$m15,'m20'=>$m20,'m30'=>$m30,'mTot'=>$extras_total],
  'cargas' => [
    'employee' => ['items'=>$cargas_emp_items, 'total'=>$tot_emp],
    'employer' => ['items'=>$cargas_pat_items, 'total'=>$tot_pat]
  ],
  'isr' => ['base'=>$isr_base, 'monto'=>$isr_monto],
  'neto_estimado' => $neto_estimado,
  'costo_patrono' => $costo_patrono
]);
