<?php
// rrhh_home/ausencias_api.php
// API v1 – Ausencias/Retrasos (editor mensual tipo calendario)
// Acciones: get_month, upsert_day, delete_day
// PHP 8.1.33 – UTF-8

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');
header('Content-Type: application/json; charset=utf-8');

if (!isset($_SESSION["idusuario"])) { echo json_encode(['ok'=>false,'msg'=>'Sesión expirada']); exit; }

require_once __DIR__ . '/dbcon.php';
if (!isset($con) || !($con instanceof mysqli)) { echo json_encode(['ok'=>false,'msg'=>'Sin conexión a BD']); exit; }
mysqli_set_charset($con,'utf8mb4');

/* ===== Permisos ===== */
$READ_ONLY = false;
if (isset($_SESSION['rol']) && $_SESSION['rol'] === 'viewer') $READ_ONLY = true;

$action = $_POST['action'] ?? $_GET['action'] ?? '';
function respond($arr){ echo json_encode($arr, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }
function valid_mes(string $mes): bool { return (bool)preg_match('/^\d{4}\-\d{2}$/',$mes); }

/* ================== ROUTER ================== */
switch ($action) {

  case 'get_month': {
    $idempleado = (int)($_POST['idempleado'] ?? 0);
    $mes = $_POST['mes'] ?? date('Y-m');
    if ($idempleado<=0 || !valid_mes($mes)) respond(['ok'=>false,'msg'=>'Parámetros inválidos']);

    $id_am = ensure_header($con, $idempleado, $mes);

    // detalle del mes
    $detalle=[];
    $sql="SELECT id, fecha, tipo, horas, descripcion
          FROM ausencias_detalle
          WHERE id_ausencias_mes=?
          ORDER BY fecha ASC";
    if($st=mysqli_prepare($con,$sql)){
      mysqli_stmt_bind_param($st,'i',$id_am);
      mysqli_stmt_execute($st);
      $rs=mysqli_stmt_get_result($st);
      while($r=mysqli_fetch_assoc($rs)){
        $detalle[]=[
          'id'=>(int)$r['id'],
          'fecha'=>$r['fecha'],
          'tipo'=>$r['tipo'],
          'horas'=>(float)$r['horas'],
          'descripcion'=>$r['descripcion'] ?? ''
        ];
      }
      mysqli_stmt_close($st);
    }

    $pref   = get_pref($con); // horas_dia, dias_mes
    $tot    = compute_totals($con, $id_am); // métricas
    $emp    = get_empleado_basic($con, $idempleado);

    respond([
      'ok'=>true,
      'empleado'=>$emp,
      'pref'=>$pref,
      'detalle'=>$detalle,
      'totales'=>$tot,
      'readonly'=>$READ_ONLY
    ]);
  } break;

  case 'upsert_day': {
    if ($READ_ONLY) respond(['ok'=>false,'msg'=>'Sin permiso para modificar']);

    $idempleado = (int)($_POST['idempleado'] ?? 0);
    $mes   = $_POST['mes'] ?? '';
    $fecha = $_POST['fecha'] ?? '';
    $tipo  = strtoupper(trim((string)($_POST['tipo'] ?? 'AUSENCIA'))); // 'AUSENCIA'|'RETRASO'
    $horas = isset($_POST['horas']) ? (float)$_POST['horas'] : 0.0;
    $desc  = trim((string)($_POST['descripcion'] ?? ''));

    if ($idempleado<=0 || !valid_mes($mes) || !preg_match('/^\d{4}\-\d{2}\-\d{2}$/',$fecha)) {
      respond(['ok'=>false,'msg'=>'Parámetros inválidos']);
    }
    if (!in_array($tipo,['AUSENCIA','RETRASO'],true)) respond(['ok'=>false,'msg'=>'Tipo inválido']);
    if (strpos($fecha,$mes)!==0) respond(['ok'=>false,'msg'=>'La fecha no pertenece al mes']);
    if (strlen($desc)>255) $desc = substr($desc,0,255);

    $pref = get_pref($con);
    $horas_dia = (float)($pref['horas_dia'] ?? 8.0);
    if ($horas_dia<=0) $horas_dia=8.0;

    // Validación flexible:
    // - AUSENCIA: si horas<=0 → usar horas_dia; si >0, limitar a [0, horas_dia]
    // - RETRASO:  horas debe estar en (0, horas_dia]
    if ($tipo==='AUSENCIA') {
      if ($horas <= 0) $horas = $horas_dia;
      if ($horas > $horas_dia) $horas = $horas_dia;
    } else { // RETRASO
      if ($horas <= 0 || $horas > $horas_dia) {
        respond(['ok'=>false,'msg'=>'Horas de retraso inválidas (debe ser >0 y <= horas del día)']);
      }
    }

    $id_am = ensure_header($con, $idempleado, $mes);
    $uid   = (int)($_SESSION['idusuario'] ?? 0);

    // old para log
    $old=['tipo'=>null,'horas'=>null,'descripcion'=>null];
    if($st0=mysqli_prepare($con,"SELECT tipo, horas, descripcion FROM ausencias_detalle WHERE id_ausencias_mes=? AND fecha=? LIMIT 1")){
      mysqli_stmt_bind_param($st0,'is',$id_am,$fecha);
      mysqli_stmt_execute($st0);
      mysqli_stmt_bind_result($st0,$ot,$oh,$od);
      if(mysqli_stmt_fetch($st0)) $old=['tipo'=>$ot,'horas'=>$oh,'descripcion'=>$od];
      mysqli_stmt_close($st0);
    }

    // UPSERT robusto (no requiere índice único):
    // 1) UPDATE por (cabecera, fecha)
    $aff = 0;
    if($stU=mysqli_prepare($con,"UPDATE ausencias_detalle SET tipo=?, horas=?, descripcion=?, updated_by=?, updated_at=NOW() WHERE id_ausencias_mes=? AND fecha=?")){
      mysqli_stmt_bind_param($stU,'sdsiss',$tipo,$horas,$desc,$uid,$id_am,$fecha);
      mysqli_stmt_execute($stU);
      $aff = mysqli_stmt_affected_rows($stU);
      mysqli_stmt_close($stU);
    }

    // 2) Si no existía fila, INSERT
    if ($aff===0) {
      if($stI=mysqli_prepare($con,"INSERT INTO ausencias_detalle (id_ausencias_mes, fecha, tipo, horas, descripcion, creado_por) VALUES (?,?,?,?,?,?)")){
        mysqli_stmt_bind_param($stI,'issdsi',$id_am,$fecha,$tipo,$horas,$desc,$uid);
        if(!mysqli_stmt_execute($stI)){
          respond(['ok'=>false,'msg'=>'No se pudo guardar (INSERT): '.mysqli_error($con)]);
        }
        mysqli_stmt_close($stI);
      } else {
        respond(['ok'=>false,'msg'=>'No se pudo preparar INSERT: '.mysqli_error($con)]);
      }
    }

    // Log del cambio
    log_change($con,$id_am,$fecha,$old['tipo'],$old['horas'],$old['descripcion'],$tipo,$horas,$desc,$uid);

    $tot = compute_totals($con,$id_am);
    respond(['ok'=>true,'totales'=>$tot]);
  } break;

  case 'delete_day': {
    if ($READ_ONLY) respond(['ok'=>false,'msg'=>'Sin permiso para modificar']);

    $idempleado = (int)($_POST['idempleado'] ?? 0);
    $mes   = $_POST['mes'] ?? '';
    $fecha = $_POST['fecha'] ?? '';

    if ($idempleado<=0 || !valid_mes($mes) || !preg_match('/^\d{4}\-\d{2}\-\d{2}$/',$fecha)) {
      respond(['ok'=>false,'msg'=>'Parámetros inválidos']);
    }
    $id_am = ensure_header($con, $idempleado, $mes);

    // Log guardar old si existía
    $old=['tipo'=>null,'horas'=>null,'descripcion'=>null];
    if($st0=mysqli_prepare($con,"SELECT tipo, horas, descripcion FROM ausencias_detalle WHERE id_ausencias_mes=? AND fecha=? LIMIT 1")){
      mysqli_stmt_bind_param($st0,'is',$id_am,$fecha);
      mysqli_stmt_execute($st0);
      mysqli_stmt_bind_result($st0,$ot,$oh,$od);
      if(mysqli_stmt_fetch($st0)) $old=['tipo'=>$ot,'horas'=>$oh,'descripcion'=>$od];
      mysqli_stmt_close($st0);
    }

    if($st=mysqli_prepare($con,"DELETE FROM ausencias_detalle WHERE id_ausencias_mes=? AND fecha=?")){
      mysqli_stmt_bind_param($st,'is',$id_am,$fecha);
      mysqli_stmt_execute($st);
      mysqli_stmt_close($st);
    } else {
      respond(['ok'=>false,'msg'=>'No se pudo borrar: '.mysqli_error($con)]);
    }

    // Registra log con new_* en cero/nulos
    log_change($con,$id_am,$fecha,$old['tipo'],$old['horas'],$old['descripcion'], null, 0.0, null,(int)($_SESSION['idusuario']??0));

    $tot = compute_totals($con,$id_am);
    respond(['ok'=>true,'totales'=>$tot]);
  } break;

  default:
    respond(['ok'=>false,'msg'=>'Acción no válida']);
}

/* ================= Helpers ================= */

function ensure_header(mysqli $con, int $idempleado, string $mes): int {
  $id=0;
  $q="SELECT id FROM ausencias_mes WHERE idempleado=? AND mes=? LIMIT 1";
  if($st=mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'is',$idempleado,$mes);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$tmp);
    if(mysqli_stmt_fetch($st)) $id=(int)$tmp;
    mysqli_stmt_close($st);
  }
  if($id>0) return $id;

  $uid=(int)($_SESSION['idusuario'] ?? 0);
  $ins="INSERT INTO ausencias_mes (idempleado, mes, creado_por) VALUES (?,?,?)";
  if($st=mysqli_prepare($con,$ins)){
    mysqli_stmt_bind_param($st,'isi',$idempleado,$mes,$uid);
    mysqli_stmt_execute($st);
    $id=(int)mysqli_insert_id($con);
    mysqli_stmt_close($st);
  }
  return $id;
}

function get_pref(mysqli $con): array {
  $pref=['horas_dia'=>8.0,'dias_mes'=>30.0];
  $q="SELECT horas_dia, dias_mes FROM payroll_pref WHERE activo=1 ORDER BY id DESC LIMIT 1";
  if($rs=mysqli_query($con,$q)){
    if($r=mysqli_fetch_assoc($rs)){
      $pref=['horas_dia'=>(float)$r['horas_dia'],'dias_mes'=>(float)$r['dias_mes']];
    }
    mysqli_free_result($rs);
  }
  return $pref;
}

function get_empleado_basic(mysqli $con, int $idempleado): array {
  $emp=['id'=>$idempleado,'nombre'=>'','cedula'=>'','categoria'=>'','salario_bruto'=>0.0];
  $q="SELECT nombre_completo, cedula, categoria, COALESCE(salario_bruto,0) sb
      FROM empleados_planilla WHERE id=?";
  if($st=mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'i',$idempleado);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$n,$c,$cat,$sb);
    if(mysqli_stmt_fetch($st)) $emp=['id'=>$idempleado,'nombre'=>$n,'cedula'=>$c,'categoria'=>$cat,'salario_bruto'=>(float)$sb];
    mysqli_stmt_close($st);
  }
  return $emp;
}

function compute_totals(mysqli $con, int $id_am): array {
  $out=['dias_ausencia'=>0,'horas_ausencia'=>0.0,'eventos_retraso'=>0,'horas_retraso'=>0.0,'horas_descuento_total'=>0.0];
  $q="SELECT
        SUM(CASE WHEN tipo='AUSENCIA' THEN 1 ELSE 0 END) AS dias_aus,
        COALESCE(SUM(CASE WHEN tipo='AUSENCIA' THEN horas ELSE 0 END),0) AS h_aus,
        SUM(CASE WHEN tipo='RETRASO'  THEN 1 ELSE 0 END) AS ev_ret,
        COALESCE(SUM(CASE WHEN tipo='RETRASO'  THEN horas ELSE 0 END),0) AS h_ret
      FROM ausencias_detalle
      WHERE id_ausencias_mes=?";
  if($st=mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'i',$id_am);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$dA,$hA,$eR,$hR);
    if(mysqli_stmt_fetch($st)){
      $out=[
        'dias_ausencia'=>(int)$dA,
        'horas_ausencia'=>(float)$hA,
        'eventos_retraso'=>(int)$eR,
        'horas_retraso'=>(float)$hR,
        'horas_descuento_total'=> (float)$hA + (float)$hR
      ];
    }
    mysqli_stmt_close($st);
  }
  return $out;
}

function log_change(mysqli $con, int $id_am, string $fecha,
  $old_tipo, $old_horas, $old_desc, $new_tipo, $new_horas, $new_desc, int $uid): void {

  $sql="INSERT INTO ausencias_detalle_log
        (id_ausencias_mes, fecha, old_tipo, old_horas, old_desc, new_tipo, new_horas, new_desc, changed_by)
        VALUES (?,?,?,?,?,?,?,?,?)";
  if($st=mysqli_prepare($con,$sql)){
    mysqli_stmt_bind_param($st,'issdssdsi',
      $id_am, $fecha,
      $old_tipo, $old_horas, $old_desc,
      $new_tipo, $new_horas, $new_desc,
      $uid
    );
    mysqli_stmt_execute($st);
    mysqli_stmt_close($st);
  }
}
