<?php
declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');

if (!isset($_SESSION['gestor_id']) || (int)$_SESSION['gestor_id'] <= 0) {
  header('Location: login.php'); exit;
}
$gestorNombre = (string)($_SESSION['gestor_nombre'] ?? 'Gestor');

$isHttps = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
  || (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] === '443');

$host    = $_SERVER['HTTP_HOST'] ?? '';
$uri     = $_SERVER['REQUEST_URI'] ?? '/asistencia/scan.php';

if (!$isHttps && stripos($host, 'hr.jportales.com') !== false) {
  header('Location: https://' . $host . $uri, true, 301);
  exit;
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <meta http-equiv="X-UA-Compatible" content="IE=edge">
  <title>Asistencia | Escanear</title>
  <meta content="width=device-width, initial-scale=1, maximum-scale=1, user-scalable=no" name="viewport">

  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/twitter-bootstrap/3.4.1/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/4.7.0/css/font-awesome.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/2.4.18/css/AdminLTE.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/2.4.18/css/skins/skin-blue.min.css">

  <style>
    .user-image-circle{ width:28px;height:28px;border-radius:50%;object-fit:cover;border:2px solid rgba(255,255,255,.7); }
    .badge-pill{ display:inline-block;padding:6px 10px;border-radius:999px;background:#f3f4f6;border:1px solid #e5e7eb;font-weight:800; }

    .scan-card{ background:#fff; border:1px solid #e5e7eb; border-radius:12px; box-shadow:0 6px 18px rgba(0,0,0,.06); padding:16px; }
    .actions{ display:flex; gap:10px; flex-wrap:wrap; align-items:center; margin-bottom:10px; }
    .btnx{ border-radius:10px; font-weight:800; padding:10px 14px; }
    .btnx i{ margin-right:6px; }
    .videoWrap{ border:1px solid #e5e7eb; border-radius:12px; overflow:hidden; background:#000; min-height:280px; }
    video{ width:100%; height:auto; display:block; object-fit:cover; }
    .statusLine{ margin-top:10px; font-size:13px; color:#374151; }
    .statusLine .ok{ color:#065f46; font-weight:800; }
    .statusLine .bad{ color:#b91c1c; font-weight:800; }
    .result{ display:none; margin-top:10px; border-radius:10px; }

    .stateBox{ display:flex; gap:12px; align-items:center; flex-wrap:wrap; padding:12px; border-radius:12px; border:1px solid #e5e7eb; background:#fafafa; margin-bottom:12px; }
    .stateBadge{ padding:10px 14px; border-radius:12px; font-weight:900; font-size:16px; display:inline-flex; align-items:center; gap:10px; }
    .sb-green{ background:#dcfce7; border:1px solid #86efac; color:#065f46; }
    .sb-amber{ background:#fffbeb; border:1px solid #fcd34d; color:#92400e; }
    .sb-red{ background:#fee2e2; border:1px solid #fca5a5; color:#991b1b; }
    .sb-gray{ background:#e5e7eb; border:1px solid #d1d5db; color:#111827; }

    .pill{ display:inline-block; padding:6px 10px; border-radius:999px; background:#eef2ff; border:1px solid #c7d2fe; color:#1e3a8a; font-weight:900; }
    .meta{ color:#6b7280; font-size:13px; }
    .hint{ color:#6b7280; margin:8px 0 14px; }
    .smallNote{ font-size:12px; color:#6b7280; margin-top:8px; }

    /* Mensaje de bloqueo fuerte (igual estilo manual) */
    .hardLockNote{
      margin-bottom:12px;
      border-radius:12px;
      border:2px solid #f59e0b;
      background:#fffbeb;
      padding:12px 14px;
      color:#92400e;
      font-weight:900;
      box-shadow:0 10px 22px rgba(0,0,0,.08);
    }

    .menu-disabled > a{ opacity:.45; pointer-events:none; cursor:not-allowed; }
    .menu-disabled a{ pointer-events:none; }

    .btn-manual{
      border-radius:12px !important;
      font-weight:900 !important;
      padding:10px 16px !important;
      border:2px solid rgba(255,255,255,.35) !important;
      box-shadow:0 10px 22px rgba(0,0,0,.12);
      position:relative;
      overflow:hidden;
    }
    .btn-manual:before{
      content:'';
      position:absolute; top:-40%; left:-40%;
      width:80%; height:180%;
      background:rgba(255,255,255,.18);
      transform:rotate(25deg);
      transition:all .25s ease;
    }
    .btn-manual:hover:before{ left:20%; }
    .btn-manual i{ margin-right:8px !important; }
    .manual-sub{
      display:inline-block;
      font-size:11px;
      font-weight:800;
      opacity:.9;
      margin-left:8px;
      padding:2px 8px;
      border-radius:999px;
      background:rgba(255,255,255,.22);
      border:1px solid rgba(255,255,255,.25);
    }
  </style>
</head>

<body class="hold-transition skin-blue sidebar-mini">
<div class="wrapper">

  <header class="main-header">
    <a href="dashboard.php" class="logo" id="logoTop">
      <span class="logo-mini"><b>G</b></span>
      <span class="logo-lg"><b>Gestores</b> Asistencia</span>
    </a>

    <nav class="navbar navbar-static-top" role="navigation">
      <a href="#" class="sidebar-toggle" data-toggle="push-menu" role="button">
        <span class="sr-only">Toggle navigation</span>
      </a>

      <div class="navbar-custom-menu">
        <ul class="nav navbar-nav">
          <li class="dropdown user user-menu">
            <a href="#" class="dropdown-toggle" data-toggle="dropdown">
              <img src="https://via.placeholder.com/80x80.png?text=U" class="user-image user-image-circle" alt="User">
              <span class="hidden-xs"><?= htmlspecialchars($gestorNombre, ENT_QUOTES, 'UTF-8') ?></span>
            </a>
            <ul class="dropdown-menu">
              <li class="user-header">
                <img src="https://via.placeholder.com/120x120.png?text=U" class="img-circle" alt="User">
                <p style="margin-top:10px;">
                  <?= htmlspecialchars($gestorNombre, ENT_QUOTES, 'UTF-8') ?><br>
                  <small id="topEstado">Estado: --</small><br>
                  <small id="topUltimo">Último: --</small>
                </p>
              </li>
              <li class="user-footer">
                <div class="pull-left">
                  <a href="dashboard.php" class="btn btn-default btn-flat" id="topBtnDash" style="display:none;">Dashboard</a>
                </div>
                <div class="pull-right">
                  <a href="logout.php" class="btn btn-default btn-flat">Salir</a>
                </div>
              </li>
            </ul>
          </li>
        </ul>
      </div>
    </nav>
  </header>

  <aside class="main-sidebar">
    <section class="sidebar">

      <div class="user-panel">
        <div class="pull-left image">
          <img src="https://via.placeholder.com/120x120.png?text=U" class="img-circle" alt="User">
        </div>
        <div class="pull-left info">
          <p style="margin-bottom:2px;"><?= htmlspecialchars($gestorNombre, ENT_QUOTES, 'UTF-8') ?></p>
          <a href="#"><i class="fa fa-circle text-success"></i> <span id="sideEstado">--</span></a>
        </div>
      </div>

      <ul class="sidebar-menu" data-widget="tree">
        <li class="header">MENÚ</li>

        <li id="menuDash" class="menu-disabled">
          <a href="dashboard.php"><i class="fa fa-dashboard"></i> <span>Dashboard</span></a>
        </li>

        <li class="treeview active">
          <a href="#"><i class="fa fa-qrcode"></i> <span>Scaneo asistencia</span>
            <span class="pull-right-container"><i class="fa fa-angle-left pull-right"></i></span>
          </a>
          <ul class="treeview-menu">
            <li class="active"><a href="scan.php"><i class="fa fa-camera"></i> Escanear</a></li>
            <li id="menuHist" class="menu-disabled"><a href="historial.php"><i class="fa fa-list"></i> Historial</a></li>
          </ul>
        </li>

        <li class="treeview">
          <a href="#"><i class="fa fa-wrench"></i> <span>Herramientas</span>
            <span class="pull-right-container"><i class="fa fa-angle-left pull-right"></i></span>
          </a>
          <ul class="treeview-menu">
            <li><a href="convertidor_pdf.php"><i class="fa fa-file-pdf-o"></i> Convertidor PDF</a></li>
          </ul>
        </li>

        <li><a href="recordatorios.php"><i class="fa fa-bell"></i> <span>Recordatorios</span></a></li>
        <li><a href="chat.php"><i class="fa fa-comments"></i> <span>Chat</span></a></li>

        <li><a href="logout.php"><i class="fa fa-sign-out"></i> <span>Salir</span></a></li>
      </ul>

    </section>
  </aside>

  <div class="content-wrapper">
    <section class="content-header">
      <h1><i class="fa fa-qrcode"></i> Escanear QR <small>Registro automático</small></h1>
    </section>

    <section class="content">

      <!-- 🔒 mensaje lock duro -->
      <div id="hardLockNote" class="hardLockNote" style="display:none;"></div>

      <div class="scan-card">

        <div class="stateBox">
          <span id="badge" class="stateBadge sb-gray"><i class="fa fa-question-circle"></i> Estado: --</span>
          <span class="pill" id="lastMark">Último marcaje: --</span>
          <span class="meta" id="meta">Cargando estado actual…</span>
        </div>

        <div class="actions">
          <button id="btnStart" class="btn btn-primary btnx"><i class="fa fa-play-circle"></i> Activar cámara</button>

          <a id="btnManual" href="registro_manual.php" class="btn btn-success btnx btn-manual">
            <i class="fa fa-map-marker"></i> Registro manual
            <span class="manual-sub">GPS + Foto</span>
          </a>

          <button id="btnRetry" class="btn btn-warning btnx" style="display:none;"><i class="fa fa-refresh"></i> Reintentar</button>
          <button id="btnStop"  class="btn btn-default btnx" disabled><i class="fa fa-stop"></i> Detener</button>

          <select id="camSel" class="form-control" style="max-width:360px;" disabled></select>

          <span id="supportPill" class="pill">Detector: verificando…</span>
        </div>

        <div class="hint">
          Apunta la cámara al QR de la pantalla pública.
        </div>

        <div class="videoWrap">
          <video id="video" playsinline></video>
        </div>

        <div class="statusLine" id="status">Listo. Presiona <b>Activar cámara</b>.</div>
        <?php if (!$isHttps): ?>
          <div class="statusLine"><span class="bad">⚠ Debes usar HTTPS para acceder a la cámara.</span></div>
        <?php endif; ?>

        <div class="result alert" id="result"></div>

        <div class="smallNote">
          Si el navegador bloquea la cámara: candado → Cámara → Permitir.
        </div>

      </div>

    </section>
  </div>

  <footer class="main-footer" style="font-size:12px;">
    <div class="pull-right hidden-xs">Asistencia Gestores</div>
    <strong>© <?= date('Y') ?> JPortales</strong>
  </footer>

</div>

<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.7.1/jquery.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/twitter-bootstrap/3.4.1/js/bootstrap.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/2.4.18/js/adminlte.min.js"></script>

<script>
const statusEl = document.getElementById('status');
const resEl    = document.getElementById('result');
const video    = document.getElementById('video');

const btnStart = document.getElementById('btnStart');
const btnStop  = document.getElementById('btnStop');
const btnRetry = document.getElementById('btnRetry');

const camSel   = document.getElementById('camSel');
const pill     = document.getElementById('supportPill');

const badgeEl  = document.getElementById('badge');
const lastEl   = document.getElementById('lastMark');
const metaEl   = document.getElementById('meta');

const topEstado = document.getElementById('topEstado');
const topUltimo = document.getElementById('topUltimo');
const sideEstado = document.getElementById('sideEstado');
const topBtnDash = document.getElementById('topBtnDash');

const menuDash = document.getElementById('menuDash');
const menuHist = document.getElementById('menuHist');

const btnManual = document.getElementById('btnManual');

const hardLockNote = document.getElementById('hardLockNote');

let stream = null;
let detector = null;
let scanning = false;

let lastToken = '';
let lastTokenTs = 0;
let lastMarkTs = 0;

let currentEstado = 'SIN_MARCA';
let HARD_LOCK = false; // 🔒 candado duro por horas extra

function ok(m){ statusEl.innerHTML = '<span class="ok">'+m+'</span>'; }
function bad(m){ statusEl.innerHTML = '<span class="bad">'+m+'</span>'; }
function showResult(kind, html){
  resEl.className = 'result alert ' + (kind==='ok' ? 'alert-success' : 'alert-danger');
  resEl.style.display = 'block';
  resEl.innerHTML = html;
}
function hideResult(){ resEl.style.display='none'; resEl.innerHTML=''; }

function setBadge(estado){
  currentEstado = estado || 'SIN_MARCA';
  let cls='sb-gray', icon='fa-question-circle', text='Estado: SIN MARCA';
  if (estado === 'WORKING'){ cls='sb-green'; icon='fa-check-circle'; text='Estado: TRABAJANDO'; }
  else if (estado === 'AWAY'){ cls='sb-amber'; icon='fa-pause-circle'; text='Estado: AUSENTE'; }
  else if (estado === 'DONE'){ cls='sb-red'; icon='fa-flag-checkered'; text='Estado: FINALIZADO'; }
  badgeEl.className = 'stateBadge ' + cls;
  badgeEl.innerHTML = `<i class="fa ${icon}"></i> ${text}`;
}

function estadoES(s){
  if (s === 'WORKING') return 'TRABAJANDO';
  if (s === 'AWAY') return 'AUSENTE / PAUSA';
  if (s === 'DONE') return 'FINALIZADO';
  return s || '--';
}

function eventoES(ev){
  if (ev === 'IN_INICIO') return 'ENTRADA (INICIO)';
  if (ev === 'IN_REANUDA') return 'ENTRADA (REANUDACIÓN)';
  if (ev === 'OUT_PENDING') return 'SALIDA (PAUSA PENDIENTE)';
  if (ev === 'OUT_PAUSE') return 'SALIDA (PAUSA)';
  if (ev === 'OUT_FINAL') return 'SALIDA FINAL';
  if (ev === 'OUT_LUNCH') return 'SALIDA ALMUERZO';
  if (ev === 'IN_LUNCH') return 'ENTRADA ALMUERZO';
  if (ev === 'OUT_CAFE1') return 'SALIDA CAFÉ 1';
  if (ev === 'IN_CAFE1') return 'ENTRADA CAFÉ 1';
  if (ev === 'OUT_CAFE2') return 'SALIDA CAFÉ 2';
  if (ev === 'IN_CAFE2') return 'ENTRADA CAFÉ 2';
  return ev || '-';
}

function esEventoEntrada(ev){
  return (ev === 'IN_INICIO' || ev === 'IN_REANUDA');
}

// 🔒 BLOQUEO NORMAL por estado (dashboard/historial)
function aplicarBloqueoPanel(estado){
  const working = (estado === 'WORKING');
  if (menuDash) menuDash.classList.toggle('menu-disabled', !working);
  if (menuHist) menuHist.classList.toggle('menu-disabled', !working);
  if (topBtnDash) topBtnDash.style.display = working ? 'inline-block' : 'none';
}

// 🔒 BLOQUEO DURO (horas extra activas desde ayer/hoy)
function aplicarHardLock(msg){
  HARD_LOCK = true;

  // Mostrar mensaje grande
  hardLockNote.style.display = 'block';
  hardLockNote.innerHTML = `
    ${msg}<br>
    <span style="font-weight:800; display:block; margin-top:6px;">
      ⚠ Hasta que NO se cierre esa Hora Extra, no podrás usar: Cámara, Registro manual, Dashboard ni Historial.
    </span>
    <div style="margin-top:10px;">
      <button class="btn btn-warning btnx" onclick="window.location.reload()">
        <i class="fa fa-refresh"></i> Actualizar / Reintentar
      </button>
    </div>
  `;

  // Apagar cámara si estaba prendida
  stopCamera();

  // Deshabilitar TODO
  btnStart.disabled = true;
  btnStop.disabled = true;
  camSel.disabled = true;

  if (btnManual) {
    btnManual.style.pointerEvents = 'none';
    btnManual.style.opacity = '.45';
  }

  // Forzar menús deshabilitados
  if (menuDash) menuDash.classList.add('menu-disabled');
  if (menuHist) menuHist.classList.add('menu-disabled');
  if (topBtnDash) topBtnDash.style.display = 'none';

  // Mensaje estado
  bad('Bloqueado por Horas Extra ACTIVAS (debe cerrarse primero).');
}

// ===================== CAMERA =====================
function stopCamera(){
  scanning = false;
  if (stream){
    stream.getTracks().forEach(t=>t.stop());
    stream = null;
  }
  video.srcObject = null;
  btnStop.disabled = true;
  btnStart.disabled = false;
  camSel.disabled = false;
}

async function enumerateCams(){
  const devs = await navigator.mediaDevices.enumerateDevices();
  const cams = devs.filter(d => d.kind === 'videoinput');
  camSel.innerHTML = '';
  cams.forEach((c, i)=>{
    const opt = document.createElement('option');
    opt.value = c.deviceId;
    opt.textContent = c.label || ('Cámara ' + (i+1));
    camSel.appendChild(opt);
  });
  camSel.disabled = cams.length === 0;
  const saved = localStorage.getItem('asist_cam_id');
  if (saved && cams.some(x=>x.deviceId===saved)) camSel.value = saved;
  return cams.length;
}

async function startCamera(deviceId=null){
  if (HARD_LOCK) return; // 🔒 no permitir

  hideResult();
  btnRetry.style.display = 'none';

  if (!navigator.mediaDevices?.getUserMedia){
    bad('Tu navegador no soporta cámara (getUserMedia).');
    return;
  }

  if (!('BarcodeDetector' in window)) {
    pill.textContent = 'Detector: NO soportado';
    bad('Este navegador NO soporta detector QR. Usa Chrome (Android/PC).');
    return;
  }

  pill.textContent = 'Detector: OK';
  detector = new BarcodeDetector({formats: ['qr_code']});

  const constraints = {
    audio: false,
    video: deviceId
      ? { deviceId: { exact: deviceId } }
      : { facingMode: { ideal: 'environment' } }
  };

  try{
    stream = await navigator.mediaDevices.getUserMedia(constraints);
    video.srcObject = stream;
    await video.play();

    btnStop.disabled = false;
    btnStart.disabled = true;

    try{
      const n = await enumerateCams();
      if (n > 0) camSel.disabled = false;
      if (camSel.value) localStorage.setItem('asist_cam_id', camSel.value);
    }catch(_){}

    ok('Cámara activa. Escaneando…');
    scanning = true;
    scanLoop();
  }catch(e){
    btnRetry.style.display = 'inline-block';
    bad('No se pudo activar la cámara. Reintentar.');
  }
}

async function marcarToken(token){
  if (HARD_LOCK) return; // 🔒 no permitir

  const now = Date.now();
  if (token === lastToken && (now - lastTokenTs) < 2500) return;
  lastToken = token; lastTokenTs = now;
  if ((now - lastMarkTs) < 1800) return;
  lastMarkTs = now;

  ok('QR leído. Registrando…');

  const fd = new FormData();
  fd.append('action','auto_mark');
  fd.append('token', token);

  try{
    const r = await fetch('jornada_api.php', { method:'POST', body: fd });
    const j = await r.json();

    // 🔒 si el API dice lock duro
    if (j && j.ok === false && j.code === 'OVERTIME_LOCK') {
      aplicarHardLock(j.msg || '🔒 BLOQUEADO: Hora Extra ACTIVA.');
      showResult('err', `<b>${j.msg || 'Bloqueado'}</b>`);
      return;
    }

    if (!j || !j.ok) throw new Error(j?.msg || 'Error');

    setBadge(j.estado || 'SIN_MARCA');
    aplicarBloqueoPanel(j.estado || '');

    if (topEstado) topEstado.textContent = 'Estado: ' + estadoES(j.estado || '');
    if (sideEstado) sideEstado.textContent = estadoES(j.estado || '');

    lastEl.textContent = 'Último marcaje: ' + (j.ahora || '') + ' • ' + eventoES(j.evento);
    metaEl.textContent = 'Estado actualizado automáticamente.';
    if (topUltimo) topUltimo.textContent = 'Último: ' + (j.ahora || '');

    showResult('ok', `<b>${j.msg}</b><br>
      <b>Hora:</b> ${j.ahora}<br>
      <b>Ajuste aplicado:</b> ${Number(j.rebaja_horas||0).toFixed(2)} horas<br>
      <small>${j.rebaja_desc || ''}</small>
    `);

    ok('Listo. Puedes volver a escanear cuando corresponda.');
    try{ if (navigator.vibrate) navigator.vibrate(120); }catch(_){}

    if ((j.estado === 'WORKING') && esEventoEntrada(j.evento || '')) {
      setTimeout(()=>{ window.location.href = 'dashboard.php'; }, 700);
      return;
    }

  }catch(e){
    showResult('err', `<b>Error:</b> ${e.message}`);
    bad('No se pudo registrar. Escanea el QR actual.');
  }
}

async function scanLoop(){
  if (!scanning || !detector || HARD_LOCK) return;
  try{
    const barcodes = await detector.detect(video);
    if (barcodes && barcodes.length){
      const token = barcodes[0].rawValue || '';
      if (token) await marcarToken(token);
    }
  }catch(_){}
  requestAnimationFrame(scanLoop);
}

// ===================== INIT =====================
async function cargarEstadoActual(){
  try{
    const r = await fetch('jornada_api.php?action=estado_actual', {cache:'no-store'});
    const j = await r.json();
    if (!j || !j.ok) throw new Error(j?.msg || 'No se pudo obtener estado');

    // Estado visual
    const est = (j.estado || 'SIN_MARCA');
    setBadge(est);
    aplicarBloqueoPanel(est);

    if (topEstado) topEstado.textContent = 'Estado: ' + estadoES(est);
    if (sideEstado) sideEstado.textContent = estadoES(est);

    if (j.fecha_hora) {
      lastEl.textContent = 'Último marcaje: ' + (j.fecha_hora) + ' • ' + eventoES(j.evento);
      metaEl.textContent = 'Hoy: ' + (j.hoy || '') + ' • Estado actualizado.';
      if (topUltimo) topUltimo.textContent = 'Último: ' + j.fecha_hora;
    } else {
      lastEl.textContent = 'Último marcaje: (aún no has marcado hoy)';
      metaEl.textContent = 'Hoy: ' + (j.hoy || '') + ' • Sin marcajes.';
      if (topUltimo) topUltimo.textContent = 'Último: --';
    }

    // 🔒 Si viene lock por overtime (ayer/hoy): BLOQUEO DURO
    if (j.overtime_lock) {
      aplicarHardLock(j.overtime_lock_msg || '🔒 BLOQUEADO por Hora Extra ACTIVA.');
      return;
    }

  }catch(e){
    metaEl.textContent = 'No se pudo cargar el estado: ' + e.message;
  }
}

btnStart.addEventListener('click', async ()=>{
  const id = localStorage.getItem('asist_cam_id') || null;
  await startCamera(id);
});
btnRetry.addEventListener('click', async ()=>{
  if (HARD_LOCK) { window.location.reload(); return; }
  const id = localStorage.getItem('asist_cam_id') || null;
  await startCamera(id);
});
btnStop.addEventListener('click', stopCamera);

camSel.addEventListener('change', async ()=>{
  if (HARD_LOCK) return;
  const id = camSel.value;
  if (!id) return;
  localStorage.setItem('asist_cam_id', id);
  stopCamera();
  await startCamera(id);
});

(async ()=>{
  if (!('BarcodeDetector' in window)) pill.textContent = 'Detector: NO soportado';
  else pill.textContent = 'Detector: OK';

  await cargarEstadoActual();

  // Solo enumerar cámaras si NO hay lock
  if (!HARD_LOCK) {
    try{ await enumerateCams(); }catch(_){}
  }
})();
</script>
</body>
</html>
