<?php
// rrhh_home/asistencia/registro_manual_api.php
// Registro manual = MISMA UX que QR (auto), pero sin escanear.
// - Un botón: decide "next_evento" según estado actual + hora oficial de salida
// - Bloqueo TOTAL si hay OT abierta de AYER (no deja registrar, ni cámara, ni nada)
// - Bloquea SALIDAS (OUT_*) si hay OT activa (regla original)
// - Guarda GPS + 1..N fotos (auditoría)
// - AUTOCIERRE + BLOQUEO:
//   * Si quedó marcaje abierto de día anterior y NO hay OT activa => autocierra (OUT_FINAL)
//   * Si es día siguiente y SÍ hay OT activa => bloqueo
// - NUEVO (pedido):
//   * Si evento = OUT_FINAL => cierra sesión (session_destroy) y responde logout=true
// PHP 8.1.33

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');

header('Content-Type: application/json; charset=utf-8');

function json_out(array $a, int $code=200): void {
  http_response_code($code);
  echo json_encode($a, JSON_UNESCAPED_UNICODE);
  exit;
}

if (!isset($_SESSION['gestor_id']) || (int)$_SESSION['gestor_id'] <= 0) {
  json_out(['ok'=>false,'msg'=>'Sesión expirada. Inicia sesión nuevamente.'], 401);
}

// CSRF por sesión (simple, efectivo)
$csrf = (string)($_POST['csrf'] ?? $_GET['csrf'] ?? '');
if ($csrf === '' || !isset($_SESSION['csrf_manual']) || !hash_equals((string)$_SESSION['csrf_manual'], $csrf)) {
  json_out(['ok'=>false,'msg'=>'CSRF inválido. Recarga la página e inténtalo de nuevo.'], 403);
}

require_once __DIR__ . '/../dbcon.php';
mysqli_set_charset($con, 'utf8mb4');

$gestor_id = (int)$_SESSION['gestor_id'];

// empleado en sesión (compatibilidad)
$idempleado = (int)($_SESSION['idempleado'] ?? 0);
if ($idempleado <= 0) $idempleado = (int)($_SESSION['gestor_empleado_id'] ?? 0);

if ($idempleado <= 0) {
  json_out(['ok'=>false,'msg'=>'Gestor sin idempleado asociado en sesión.'], 400);
}

/* ============================= Helpers ============================= */

function table_exists(mysqli $con, string $table): bool {
  $t = mysqli_real_escape_string($con, $table);
  $rs = mysqli_query($con, "SHOW TABLES LIKE '{$t}'");
  return ($rs && mysqli_num_rows($rs) > 0);
}

function get_columns(mysqli $con, string $table): array {
  $cols = [];
  $rs = mysqli_query($con, "SHOW COLUMNS FROM `$table`");
  while ($rs && ($r = mysqli_fetch_assoc($rs))) $cols[] = (string)$r['Field'];
  return $cols;
}

function ensure_dir(string $p): void {
  if (!is_dir($p)) @mkdir($p, 0775, true);
}

function parse_dataurl_image(string $dataUrl): array {
  $dataUrl = trim($dataUrl);
  if ($dataUrl === '') return ['ok'=>false,'msg'=>'Foto vacía'];

  if (!preg_match('#^data:(image/\w+);base64,#i', $dataUrl, $m)) {
    return ['ok'=>false,'msg'=>'Formato de foto inválido'];
  }
  $mime = strtolower($m[1]);
  $ext  = match($mime){
    'image/jpeg','image/jpg' => 'jpg',
    'image/png' => 'png',
    'image/webp' => 'webp',
    default => 'jpg'
  };

  $b64 = preg_replace('#^data:image/\w+;base64,#i', '', $dataUrl);
  $bin = base64_decode($b64, true);
  if ($bin === false || strlen($bin) < 1500) return ['ok'=>false,'msg'=>'Foto corrupta o muy pequeña'];
  if (strlen($bin) > 7_000_000) return ['ok'=>false,'msg'=>'Foto demasiado grande (reduce calidad).'];

  return ['ok'=>true,'mime'=>$mime,'ext'=>$ext,'bin'=>$bin];
}

function evento_to_estado(string $evento): string {
  return match(strtoupper($evento)){
    'IN','IN_INICIO','IN_REANUDA' => 'WORKING',
    'OUT_PAUSE','OUT_PENDING' => 'AWAY',
    'OUT_FINAL' => 'DONE',
    default => 'AWAY'
  };
}

function evento_label(string $evento): string {
  return match(strtoupper($evento)){
    'IN_INICIO' => 'Entrada (inicio)',
    'IN_REANUDA' => 'Entrada (reanudar)',
    'OUT_PAUSE' => 'Salida (pausa)',
    'OUT_FINAL' => 'Salida (final)',
    default => $evento
  };
}

/**
 * Devuelve último registro (incluye id + fecha_hora completa)
 */
function get_last_row(mysqli $con, int $idempleado): array {
  $sql = "SELECT id, estado, evento, DATE_FORMAT(fecha_hora,'%Y-%m-%d %H:%i:%s') AS fecha_hora
          FROM ausencias_marcajes
          WHERE idempleado=?
          ORDER BY id DESC
          LIMIT 1";
  $out = ['id'=>0,'estado'=>'SIN_MARCA','evento'=>'','fecha_hora'=>''];
  if ($st = mysqli_prepare($con, $sql)) {
    mysqli_stmt_bind_param($st, 'i', $idempleado);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    if ($rs && ($row = mysqli_fetch_assoc($rs))) {
      $out['id'] = (int)($row['id'] ?? 0);
      $out['estado'] = (string)($row['estado'] ?? 'SIN_MARCA');
      $out['evento'] = (string)($row['evento'] ?? '');
      $out['fecha_hora'] = (string)($row['fecha_hora'] ?? '');
    }
    mysqli_stmt_close($st);
  }
  return $out;
}

/* ===================== Horario oficial (para decidir OUT_FINAL) ===================== */

function get_today_schedule_end(mysqli $con): ?string {
  // 1=Lun ... 7=Dom
  if (!table_exists($con, 'work_schedule')) return null;

  $dow = (int)date('N');
  $q = "SELECT active, end_time FROM work_schedule WHERE day_of_week=? LIMIT 1";
  if ($st = mysqli_prepare($con, $q)) {
    mysqli_stmt_bind_param($st,'i',$dow);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$active,$end);
    if (mysqli_stmt_fetch($st)) {
      mysqli_stmt_close($st);
      if ((int)$active !== 1) return null;
      if (!$end) return null;
      // esperamos HH:MM:SS o HH:MM
      return (string)$end;
    }
    mysqli_stmt_close($st);
  }
  return null;
}

function minutes_between_hhmmss(string $a, string $b): int {
  // a/b "HH:MM" o "HH:MM:SS"
  $a = strlen($a) === 5 ? ($a.':00') : $a;
  $b = strlen($b) === 5 ? ($b.':00') : $b;
  $t1 = strtotime('2000-01-01 '.$a);
  $t2 = strtotime('2000-01-01 '.$b);
  if ($t1===false || $t2===false) return 0;
  return (int)floor(($t2-$t1)/60);
}

/**
 * Lógica QR “automática”:
 * - SIN MARCA o DONE => IN_INICIO
 * - AWAY => IN_REANUDA
 * - WORKING => si ya pasó la hora oficial => OUT_FINAL, si no => OUT_PAUSE
 */
function decidir_evento_auto(string $estado, string $ultimo_evento, ?string $schedule_end): string {
  $estado = strtoupper(trim($estado));
  $ultimo_evento = strtoupper(trim($ultimo_evento));

  if ($estado === '' || $estado === 'SIN_MARCA' || $estado === 'DONE') return 'IN_INICIO';
  if ($estado === 'AWAY') return 'IN_REANUDA';

  if ($estado === 'WORKING') {
    if ($schedule_end) {
      $now = date('H:i:s');
      $mins = minutes_between_hhmmss($schedule_end, $now);
      if ($mins >= 0) return 'OUT_FINAL';
    }
    return 'OUT_PAUSE';
  }

  return 'IN_INICIO';
}

/* ===================== Hora extra activa (REAL) ===================== */

function overtime_is_active(mysqli $con, int $idempleado): bool {
  if (table_exists($con, 'overtime_qr')) {
    $cols = get_columns($con, 'overtime_qr');

    $hasEmp = in_array('idempleado', $cols, true);
    $hasClosed = in_array('closed_at', $cols, true);
    $hasActual = in_array('actual_end', $cols, true);

    if ($hasEmp && ($hasClosed || $hasActual)) {
      $conds = [];
      if ($hasActual) $conds[] = "(actual_end IS NULL OR actual_end='')";
      if ($hasClosed) $conds[] = "(closed_at IS NULL OR closed_at='')";

      $whereActive = implode(' AND ', $conds);
      if ($whereActive === '') $whereActive = "1=0";

      $sql = "SELECT id
              FROM overtime_qr
              WHERE idempleado=?
                AND $whereActive
              ORDER BY id DESC
              LIMIT 1";
      if ($st = mysqli_prepare($con, $sql)) {
        mysqli_stmt_bind_param($st, 'i', $idempleado);
        mysqli_stmt_execute($st);
        $rs = mysqli_stmt_get_result($st);
        $ok = ($rs && mysqli_fetch_assoc($rs));
        mysqli_stmt_close($st);
        if ($ok) return true;
      }
    }
  }

  foreach (['asistencia_hora_extra_activa','horas_extra_activa','horas_extras_activas','hora_extra_activa'] as $t) {
    if (!table_exists($con, $t)) continue;
    $cols = get_columns($con, $t);

    $endedCol = null;
    foreach (['closed_at','actual_end','ended_at','fin_en','hora_fin','fecha_fin','end_time'] as $c) {
      if (in_array($c, $cols, true)) { $endedCol = $c; break; }
    }

    if ($endedCol) {
      $sql = "SELECT id FROM `$t` WHERE idempleado=? AND ($endedCol IS NULL OR $endedCol='') LIMIT 1";
      if ($st = mysqli_prepare($con, $sql)) {
        mysqli_stmt_bind_param($st, 'i', $idempleado);
        mysqli_stmt_execute($st);
        $rs = mysqli_stmt_get_result($st);
        $ok = ($rs && mysqli_fetch_assoc($rs));
        mysqli_stmt_close($st);
        if ($ok) return true;
      }
    } elseif (in_array('is_active', $cols, true)) {
      $sql = "SELECT id FROM `$t` WHERE idempleado=? AND is_active=1 LIMIT 1";
      if ($st = mysqli_prepare($con, $sql)) {
        mysqli_stmt_bind_param($st, 'i', $idempleado);
        mysqli_stmt_execute($st);
        $rs = mysqli_stmt_get_result($st);
        $ok = ($rs && mysqli_fetch_assoc($rs));
        mysqli_stmt_close($st);
        if ($ok) return true;
      }
    }
  }

  return false;
}

/**
 * Bloqueo TOTAL si OT pendiente de AYER
 */
function overtime_open_from_yesterday(mysqli $con, int $idempleado): array {
  if (!table_exists($con, 'overtime_qr')) return ['has'=>false];

  $cols = get_columns($con, 'overtime_qr');
  if (!in_array('idempleado', $cols, true) || !in_array('fecha', $cols, true)) return ['has'=>false];

  $y = date('Y-m-d', strtotime('-1 day'));

  $cond = [];
  if (in_array('planned_end', $cols, true)) $cond[] = "(planned_end IS NOT NULL AND planned_end<>'')";
  if (in_array('actual_end', $cols, true))  $cond[] = "(actual_end IS NULL OR actual_end='')";
  if (in_array('closed_at', $cols, true))   $cond[] = "(closed_at IS NULL OR closed_at='')";

  if (!$cond) return ['has'=>false];
  $where = implode(' AND ', $cond);

  $q = "SELECT fecha,
               ".(in_array('schedule_end',$cols,true) ? "schedule_end" : "NULL AS schedule_end").",
               ".(in_array('planned_end',$cols,true)  ? "planned_end"  : "NULL AS planned_end")."
        FROM overtime_qr
        WHERE idempleado=? AND fecha=? AND $where
        ORDER BY id DESC
        LIMIT 1";
  if ($st = mysqli_prepare($con,$q)) {
    mysqli_stmt_bind_param($st,'is',$idempleado,$y);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    $r = $rs ? mysqli_fetch_assoc($rs) : null;
    mysqli_stmt_close($st);
    if ($r) {
      return [
        'has'=>true,
        'fecha'=>$r['fecha'],
        'schedule_end'=> isset($r['schedule_end']) ? substr((string)$r['schedule_end'],0,5) : null,
        'planned_end' => isset($r['planned_end'])  ? substr((string)$r['planned_end'],0,5)  : null,
      ];
    }
  }
  return ['has'=>false];
}

/* ===================== Autocierre + bloqueo (día siguiente) ===================== */

function enforce_autoclose_and_block(mysqli $con, int $idempleado, int $gestor_id): array {
  $res = [
    'blocked'=>false,'blocked_msg'=>'',
    'autoclose_done'=>false,'autoclose_msg'=>'',
    'lock_ot_yesterday'=>['has'=>false]
  ];

  $lockY = overtime_open_from_yesterday($con, $idempleado);
  $res['lock_ot_yesterday'] = $lockY;
  if (!empty($lockY['has'])) {
    $res['blocked'] = true;
    $res['blocked_msg'] = "🔒 SISTEMA BLOQUEADO: hay Horas Extra pendientes de AYER ({$lockY['fecha']}). Debes corregirlo desde el Editor de Horas Extra.";
    return $res;
  }

  $last = get_last_row($con, $idempleado);
  $fh = trim((string)($last['fecha_hora'] ?? ''));
  $estado = strtoupper((string)($last['estado'] ?? 'SIN_MARCA'));

  if ($fh === '' || $estado === 'SIN_MARCA') return $res;

  $lastDate = substr($fh, 0, 10);
  $today = date('Y-m-d');

  if ($lastDate >= $today) return $res;

  if (!in_array($estado, ['WORKING','AWAY'], true)) return $res;

  if (overtime_is_active($con, $idempleado)) {
    $res['blocked'] = true;
    $res['blocked_msg'] = 'Existe una hora extra activa pendiente del día anterior. Debes cerrarla antes de registrar asistencia.';
    return $res;
  }

  $ip = (string)($_SERVER['REMOTE_ADDR'] ?? '');
  $ua = substr('AUTO-CLOSE', 0, 240);
  $token = 'AUTO-CLOSE-' . bin2hex(random_bytes(10)) . '-' . time();
  $evento = 'OUT_FINAL';
  $estadoNew = evento_to_estado($evento);

  $sql = "INSERT INTO ausencias_marcajes
          (idempleado, gestor_id, token, evento, estado, fecha_hora, ip, user_agent)
          VALUES (?,?,?,?,?,NOW(),?,?)";
  if ($st = mysqli_prepare($con, $sql)) {
    mysqli_stmt_bind_param($st, 'iisssss', $idempleado, $gestor_id, $token, $evento, $estadoNew, $ip, $ua);
    @mysqli_stmt_execute($st);
    mysqli_stmt_close($st);

    $res['autoclose_done'] = true;
    $res['autoclose_msg'] = 'Se realizó autocierre del día anterior (Salida final automática).';
  }

  return $res;
}

/* ============================= Actions ============================= */

$action = (string)($_POST['action'] ?? $_GET['action'] ?? '');

if ($action === 'estado_actual') {
  $pre = enforce_autoclose_and_block($con, $idempleado, $gestor_id);

  $last = get_last_row($con, $idempleado);
  $estado = (string)($last['estado'] ?? 'SIN_MARCA');
  $evento = (string)($last['evento'] ?? '');
  $fh = (string)($last['fecha_hora'] ?? '');

  $schedule_end = get_today_schedule_end($con);
  $next = decidir_evento_auto($estado, $evento, $schedule_end);

  $ot = overtime_is_active($con, $idempleado);

  json_out([
    'ok'=>true,
    'estado'=>$estado,
    'evento'=>$evento,
    'fecha_hora'=>$fh,
    'schedule_end'=>$schedule_end ? substr($schedule_end,0,5) : null,

    'next_evento'=>$next,
    'next_label'=>evento_label($next),

    'overtime_active'=>$ot,

    'blocked'=>$pre['blocked'],
    'blocked_msg'=>$pre['blocked_msg'],
    'autoclose_done'=>$pre['autoclose_done'],
    'autoclose_msg'=>$pre['autoclose_msg'],
    'lock_ot_yesterday'=>$pre['lock_ot_yesterday'],
  ]);
}

if ($action !== 'registrar_auto') {
  json_out(['ok'=>false,'msg'=>'Acción inválida'], 400);
}

// PRECHECK: si está bloqueado, no permitimos registrar (bloqueo TOTAL)
$pre = enforce_autoclose_and_block($con, $idempleado, $gestor_id);
if (!empty($pre['blocked'])) {
  json_out([
    'ok'=>false,
    'msg'=>$pre['blocked_msg'] ?: 'Bloqueado.',
    'blocked'=>true,
    'lock_ot_yesterday'=>$pre['lock_ot_yesterday']
  ], 409);
}

// GPS
$lat = trim((string)($_POST['lat'] ?? ''));
$lng = trim((string)($_POST['lng'] ?? ''));
$acc = trim((string)($_POST['acc'] ?? ''));

if ($lat==='' || $lng==='') {
  json_out(['ok'=>false,'msg'=>'No se recibió GPS. Debes permitir ubicación.'], 400);
}

// Fotos (dataURLs)
$photos_json = (string)($_POST['photos_json'] ?? '');
$photos = [];
if ($photos_json !== '') {
  $tmp = json_decode($photos_json, true);
  if (is_array($tmp)) $photos = $tmp;
}
if (count($photos) < 1) {
  json_out(['ok'=>false,'msg'=>'No se recibió foto.'], 400);
}
if (count($photos) > 4) $photos = array_slice($photos, 0, 4);

// Estado actual y evento auto
$lastRow = get_last_row($con, $idempleado);
$last_estado = (string)($lastRow['estado'] ?? 'SIN_MARCA');
$last_evento = (string)($lastRow['evento'] ?? '');
$last_fh     = (string)($lastRow['fecha_hora'] ?? '');

$schedule_end = get_today_schedule_end($con);
if (!$schedule_end) {
  json_out(['ok'=>false,'msg'=>'No hay horario activo hoy (work_schedule) o falta end_time.'], 400);
}

$evento = decidir_evento_auto($last_estado, $last_evento, $schedule_end);

// Bloqueo de SALIDAS si hay hora extra activa (regla original)
if (str_starts_with($evento, 'OUT_') && overtime_is_active($con, $idempleado)) {
  json_out([
    'ok'=>false,
    'msg'=>'No puedes registrar SALIDA porque hay una hora extra activa. Finaliza/cierra la hora extra primero.',
    'overtime_active'=>true,
    'next_evento'=>$evento
  ], 409);
}

$estado = evento_to_estado($evento);

// Auditoría
$ip = (string)($_SERVER['REMOTE_ADDR'] ?? '');
$ua = substr((string)($_SERVER['HTTP_USER_AGENT'] ?? ''), 0, 240);
$token = 'MANUAL-AUTO-' . bin2hex(random_bytes(10)) . '-' . time();

// Guardar fotos
$ym = date('Ym');
$dirAbs = __DIR__ . '/uploads/manual/' . $ym;
ensure_dir($dirAbs);

$saved = [];
foreach ($photos as $i => $dataUrl) {
  $img = parse_dataurl_image((string)$dataUrl);
  if (!$img['ok']) {
    json_out(['ok'=>false,'msg'=>"Foto ".($i+1).": ".$img['msg']], 400);
  }
  $fname = 'emp'.$idempleado.'_'.date('Ymd_His').'_cam'.($i+1).'_'.bin2hex(random_bytes(3)).'.'.$img['ext'];
  $abs = $dirAbs . '/' . $fname;
  if (@file_put_contents($abs, $img['bin']) === false) {
    json_out(['ok'=>false,'msg'=>'No se pudo guardar la foto (permisos carpeta uploads/manual).'], 500);
  }
  $saved[] = 'uploads/manual/'.$ym.'/'.$fname;
}

// Insert marcaje
$sql = "INSERT INTO ausencias_marcajes
        (idempleado, gestor_id, token, evento, estado, fecha_hora, ip, user_agent)
        VALUES (?,?,?,?,?,NOW(),?,?)";
$st = mysqli_prepare($con, $sql);
if (!$st) json_out(['ok'=>false,'msg'=>'DB prepare failed: '.mysqli_error($con)], 500);

mysqli_stmt_bind_param($st, 'iisssss', $idempleado, $gestor_id, $token, $evento, $estado, $ip, $ua);

if (!mysqli_stmt_execute($st)) {
  $err = mysqli_error($con);
  mysqli_stmt_close($st);
  json_out(['ok'=>false,'msg'=>'DB insert failed: '.$err], 500);
}
$marcaje_id = (int)mysqli_insert_id($con);
mysqli_stmt_close($st);

// Evidencia en BD (si existe tabla)
if (table_exists($con, 'ausencias_marcajes_evid')) {
  $cols = get_columns($con, 'ausencias_marcajes_evid');

  $pathCol = null;
  foreach (['photo_path','ruta','path','file_path','foto'] as $c) {
    if (in_array($c, $cols, true)) { $pathCol = $c; break; }
  }

  foreach ($saved as $relPath) {
    $data = [];
    $mapBase = [
      'idempleado' => $idempleado,
      'gestor_id' => $gestor_id,
      'marcaje_id' => $marcaje_id,
      'idmarcaje' => $marcaje_id,
      'ausencias_marcajes_id' => $marcaje_id,
      'lat' => $lat,
      'lng' => $lng,
      'acc' => $acc,
      'accuracy' => $acc,
      'creado_en' => date('Y-m-d H:i:s'),
      'created_at' => date('Y-m-d H:i:s'),
      'ip' => $ip,
      'user_agent' => $ua,
    ];

    foreach ($mapBase as $k=>$v) if (in_array($k, $cols, true)) $data[$k] = $v;
    if ($pathCol) $data[$pathCol] = $relPath;

    if (empty($data)) continue;

    $fields = array_keys($data);
    $place  = implode(',', array_fill(0, count($fields), '?'));

    $types = '';
    $vals  = [];
    foreach ($fields as $f) {
      $v = $data[$f];
      if (is_int($v)) { $types .= 'i'; $vals[] = $v; }
      else { $types .= 's'; $vals[] = (string)$v; }
    }

    $sqlE = "INSERT INTO ausencias_marcajes_evid (`" . implode('`,`',$fields) . "`) VALUES ($place)";
    if ($stE = mysqli_prepare($con, $sqlE)) {
      mysqli_stmt_bind_param($stE, $types, ...$vals);
      @mysqli_stmt_execute($stE);
      mysqli_stmt_close($stE);
    }
  }
}

/* ===================== NUEVO: cerrar sesión si OUT_FINAL ===================== */
$doLogout = (strtoupper($evento) === 'OUT_FINAL');

if ($doLogout) {
  // Limpieza fuerte de sesión
  $_SESSION = [];

  // matar cookie de sesión
  $cookieParams = session_get_cookie_params();
  setcookie(session_name(), '', time() - 42000,
    $cookieParams['path'] ?? '/',
    $cookieParams['domain'] ?? '',
    (bool)($cookieParams['secure'] ?? false),
    (bool)($cookieParams['httponly'] ?? true)
  );

  @session_destroy();
}

json_out([
  'ok' => true,
  'msg' => 'Registro guardado (GPS + Foto).',
  'ahora' => date('Y-m-d H:i:s'),
  'evento' => $evento,
  'estado' => $estado,
  'schedule_end' => substr((string)$schedule_end,0,5),
  'saved_photos' => count($saved),
  'gps' => ['lat'=>$lat,'lng'=>$lng,'acc'=>$acc],
  'marcaje_id' => $marcaje_id,
  'prev' => ['estado'=>$last_estado,'evento'=>$last_evento,'fh'=>$last_fh],

  // NUEVO: flags para el front
  'logout' => $doLogout,
  'logout_redirect' => 'login.php?bye=1'
]);
