<?php
// rrhh_home/asistencia/registro_manual.php
// Registro manual (sin QR) = igual que QR en UX: un botón, el sistema decide.
// - Captura GPS + foto(s) y guarda en BD por API
// - NUEVO (pedido):
//   * Si OUT_PAUSE: se queda aquí y muestra "Estás en pausa" + botón "Reanudar"
//   * Si OUT_FINAL: cierra sesión (API) y redirige a login
//   * Si IN_INICIO: redirige a dashboard con mensaje "Ingreso manual exitoso."
// PHP 8.1.33

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');

if (!isset($_SESSION['gestor_id']) || (int)$_SESSION['gestor_id'] <= 0) {
  header('Location: login.php'); exit;
}

$gestorNombre = (string)($_SESSION['gestor_nombre'] ?? 'Gestor');

if (empty($_SESSION['csrf_manual'])) {
  $_SESSION['csrf_manual'] = bin2hex(random_bytes(16));
}
$csrf = (string)$_SESSION['csrf_manual'];

// Forzar HTTPS (cámara requiere HTTPS)
$isHttps = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
        || (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] === '443');

$host = $_SERVER['HTTP_HOST'] ?? '';
$uri  = $_SERVER['REQUEST_URI'] ?? '/asistencia/registro_manual.php';

if (!$isHttps && stripos($host, 'hr.jportales.com') !== false) {
  header('Location: https://' . $host . $uri, true, 301);
  exit;
}

function h(string $s): string { return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <meta http-equiv="X-UA-Compatible" content="IE=edge">
  <title>Registro manual | GPS + Foto</title>
  <meta content="width=device-width, initial-scale=1, maximum-scale=1, user-scalable=no" name="viewport">

  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/twitter-bootstrap/3.4.1/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/4.7.0/css/font-awesome.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/2.4.18/css/AdminLTE.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/2.4.18/css/skins/skin-blue.min.css">

  <style>
    .user-image-circle{ width:28px;height:28px;border-radius:50%;object-fit:cover;border:2px solid rgba(255,255,255,.7); }
    .badge-pill{ display:inline-block;padding:6px 10px;border-radius:999px;background:#f3f4f6;border:1px solid #e5e7eb;font-weight:800; }
    .panel-card{ background:#fff;border:1px solid #e5e7eb;border-radius:12px; padding:16px; box-shadow:0 6px 18px rgba(0,0,0,.06); }
    .btnSuper{ width:100%; border-radius:12px; padding:16px; font-weight:900; font-size:18px; }
    #preview { width:100%; max-height:380px; background:#000; border-radius:10px; object-fit:cover; }
    .helpBox{ display:none; border-radius:12px; border:1px solid #fecaca; background:#fee2e2; color:#991b1b; padding:12px; font-weight:800; margin-top:10px; }
    .okBox{ display:none; border-radius:12px; border:1px solid #bbf7d0; background:#dcfce7; color:#065f46; padding:12px; font-weight:800; margin-top:10px; }
    .mini{ font-size:12px; color:#6b7280; margin-top:6px; }
    .soft{ color:#6b7280; }
    .menu-disabled > a{ opacity:.45; pointer-events:none; cursor:not-allowed; }
    .menu-disabled a{ pointer-events:none; }
    .pill-yellow{ background:#fef9c3; border-color:#fde68a; color:#854d0e; }
    .pill-red{ background:#fee2e2; border-color:#fecaca; color:#991b1b; }
    .pauseBox{ display:none; border-radius:12px; border:1px solid #bfdbfe; background:#dbeafe; color:#1e3a8a; padding:12px; font-weight:900; margin-top:10px; }
    .btnPause{ margin-top:10px; border-radius:12px; padding:12px; font-weight:900; width:100%; }
  </style>
</head>

<body class="hold-transition skin-blue sidebar-mini">
<div class="wrapper">

  <header class="main-header">
    <a href="dashboard.php" class="logo">
      <span class="logo-mini"><b>G</b></span>
      <span class="logo-lg"><b>Gestores</b> Asistencia</span>
    </a>
    <nav class="navbar navbar-static-top" role="navigation">
      <a href="#" class="sidebar-toggle" data-toggle="push-menu" role="button"><span class="sr-only">Toggle navigation</span></a>

      <div class="navbar-custom-menu">
        <ul class="nav navbar-nav">
          <li class="dropdown user user-menu">
            <a href="#" class="dropdown-toggle" data-toggle="dropdown">
              <img src="https://via.placeholder.com/80x80.png?text=U" class="user-image user-image-circle" alt="User">
              <span class="hidden-xs"><?= h($gestorNombre) ?></span>
            </a>
            <ul class="dropdown-menu">
              <li class="user-header">
                <img src="https://via.placeholder.com/120x120.png?text=U" class="img-circle" alt="User">
                <p style="margin-top:10px;">
                  <?= h($gestorNombre) ?><br>
                  <small>Estado: <b id="topEstado">--</b></small><br>
                  <small id="topUltimo">Último: --</small>
                </p>
              </li>
              <li class="user-footer">
                <div class="pull-left"><a href="dashboard.php" class="btn btn-default btn-flat">Dashboard</a></div>
                <div class="pull-right"><a href="logout.php" class="btn btn-default btn-flat">Salir</a></div>
              </li>
            </ul>
          </li>

        </ul>
      </div>
    </nav>
  </header>

  <aside class="main-sidebar">
    <section class="sidebar">
      <div class="user-panel">
        <div class="pull-left image"><img src="https://via.placeholder.com/120x120.png?text=U" class="img-circle" alt="User"></div>
        <div class="pull-left info">
          <p style="margin-bottom:2px;"><?= h($gestorNombre) ?></p>
          <a href="#"><i class="fa fa-circle text-success"></i> <span id="sideEstado">--</span></a>
        </div>
      </div>

      <ul class="sidebar-menu" data-widget="tree">
        <li class="header">MENÚ</li>

        <li id="menuDash" class="menu-disabled">
          <a href="dashboard.php"><i class="fa fa-dashboard"></i> <span>Dashboard</span></a>
        </li>

        <li class="treeview active">
          <a href="#"><i class="fa fa-qrcode"></i> <span>Scaneo asistencia</span>
            <span class="pull-right-container"><i class="fa fa-angle-left pull-right"></i></span>
          </a>
          <ul class="treeview-menu">
            <li><a href="scan.php"><i class="fa fa-camera"></i> Escanear</a></li>
            <li class="active"><a href="registro_manual.php"><i class="fa fa-map-marker"></i> Registro manual</a></li>
            <li id="menuHist" class="menu-disabled"><a href="historial.php"><i class="fa fa-list"></i> Historial</a></li>
          </ul>
        </li>

        <li class="treeview">
          <a href="#"><i class="fa fa-wrench"></i> <span>Herramientas</span>
            <span class="pull-right-container"><i class="fa fa-angle-left pull-right"></i></span>
          </a>
          <ul class="treeview-menu">
            <li><a href="convertidor_pdf.php"><i class="fa fa-file-pdf-o"></i> Convertidor PDF</a></li>
          </ul>
        </li>

        <li><a href="recordatorios.php"><i class="fa fa-bell"></i> <span>Recordatorios</span></a></li>
        <li><a href="chat.php"><i class="fa fa-comments"></i> <span>Chat</span></a></li>
        <li><a href="logout.php"><i class="fa fa-sign-out"></i> <span>Salir</span></a></li>
      </ul>
    </section>
  </aside>

  <div class="content-wrapper">
    <section class="content-header">
      <h1><i class="fa fa-map-marker"></i> Registro manual <small>GPS + Foto (igual que QR)</small></h1>
    </section>

    <section class="content">
      <div class="panel-card">

        <div style="display:flex;gap:8px;flex-wrap:wrap;margin-bottom:10px;">
          <span class="badge-pill">Estado: <b id="pillEstado">--</b></span>
          <span class="badge-pill">Último: <b id="pillUltimo">--</b></span>
          <span class="badge-pill">Evento: <b id="pillEvento">--</b></span>
          <span class="badge-pill">Siguiente: <b id="pillNext">--</b></span>
          <span class="badge-pill pill-yellow" id="pillOT" style="display:none;"></span>
          <span class="badge-pill pill-red" id="pillBlock" style="display:none;"></span>
        </div>

        <div class="alert alert-info" style="border-radius:12px;">
          Presiona <b>Registrar</b>. El sistema decide si es <b>Entrada</b>, <b>Reanudar</b>, <b>Pausa</b> o <b>Salida Final</b>.
          Se guardan <b>GPS + Foto(s)</b> para auditoría.
        </div>

        <div class="row">
          <div class="col-md-7">

            <button id="btnGo" class="btn btn-success btnSuper" disabled>
              <i class="fa fa-check"></i> Registrar
            </button>

            <button id="btnRetryPerms" class="btn btn-default btnSuper" style="display:none;margin-top:10px;">
              <i class="fa fa-unlock"></i> Reintentar permisos (GPS + Cámara)
            </button>

            <!-- NUEVO: Caja “en pausa” -->
            <div id="pauseBox" class="pauseBox"></div>
            <button id="btnReanudar" class="btn btn-primary btnPause" style="display:none;">
              <i class="fa fa-play"></i> ¿Deseas reanudar?
            </button>

            <div id="errBox" class="helpBox"></div>
            <div id="okBox" class="okBox"></div>

            <hr>

            <div class="soft"><b>GPS:</b> <span id="gpsTxt">--</span></div>
            <div class="soft"><b>Fotos capturadas:</b> <span id="photosTxt">0</span></div>

            <div class="mini">
              Si bloqueaste permisos: candado → Cámara/Ubicación → Permitir → recargar.
            </div>

          </div>

          <div class="col-md-5">
            <div class="soft" style="margin-bottom:6px;"><b>Vista previa (última foto):</b></div>
            <video id="video" playsinline autoplay muted style="display:none;"></video>
            <img id="preview" alt="preview" src="data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='800' height='450'%3E%3Crect width='100%25' height='100%25' fill='%23000'/%3E%3Ctext x='50%25' y='50%25' fill='%23fff' font-size='22' font-family='Arial' text-anchor='middle'%3EEsperando%20c%C3%A1mara...%3C/text%3E%3C/svg%3E">
          </div>
        </div>

      </div>
    </section>
  </div>

  <footer class="main-footer" style="font-size:12px;">
    <div class="pull-right hidden-xs">Asistencia Gestores</div>
    <strong>© <?= date('Y') ?> JPortales</strong>
  </footer>

</div>

<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.7.1/jquery.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/twitter-bootstrap/3.4.1/js/bootstrap.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/2.4.18/js/adminlte.min.js"></script>

<script>
const CSRF = <?= json_encode($csrf) ?>;

const btnGo = document.getElementById('btnGo');
const btnRetryPerms = document.getElementById('btnRetryPerms');

const errBox  = document.getElementById('errBox');
const okBox   = document.getElementById('okBox');

const pauseBox = document.getElementById('pauseBox');
const btnReanudar = document.getElementById('btnReanudar');

const gpsTxt  = document.getElementById('gpsTxt');
const photosTxt = document.getElementById('photosTxt');

const pillEstado = document.getElementById('pillEstado');
const pillUltimo = document.getElementById('pillUltimo');
const pillEvento = document.getElementById('pillEvento');
const pillNext   = document.getElementById('pillNext');
const pillOT     = document.getElementById('pillOT');
const pillBlock  = document.getElementById('pillBlock');

const topEstado = document.getElementById('topEstado');
const topUltimo = document.getElementById('topUltimo');
const sideEstado = document.getElementById('sideEstado');

const menuDash = document.getElementById('menuDash');
const menuHist = document.getElementById('menuHist');

const video   = document.getElementById('video');
const preview = document.getElementById('preview');

let currentNextEvento = null;
let currentNextLabel  = null;
let currentEstado     = null;
let currentBlocked    = false;
let currentBlockedMsg = '';

function showErr(html){
  errBox.style.display = 'block';
  errBox.innerHTML = html;
  okBox.style.display = 'none';
  btnRetryPerms.style.display = 'inline-block';
  hidePauseUI();
}
function showOk(html){
  okBox.style.display = 'block';
  okBox.innerHTML = html;
  errBox.style.display = 'none';
  btnRetryPerms.style.display = 'none';
}
function clearBoxes(){
  errBox.style.display = 'none';
  okBox.style.display  = 'none';
}

function hidePauseUI(){
  pauseBox.style.display = 'none';
  pauseBox.innerHTML = '';
  btnReanudar.style.display = 'none';
}

function showPauseUI(extraMsg){
  pauseBox.style.display = 'block';
  pauseBox.innerHTML = `
    <div style="font-size:16px;">⏸ Estás en <b>PAUSA</b>.</div>
    <div style="margin-top:6px;font-weight:800;">${extraMsg || 'Cuando estés listo, presiona reanudar.'}</div>
  `;
  btnReanudar.style.display = 'block';
}

function estadoES(s){
  if (s === 'WORKING') return 'TRABAJANDO';
  if (s === 'AWAY') return 'AUSENTE / PAUSA';
  if (s === 'DONE') return 'FINALIZADO';
  if (s === 'SIN_MARCA') return 'SIN MARCA';
  return s || '--';
}

function aplicarMenu(estado){
  const working = (estado === 'WORKING');
  const allow = working && !currentBlocked;
  if (menuDash) menuDash.classList.toggle('menu-disabled', !allow);
  if (menuHist) menuHist.classList.toggle('menu-disabled', !allow);
}

function applyBlockedUI(){
  if (currentBlocked) {
    pillBlock.style.display = 'inline-block';
    pillBlock.textContent = '⛔ BLOQUEADO';
    btnGo.disabled = true;

    showErr(`
      <div><b>⛔ Bloqueado:</b> ${currentBlockedMsg || 'Existe un registro activo del día anterior.'}</div>
      <div style="margin-top:6px;">Debes cerrar/corregir el registro pendiente antes de continuar.</div>
    `);
  } else {
    pillBlock.style.display = 'none';
  }
}

async function refreshEstado(){
  try{
    clearBoxes();
    const u = 'registro_manual_api.php?action=estado_actual&csrf=' + encodeURIComponent(CSRF);
    const r = await fetch(u, {cache:'no-store'});
    const j = await r.json();
    if (!j || !j.ok) return;

    currentEstado = j.estado || 'SIN_MARCA';
    currentNextEvento = j.next_evento || null;
    currentNextLabel  = j.next_label || null;

    currentBlocked = !!j.blocked;
    currentBlockedMsg = j.blocked_msg || '';

    pillEstado.textContent = estadoES(currentEstado);
    pillUltimo.textContent = j.fecha_hora || '--';
    pillEvento.textContent = j.evento || '--';
    pillNext.textContent   = currentNextLabel || '--';

    if (topEstado) topEstado.textContent = estadoES(currentEstado);
    if (topUltimo) topUltimo.textContent = 'Último: ' + (j.fecha_hora || '--');
    if (sideEstado) sideEstado.textContent = estadoES(currentEstado);

    if (j.overtime_active) {
      pillOT.style.display = 'inline-block';
      pillOT.textContent = '⚠ Hora extra activa';
    } else {
      pillOT.style.display = 'none';
    }

    aplicarMenu(currentEstado);

    if (j.autoclose_done) {
      showOk(`✅ ${j.autoclose_msg || 'Se realizó autocierre del día anterior.'}`);
    }

    if (currentBlocked) {
      applyBlockedUI();
      return;
    }

    // Si el estado actual es AWAY, mostramos UI de pausa (sin redirigir)
    if (currentEstado === 'AWAY') {
      showPauseUI('Presiona el botón para reanudar.');
    } else {
      hidePauseUI();
    }

    btnGo.disabled = false;

  }catch(e){}
}

// GPS
function getGPS(){
  return new Promise((resolve, reject)=>{
    if (!navigator.geolocation) return reject(new Error('Este navegador no soporta GPS.'));
    navigator.geolocation.getCurrentPosition(
      (pos)=> resolve({
        lat: pos.coords.latitude,
        lng: pos.coords.longitude,
        acc: Math.round(pos.coords.accuracy || 0)
      }),
      (err)=>{
        let msg = 'No se pudo obtener GPS.';
        if (err && err.code === 1) msg = 'GPS denegado por el usuario.';
        if (err && err.code === 2) msg = 'GPS no disponible.';
        if (err && err.code === 3) msg = 'Tiempo de espera GPS.';
        reject(new Error(msg));
      },
      { enableHighAccuracy: true, timeout: 12000, maximumAge: 0 }
    );
  });
}

// Cámaras
async function getDevices(){
  if (!navigator.mediaDevices?.getUserMedia) throw new Error('Navegador sin soporte de cámara.');
  const tmp = await navigator.mediaDevices.getUserMedia({video:true, audio:false});
  tmp.getTracks().forEach(t=>t.stop());

  const devs = await navigator.mediaDevices.enumerateDevices();
  const cams = devs.filter(d => d.kind === 'videoinput');
  if (!cams.length) throw new Error('No se encontraron cámaras.');
  return cams;
}

async function captureFromDevice(deviceId){
  const stream = await navigator.mediaDevices.getUserMedia({
    video: { deviceId: { exact: deviceId } },
    audio: false
  });
  video.srcObject = stream;
  video.style.display = 'block';

  await new Promise(res=>setTimeout(res, 650));

  const w = video.videoWidth || 1280;
  const h = video.videoHeight || 720;

  const canvas = document.createElement('canvas');
  canvas.width = w;
  canvas.height = h;

  const ctx = canvas.getContext('2d');
  ctx.drawImage(video, 0, 0, w, h);

  stream.getTracks().forEach(t=>t.stop());
  video.srcObject = null;
  video.style.display = 'none';

  const dataUrl = canvas.toDataURL('image/jpeg', 0.88);
  preview.src = dataUrl;
  return dataUrl;
}

async function captureAllCams(){
  const cams = await getDevices();
  const photos = [];

  for (let i=0; i<cams.length; i++){
    try{
      const p = await captureFromDevice(cams[i].deviceId);
      if (p) photos.push(p);
      photosTxt.textContent = String(photos.length);
    }catch(e){
      console.warn('Cam fail', e);
    }
  }
  if (!photos.length) throw new Error('No se pudo capturar ninguna foto.');
  return photos;
}

async function postRegistroAuto(gps, photos){
  const fd = new FormData();
  fd.append('csrf', CSRF);
  fd.append('action', 'registrar_auto');
  fd.append('lat', gps.lat);
  fd.append('lng', gps.lng);
  fd.append('acc', gps.acc);
  fd.append('photos_json', JSON.stringify(photos));

  const r = await fetch('registro_manual_api.php', {method:'POST', body: fd});
  const j = await r.json();
  return j;
}

function confirmTextForNext(nextEvento, nextLabel){
  if (nextEvento === 'OUT_PAUSE') return `¿Registrar PAUSA ahora?\n\n(${nextLabel})`;
  if (nextEvento === 'IN_REANUDA') return `¿Deseas REANUDAR el trabajo?\n\n(${nextLabel})`;
  if (nextEvento === 'IN_INICIO') return `¿Registrar ENTRADA (INICIO)?\n\n(${nextLabel})`;
  if (nextEvento === 'OUT_FINAL') return `¿Registrar SALIDA FINAL y cerrar sesión?\n\n(${nextLabel})`;
  return `¿Confirmas registrar?\n\n(${nextLabel || nextEvento || ''})`;
}

async function runRegistro(){
  clearBoxes();

  if (currentBlocked) {
    applyBlockedUI();
    return;
  }

  btnGo.disabled = true;
  btnRetryPerms.disabled = true;
  btnReanudar.disabled = true;

  try{
    await refreshEstado();

    if (currentBlocked) { applyBlockedUI(); return; }
    if (!currentNextEvento) throw new Error('No se pudo determinar el siguiente evento.');

    const ok = confirm(confirmTextForNext(currentNextEvento, currentNextLabel));
    if (!ok) { btnGo.disabled = false; btnRetryPerms.disabled = false; btnReanudar.disabled=false; return; }

    const gps = await getGPS();
    gpsTxt.textContent = `lat=${gps.lat}  lng=${gps.lng}  acc=${gps.acc}m`;

    const photos = await captureAllCams();

    const j = await postRegistroAuto(gps, photos);
    if (!j || !j.ok) throw new Error(j?.msg || 'No se pudo registrar.');

    // 1) Si fue salida final, el API ya destruyó sesión => redirigimos a login
    if (j.logout) {
      showOk(`✅ ${j.msg}<br><b>Evento:</b> ${j.evento}<br><b>Salida final registrada.</b><br>Cerrando sesión...`);
      setTimeout(()=>{ window.location.href = j.logout_redirect || 'login.php?bye=1'; }, 900);
      return;
    }

    // 2) Si fue pausa: quedarse aquí, mostrar “en pausa” + botón reanudar
    if ((j.evento || '').toUpperCase() === 'OUT_PAUSE' || (j.estado || '').toUpperCase() === 'AWAY') {
      showOk(`✅ ${j.msg}<br><b>Evento:</b> ${j.evento}<br><b>Hora:</b> ${j.ahora}`);
      showPauseUI('¿Deseas reanudar? Presiona el botón.');
      await refreshEstado();
      return;
    }

    // 3) Si fue inicio de jornada: ir a dashboard con mensaje “Ingreso manual exitoso.”
    if ((j.evento || '').toUpperCase() === 'IN_INICIO') {
      showOk(`✅ ${j.msg}<br><b>Evento:</b> ${j.evento}<br><b>Hora:</b> ${j.ahora}<br>Redirigiendo al Dashboard...`);
      setTimeout(()=>{ window.location.href = 'dashboard.php?manual_ok=1'; }, 700);
      return;
    }

    // 4) Otros (ej. IN_REANUDA): puedes ir al dashboard también (sin romper tu regla)
    if ((j.estado || '').toUpperCase() === 'WORKING') {
      showOk(`✅ ${j.msg}<br><b>Evento:</b> ${j.evento}<br><b>Hora:</b> ${j.ahora}<br>Reanudación exitosa. Redirigiendo...`);
      setTimeout(()=>{ window.location.href = 'dashboard.php?manual_ok=2'; }, 700);
      return;
    }

    // fallback
    showOk(`✅ ${j.msg}<br><b>Evento:</b> ${j.evento}<br><b>Estado:</b> ${estadoES(j.estado)}<br><b>Hora:</b> ${j.ahora}`);
    await refreshEstado();

  }catch(e){
    const msg = (e && e.message) ? e.message : String(e || 'Error');

    showErr(`
      <div><b>❌ No se pudo registrar:</b> ${msg}</div>
      <hr style="margin:10px 0;">
      <div style="font-weight:900;">Permisos:</div>
      <ol style="margin:6px 0 0 16px;">
        <li>Clic en el <b>candado</b>.</li>
        <li><b>Cámara</b> y <b>Ubicación</b>: <b>Permitir</b>.</li>
        <li>Recarga e intenta de nuevo.</li>
      </ol>
    `);
  }finally{
    if (!currentBlocked) btnGo.disabled = false;
    btnRetryPerms.disabled = false;
    btnReanudar.disabled = false;
  }
}

btnGo.addEventListener('click', runRegistro);
btnRetryPerms.addEventListener('click', async ()=>{ await refreshEstado(); });

// Botón “¿Deseas reanudar?” => simplemente corre el mismo flujo (el next_evento será IN_REANUDA)
btnReanudar.addEventListener('click', async ()=>{
  await runRegistro();
});

(async ()=>{ await refreshEstado(); })();
</script>

</body>
</html>
