<?php
// rrhh_home/asistencia/jornada_api.php
// API Asistencia: QR Token + marcaje automático + breaks + cierre automático.
// + BLOQUEO total si hay Horas Extra ACTIVAS (hoy o desde AYER).
// PHP 8.1.33

declare(strict_types=1);
date_default_timezone_set('America/Costa_Rica');
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../dbcon.php';
mysqli_set_charset($con,'utf8mb4');

function respond(array $a){ echo json_encode($a, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }

// ===================== CONFIG =====================
const TOKEN_SECRET  = 'CAMBIA_ESTA_LLAVE_LARGA_32+CHARS_2026_RRHH_JPORTALES';
const TOKEN_SECONDS = 40;
const CLOSE_GRACE_MINUTES = 5;

// ===================== ROUTER =====================
$action = $_GET['action'] ?? $_POST['action'] ?? '';

if ($action === 'token') {
  $now = time();
  $windowStart = (int)(floor($now / TOKEN_SECONDS) * TOKEN_SECONDS);
  $expiresIn = ($windowStart + TOKEN_SECONDS) - $now;
  $token = make_token($windowStart);

  respond([
    'ok' => true,
    'token' => $token,
    'token_seconds' => TOKEN_SECONDS,
    'expires_in' => $expiresIn,
  ]);
}

session_start();
if (!isset($_SESSION['gestor_id']) || (int)$_SESSION['gestor_id'] <= 0) {
  respond(['ok'=>false,'msg'=>'Sesión expirada (gestor)']);
}
$gestorId = (int)$_SESSION['gestor_id'];
$idempleado = (int)($_SESSION['idempleado'] ?? ($_SESSION['gestor_empleado_id'] ?? 0));
if ($idempleado <= 0) respond(['ok'=>false,'msg'=>'Gestor sin empleado asociado']);

// ===================== HORAS EXTRA LOCK (hoy/ayer) =====================
// Devuelve: ['locked'=>bool,'date'=>'YYYY-MM-DD','msg'=>'...']
function overtime_lock_info(mysqli $con, int $idempleado): array {
  $today = date('Y-m-d');
  $yesterday = (new DateTime('yesterday'))->format('Y-m-d');

  // Prioridad: AYER primero (porque eso es lo que quieres bloquear al amanecer)
  if (overtime_is_open_for_date($con, $idempleado, $yesterday)) {
    return [
      'locked' => true,
      'date'   => $yesterday,
      'msg'    => "🔒 BLOQUEADO: Existe una Hora Extra ACTIVA desde AYER ({$yesterday}). Debes cerrarla primero para continuar."
    ];
  }

  // Luego hoy
  if (overtime_is_open_for_date($con, $idempleado, $today)) {
    return [
      'locked' => true,
      'date'   => $today,
      'msg'    => "🔒 BLOQUEADO: Existe una Hora Extra ACTIVA hoy ({$today}). Debes cerrarla primero para continuar."
    ];
  }

  return ['locked'=>false,'date'=>'','msg'=>''];
}

function overtime_is_open_for_date(mysqli $con, int $idempleado, string $dateYmd): bool {
  // Intento 1: campos típicos (fecha + closed_at/actual_end)
  $q1 = "SELECT 1
         FROM overtime_qr
         WHERE idempleado=? AND fecha=?
           AND (
             closed_at IS NULL
             OR actual_end IS NULL
           )
         LIMIT 1";
  $st = @mysqli_prepare($con, $q1);
  if ($st) {
    mysqli_stmt_bind_param($st,'is',$idempleado,$dateYmd);
    @mysqli_stmt_execute($st);
    $rs = @mysqli_stmt_get_result($st);
    $ok = ($rs && mysqli_fetch_row($rs));
    mysqli_stmt_close($st);
    return (bool)$ok;
  }

  // Intento 2: por si la tabla usa "status"
  $q2 = "SELECT 1
         FROM overtime_qr
         WHERE idempleado=? AND fecha=?
           AND (status='OPEN' OR status='ACTIVA' OR status='ACTIVE')
         LIMIT 1";
  $st2 = @mysqli_prepare($con, $q2);
  if ($st2) {
    mysqli_stmt_bind_param($st2,'is',$idempleado,$dateYmd);
    @mysqli_stmt_execute($st2);
    $rs2 = @mysqli_stmt_get_result($st2);
    $ok2 = ($rs2 && mysqli_fetch_row($rs2));
    mysqli_stmt_close($st2);
    return (bool)$ok2;
  }

  // Si falla prepare (tabla no existe / columnas distintas), no bloqueamos.
  return false;
}

// ===================== ESTADO ACTUAL =====================
if ($action === 'estado_actual') {
  $today = date('Y-m-d');
  $last = last_mark_by_date($con, $idempleado, $today);
  $sched = schedule_for_date_db($con, $today);

  $lock = overtime_lock_info($con, $idempleado);

  respond([
    'ok'=>true,
    'hoy'=>$today,
    'estado'=>$last['estado'] ?? 'SIN_MARCA',
    'evento'=>$last['evento'] ?? '',
    'fecha_hora'=>$last['fecha_hora'] ?? '',
    'horario'=>$sched['ok'] ? [
      'start'=>$sched['start'],
      'end'=>$sched['end'],
      'late_grace_minutes'=>$sched['late_grace_minutes'],
      'early_grace_minutes'=>$sched['early_grace_minutes'],
      'lunch_start'=>$sched['lunch_start'],
      'lunch_end'=>$sched['lunch_end'],
      'coffee1_start'=>$sched['coffee1_start'],
      'coffee1_end'=>$sched['coffee1_end'],
      'coffee2_start'=>$sched['coffee2_start'],
      'coffee2_end'=>$sched['coffee2_end'],
      'break_window_before_min'=>$sched['break_window_before_min'],
      'break_window_after_min'=>$sched['break_window_after_min'],
      'label'=>$sched['label'],
    ] : null,

    // 🔒 NUEVO
    'overtime_lock' => (bool)$lock['locked'],
    'overtime_lock_date' => (string)$lock['date'],
    'overtime_lock_msg'  => (string)$lock['msg'],
  ]);
}

// ===================== CIERRE PENDIENTE =====================
if ($action === 'cierre_pendiente') {
  $info = close_pending_if_needed($con, $idempleado);
  respond(['ok'=>true] + $info);
}

// ===================== HISTORIAL =====================
if ($action === 'my_marks') {
  $limit = (int)($_GET['limit'] ?? 120);
  if ($limit < 1) $limit = 120;
  if ($limit > 250) $limit = 250;

  $rows = [];
  $q = "SELECT id, evento, estado, DATE_FORMAT(fecha_hora,'%Y-%m-%d %H:%i:%s') fh
        FROM ausencias_marcajes
        WHERE idempleado=?
        ORDER BY fecha_hora DESC
        LIMIT ?";
  if ($st = mysqli_prepare($con, $q)) {
    mysqli_stmt_bind_param($st,'ii',$idempleado,$limit);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    while($r = mysqli_fetch_assoc($rs)) {
      $rows[] = [
        'id'=>(int)$r['id'],
        'evento'=>$r['evento'],
        'estado'=>$r['estado'],
        'fecha_hora'=>$r['fh'],
      ];
    }
    mysqli_stmt_close($st);
  }
  respond(['ok'=>true,'rows'=>$rows]);
}

// ===================== MARCAJE (QR) =====================
if ($action === 'auto_mark') {

  // 🔒 BLOQUEO DURO antes de hacer cualquier cosa
  $lock = overtime_lock_info($con, $idempleado);
  if (!empty($lock['locked'])) {
    respond([
      'ok'=>false,
      'code'=>'OVERTIME_LOCK',
      'locked'=>true,
      'overtime_date'=>$lock['date'],
      'msg'=>$lock['msg']
    ]);
  }

  $token = trim((string)($_POST['token'] ?? ''));
  if ($token === '') respond(['ok'=>false,'msg'=>'Token vacío']);
  if (!validate_token($token)) respond(['ok'=>false,'msg'=>'Token inválido o expirado. Escanea el QR actual.']);

  // 1) Cierre pendiente (solo si NO está lock)
  $closeInfo = close_pending_if_needed($con, $idempleado);

  $nowDT = new DateTime('now');
  $today = $nowDT->format('Y-m-d');

  $sched = schedule_for_date_db($con, $today);
  if (!$sched['ok']) respond(['ok'=>false,'msg'=>'Hoy no tiene horario laboral configurado o está inactivo.']);

  $last = last_mark_by_date($con, $idempleado, $today);

  if (!empty($last) && ($last['estado'] ?? '') === 'DONE') {
    respond(['ok'=>false,'msg'=>'Tu jornada ya fue FINALIZADA hoy. Si necesitas corregir, se gestiona en el editor.']);
  }

  // anti-duplicado
  $recent = already_used_recently($con, $idempleado, $token, 15);
  if ($recent) {
    respond([
      'ok'=>true,
      'msg'=>'Lectura duplicada evitada (ya se registró hace unos segundos).',
      'estado'=> $recent['estado'],
      'evento'=> $recent['evento'],
      'fecha_hora'=> $recent['fecha_hora'],
      'rebaja_horas'=> 0.0,
      'rebaja_desc'=> ''
    ]);
  }

  $ip = $_SERVER['REMOTE_ADDR'] ?? '';
  $ua = substr((string)($_SERVER['HTTP_USER_AGENT'] ?? ''), 0, 200);

  $evento = '';
  $estado = '';
  $rebajaHoras = 0.0;
  $rebajaDesc  = '';

  if (!$last) {
    $evento = 'IN_INICIO';
    $estado = 'WORKING';

    $pen = late_penalty_hours($nowDT, $sched);
    $rebajaHoras = $pen['hours'];
    $rebajaDesc  = $pen['desc'];

    insert_mark($con, $idempleado, $gestorId, $token, $evento, $estado, $ip, $ua);

    if ($rebajaHoras > 0) apply_day_entry_merge($con, $idempleado, $today, 'RETRASO', $rebajaHoras, $rebajaDesc);

    respond([
      'ok'=>true,
      'msg'=>'✅ Marcaje registrado: estás TRABAJANDO.',
      'estado'=>$estado, 'evento'=>$evento,
      'rebaja_horas'=>$rebajaHoras, 'rebaja_desc'=>$rebajaDesc,
      'ahora'=>date('Y-m-d H:i:s'),
      'cierre'=>$closeInfo
    ]);
  }

  if (($last['estado'] ?? '') === 'WORKING') {

    $breakOut = detect_break_out($nowDT, $sched);
    if ($breakOut) {
      $evento = $breakOut;
      $estado = 'AWAY';
      insert_mark($con, $idempleado, $gestorId, $token, $evento, $estado, $ip, $ua);

      respond([
        'ok'=>true,
        'msg'=>'⏸ Marcaje registrado: salida de descanso detectada.',
        'estado'=>$estado, 'evento'=>$evento,
        'rebaja_horas'=>0.0, 'rebaja_desc'=>'',
        'ahora'=>date('Y-m-d H:i:s'),
        'cierre'=>$closeInfo
      ]);
    }

    if (is_after_or_close_to_end($nowDT, $sched)) {
      $evento = 'OUT_FINAL';
      $estado = 'DONE';

      $pen = early_leave_penalty_hours($nowDT, $sched);
      $rebajaHoras = $pen['hours'];
      $rebajaDesc  = $pen['desc'];

      insert_mark($con, $idempleado, $gestorId, $token, $evento, $estado, $ip, $ua);

      if ($rebajaHoras > 0) apply_day_entry_merge($con, $idempleado, $today, 'AUSENCIA', $rebajaHoras, $rebajaDesc);

      respond([
        'ok'=>true,
        'msg'=>'🏁 Marcaje registrado: jornada FINALIZADA.',
        'estado'=>$estado, 'evento'=>$evento,
        'rebaja_horas'=>$rebajaHoras, 'rebaja_desc'=>$rebajaDesc,
        'ahora'=>date('Y-m-d H:i:s'),
        'cierre'=>$closeInfo
      ]);
    }

    $evento = 'OUT_PENDING';
    $estado = 'AWAY';
    insert_mark($con, $idempleado, $gestorId, $token, $evento, $estado, $ip, $ua);

    respond([
      'ok'=>true,
      'msg'=>'⏸ Marcaje registrado: estás AUSENTE (pausa).',
      'estado'=>$estado, 'evento'=>$evento,
      'rebaja_horas'=>0.0, 'rebaja_desc'=>'',
      'ahora'=>date('Y-m-d H:i:s'),
      'cierre'=>$closeInfo
    ]);
  }

  if (($last['estado'] ?? '') === 'AWAY') {

    if (is_past_end_with_grace($nowDT, $sched)) {
      $c = close_day_from_away($con, $idempleado, $today, $sched, 'Escaneo después del horario');
      respond([
        'ok'=>true,
        'msg'=>'⚠ Jornada cerrada: se registró AUSENCIA por el tiempo no marcado hasta la salida.',
        'estado'=>'DONE',
        'evento'=>'OUT_FINAL',
        'rebaja_horas'=>(float)($c['ausencia_horas'] ?? 0.0),
        'rebaja_desc'=>(string)($c['ausencia_desc'] ?? ''),
        'ahora'=>date('Y-m-d H:i:s'),
        'cierre'=>$c
      ]);
    }

    $ret = return_event_from_break((string)($last['evento'] ?? ''));
    if ($ret) {
      $evento = $ret;
      $estado = 'WORKING';
      insert_mark($con, $idempleado, $gestorId, $token, $evento, $estado, $ip, $ua);

      respond([
        'ok'=>true,
        'msg'=>'✅ Marcaje registrado: retorno de descanso.',
        'estado'=>$estado, 'evento'=>$evento,
        'rebaja_horas'=>0.0, 'rebaja_desc'=>'',
        'ahora'=>date('Y-m-d H:i:s'),
        'cierre'=>$closeInfo
      ]);
    }

    confirm_last_pending_as_pause($con, $idempleado, $today);

    $evento = 'IN_REANUDA';
    $estado = 'WORKING';
    insert_mark($con, $idempleado, $gestorId, $token, $evento, $estado, $ip, $ua);

    respond([
      'ok'=>true,
      'msg'=>'✅ Marcaje registrado: reanudaste y estás TRABAJANDO.',
      'estado'=>$estado, 'evento'=>$evento,
      'rebaja_horas'=>0.0, 'rebaja_desc'=>'',
      'ahora'=>date('Y-m-d H:i:s'),
      'cierre'=>$closeInfo
    ]);
  }

  respond(['ok'=>false,'msg'=>'Estado no reconocido.']);
}

respond(['ok'=>false,'msg'=>'Acción no válida']);

// ===================== TOKEN HELPERS =====================
function b64url(string $bin): string { return rtrim(strtr(base64_encode($bin), '+/', '-_'), '='); }
function make_token(int $windowStart): string {
  $payload = 'J1.' . $windowStart;
  $sigBin = hash_hmac('sha256', $payload, TOKEN_SECRET, true);
  $sig = b64url(substr($sigBin, 0, 16));
  return $payload . '.' . $sig;
}
function validate_token(string $token): bool {
  $p = explode('.', $token);
  if (count($p) !== 3) return false;
  if ($p[0] !== 'J1') return false;
  $windowStart = (int)$p[1];
  if ($windowStart <= 0) return false;
  if (($windowStart % TOKEN_SECONDS) !== 0) return false;

  $now = time();
  $current = (int)(floor($now / TOKEN_SECONDS) * TOKEN_SECONDS);
  $prev    = $current - TOKEN_SECONDS;

  if ($windowStart !== $current && $windowStart !== $prev) return false;

  $expected = make_token($windowStart);
  return hash_equals($expected, $token);
}

// ===================== DB: MARCAJES =====================
function insert_mark(mysqli $con, int $idempleado, int $gestorId, string $token, string $evento, string $estado, string $ip, string $ua): void {
  $sql = "INSERT INTO ausencias_marcajes (idempleado, gestor_id, token, evento, estado, fecha_hora, ip, user_agent)
          VALUES (?,?,?,?,?,NOW(),?,?)";
  $st = mysqli_prepare($con, $sql);
  if (!$st) respond(['ok'=>false,'msg'=>'Error BD (prepare marcaje): '.mysqli_error($con)]);
  mysqli_stmt_bind_param($st, 'iisssss', $idempleado, $gestorId, $token, $evento, $estado, $ip, $ua);
  if (!mysqli_stmt_execute($st)) {
    $e = mysqli_error($con);
    mysqli_stmt_close($st);
    respond(['ok'=>false,'msg'=>'Error BD (insert marcaje): '.$e]);
  }
  mysqli_stmt_close($st);
}

function already_used_recently(mysqli $con, int $idempleado, string $token, int $seconds): ?array {
  $q = "SELECT evento, estado, DATE_FORMAT(fecha_hora,'%Y-%m-%d %H:%i:%s') fh
        FROM ausencias_marcajes
        WHERE idempleado=? AND token=?
        ORDER BY id DESC LIMIT 1";
  if ($st = mysqli_prepare($con, $q)) {
    mysqli_stmt_bind_param($st,'is',$idempleado,$token);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    $r = $rs ? mysqli_fetch_assoc($rs) : null;
    mysqli_stmt_close($st);
    if ($r) {
      $ts = strtotime($r['fh']);
      if ($ts !== false && (time() - $ts) <= $seconds) {
        return ['evento'=>$r['evento'], 'estado'=>$r['estado'], 'fecha_hora'=>$r['fh']];
      }
    }
  }
  return null;
}

function last_mark_by_date(mysqli $con, int $idempleado, string $dateYmd): array {
  $q = "SELECT id, evento, estado, DATE_FORMAT(fecha_hora,'%Y-%m-%d %H:%i:%s') fh, TIME(fecha_hora) hh
        FROM ausencias_marcajes
        WHERE idempleado=? AND DATE(fecha_hora)=?
        ORDER BY id DESC
        LIMIT 1";
  $out=[];
  if ($st = mysqli_prepare($con, $q)) {
    mysqli_stmt_bind_param($st,'is',$idempleado,$dateYmd);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    $r = $rs ? mysqli_fetch_assoc($rs) : null;
    mysqli_stmt_close($st);
    if ($r) {
      $out=[
        'id'=>(int)$r['id'],
        'evento'=>$r['evento'],
        'estado'=>$r['estado'],
        'fecha_hora'=>$r['fh'],
        'hora'=>$r['hh'],
      ];
    }
  }
  return $out;
}

function confirm_last_pending_as_pause(mysqli $con, int $idempleado, string $dateYmd): void {
  $q = "SELECT id FROM ausencias_marcajes
        WHERE idempleado=? AND DATE(fecha_hora)=? AND evento='OUT_PENDING'
        ORDER BY id DESC LIMIT 1";
  $id=0;
  if($st=mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'is',$idempleado,$dateYmd);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$tmp);
    if(mysqli_stmt_fetch($st)) $id=(int)$tmp;
    mysqli_stmt_close($st);
  }
  if($id>0){
    @mysqli_query($con, "UPDATE ausencias_marcajes SET evento='OUT_PAUSE' WHERE id=".$id);
  }
}

// ===================== HORARIO (work_schedule) =====================
function schedule_for_date_db(mysqli $con, string $dateYmd): array {
  $dt = new DateTime($dateYmd);
  $dow = (int)$dt->format('N'); // 1..7

  $q = "SELECT
          start_time, end_time, late_grace_minutes, early_grace_minutes, active,
          lunch_start, lunch_end,
          coffee1_start, coffee1_end,
          coffee2_start, coffee2_end,
          break_window_before_min, break_window_after_min
        FROM work_schedule
        WHERE day_of_week=?
        LIMIT 1";
  if ($st = mysqli_prepare($con, $q)) {
    mysqli_stmt_bind_param($st,'i',$dow);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    $r = $rs ? mysqli_fetch_assoc($rs) : null;
    mysqli_stmt_close($st);

    if ($r) {
      $active = (int)($r['active'] ?? 0);
      $start  = (string)($r['start_time'] ?? '');
      $end    = (string)($r['end_time'] ?? '');
      $late   = (int)($r['late_grace_minutes'] ?? 15);
      $early  = (int)($r['early_grace_minutes'] ?? 5);

      if ($active !== 1 || $start === '' || $end === '') return ['ok'=>false];

      $label = dow_label($dow) . " {$start} - {$end}";
      return [
        'ok'=>true,
        'start'=>$start,
        'end'=>$end,
        'late_grace_minutes'=>max(0,$late),
        'early_grace_minutes'=>max(0,$early),

        'lunch_start'=>(string)($r['lunch_start'] ?? ''),
        'lunch_end'=>(string)($r['lunch_end'] ?? ''),
        'coffee1_start'=>(string)($r['coffee1_start'] ?? ''),
        'coffee1_end'=>(string)($r['coffee1_end'] ?? ''),
        'coffee2_start'=>(string)($r['coffee2_start'] ?? ''),
        'coffee2_end'=>(string)($r['coffee2_end'] ?? ''),

        'break_window_before_min'=>max(0,(int)($r['break_window_before_min'] ?? 10)),
        'break_window_after_min'=>max(0,(int)($r['break_window_after_min'] ?? 10)),

        'label'=>$label
      ];
    }
  }
  return ['ok'=>false];
}

function dow_label(int $n): string {
  return match($n){
    1=>'Lunes',2=>'Martes',3=>'Miércoles',4=>'Jueves',5=>'Viernes',6=>'Sábado',7=>'Domingo',
    default=>'Día'
  };
}

// ===================== BREAKS =====================
function detect_break_out(DateTime $now, array $sched): string {
  $date = $now->format('Y-m-d');
  $before = (int)$sched['break_window_before_min'];
  $after  = (int)$sched['break_window_after_min'];

  if (!empty($sched['lunch_start']) && !empty($sched['lunch_end'])) {
    if (in_window($now, new DateTime($date.' '.$sched['lunch_start']), $before, $after)) return 'OUT_LUNCH';
  }
  if (!empty($sched['coffee1_start']) && !empty($sched['coffee1_end'])) {
    if (in_window($now, new DateTime($date.' '.$sched['coffee1_start']), $before, $after)) return 'OUT_CAFE1';
  }
  if (!empty($sched['coffee2_start']) && !empty($sched['coffee2_end'])) {
    if (in_window($now, new DateTime($date.' '.$sched['coffee2_start']), $before, $after)) return 'OUT_CAFE2';
  }
  return '';
}

function in_window(DateTime $now, DateTime $center, int $beforeMin, int $afterMin): bool {
  $a = (clone $center)->modify('-'.$beforeMin.' minutes');
  $b = (clone $center)->modify('+'.$afterMin.' minutes');
  $ts = $now->getTimestamp();
  return ($ts >= $a->getTimestamp() && $ts <= $b->getTimestamp());
}

function return_event_from_break(string $lastEvento): string {
  return match($lastEvento){
    'OUT_LUNCH' => 'IN_LUNCH',
    'OUT_CAFE1' => 'IN_CAFE1',
    'OUT_CAFE2' => 'IN_CAFE2',
    default => ''
  };
}

// ===================== TIME RULES =====================
function is_after_or_close_to_end(DateTime $now, array $sched): bool {
  $date = $now->format('Y-m-d');
  $end  = new DateTime($date.' '.$sched['end']);
  $diffMin = (int)floor(($end->getTimestamp() - $now->getTimestamp()) / 60);
  return ($diffMin <= 15);
}

function is_past_end_with_grace(DateTime $now, array $sched): bool {
  $date = $now->format('Y-m-d');
  $end  = new DateTime($date.' '.$sched['end']);
  $end->modify('+'.CLOSE_GRACE_MINUTES.' minutes');
  return $now->getTimestamp() > $end->getTimestamp();
}

function round_up_half_hours(int $minutes): float {
  if ($minutes <= 0) return 0.0;
  $blocks = (int)ceil($minutes / 30);
  return $blocks * 0.5;
}

function late_penalty_hours(DateTime $now, array $sched): array {
  $date = $now->format('Y-m-d');
  $start = new DateTime($date.' '.$sched['start']);
  $minutesLate = (int)floor(($now->getTimestamp() - $start->getTimestamp()) / 60);
  $grace = (int)$sched['late_grace_minutes'];
  if ($minutesLate <= $grace) return ['hours'=>0.0,'desc'=>''];

  $penMin = $minutesLate - $grace;
  $hours = round_up_half_hours($penMin);
  $desc = 'Llegada tardía: '.$minutesLate.' min (gracia '.$grace.' min) → rebaja '.$hours.' horas';
  return ['hours'=>(float)$hours,'desc'=>$desc];
}

function early_leave_penalty_hours(DateTime $now, array $sched): array {
  $date = $now->format('Y-m-d');
  $end = new DateTime($date.' '.$sched['end']);
  $minutesEarly = (int)floor(($end->getTimestamp() - $now->getTimestamp()) / 60);
  $grace = (int)$sched['early_grace_minutes'];
  if ($minutesEarly <= $grace) return ['hours'=>0.0,'desc'=>''];

  $penMin = $minutesEarly - $grace;
  $hours = round_up_half_hours($penMin);
  $desc = 'Salida temprana: '.$minutesEarly.' min (gracia '.$grace.' min) → rebaja '.$hours.' horas';
  return ['hours'=>(float)$hours,'desc'=>$desc];
}

// ===================== AUTO CLOSE =====================
function close_pending_if_needed(mysqli $con, int $idempleado): array {
  $out = ['closed'=>false, 'days'=>[]];

  $today = date('Y-m-d');
  $yesterday = (new DateTime('yesterday'))->format('Y-m-d');

  $lastY = last_mark_by_date($con, $idempleado, $yesterday);
  if (!empty($lastY) && ($lastY['estado'] ?? '') === 'AWAY') {
    $schedY = schedule_for_date_db($con, $yesterday);
    if ($schedY['ok']) {
      $c = close_day_from_away($con, $idempleado, $yesterday, $schedY, 'Cierre automático (día siguiente)');
      $out['closed'] = true;
      $out['days'][] = ['date'=>$yesterday] + $c;
    }
  }

  $lastT = last_mark_by_date($con, $idempleado, $today);
  if (!empty($lastT) && ($lastT['estado'] ?? '') === 'AWAY') {
    $schedT = schedule_for_date_db($con, $today);
    if ($schedT['ok']) {
      $now = new DateTime('now');
      if (is_past_end_with_grace($now, $schedT)) {
        $c = close_day_from_away($con, $idempleado, $today, $schedT, 'Cierre automático (fin de jornada)');
        $out['closed'] = true;
        $out['days'][] = ['date'=>$today] + $c;
      }
    }
  }

  return $out;
}

function close_day_from_away(mysqli $con, int $idempleado, string $dateYmd, array $sched, string $reason): array {
  $q = "SELECT id, TIME(fecha_hora) hh
        FROM ausencias_marcajes
        WHERE idempleado=? AND DATE(fecha_hora)=?
          AND (evento='OUT_PENDING' OR evento='OUT_PAUSE' OR evento='OUT_LUNCH' OR evento='OUT_CAFE1' OR evento='OUT_CAFE2')
        ORDER BY id DESC LIMIT 1";
  $idOut = 0; $hh = '';
  if($st=mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'is',$idempleado,$dateYmd);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$id,$t);
    if(mysqli_stmt_fetch($st)){ $idOut=(int)$id; $hh=(string)$t; }
    mysqli_stmt_close($st);
  }
  if($idOut<=0 || $hh===''){
    return ['closed'=>false,'msg'=>'No había salida pendiente para cerrar.'];
  }

  @mysqli_query($con, "UPDATE ausencias_marcajes SET evento='OUT_FINAL', estado='DONE' WHERE id=".$idOut);

  $end = new DateTime($dateYmd.' '.$sched['end']);
  $outT = new DateTime($dateYmd.' '.$hh);

  $minutesMissing = (int)floor(($end->getTimestamp() - $outT->getTimestamp()) / 60);
  if ($minutesMissing < 0) $minutesMissing = 0;

  $hours = round_up_half_hours($minutesMissing);
  $desc = "Cierre: quedó AUSENTE y no reanudó. Faltante {$minutesMissing} min → AUSENCIA {$hours} h. ({$reason})";

  if ($hours > 0) apply_day_entry_merge($con, $idempleado, $dateYmd, 'AUSENCIA', (float)$hours, $desc);

  return [
    'closed'=>true,
    'ausencia_min'=>$minutesMissing,
    'ausencia_horas'=>$hours,
    'ausencia_desc'=>$desc
  ];
}

// ===================== ausencias_editor merge =====================
function get_pref_hours_dia(mysqli $con): float {
  $h = 8.0;
  $q="SELECT horas_dia FROM payroll_pref WHERE activo=1 ORDER BY id DESC LIMIT 1";
  if($rs=mysqli_query($con,$q)){
    if($r=mysqli_fetch_assoc($rs)) $h = (float)$r['horas_dia'];
    mysqli_free_result($rs);
  }
  if ($h <= 0) $h = 8.0;
  return $h;
}

function ensure_header(mysqli $con, int $idempleado, string $mes): int {
  $id=0;
  $q="SELECT id FROM ausencias_mes WHERE idempleado=? AND mes=? LIMIT 1";
  if($st=mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'is',$idempleado,$mes);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$tmp);
    if(mysqli_stmt_fetch($st)) $id=(int)$tmp;
    mysqli_stmt_close($st);
  }
  if ($id>0) return $id;

  $ins="INSERT INTO ausencias_mes (idempleado, mes, creado_por) VALUES (?,?,0)";
  if($st=mysqli_prepare($con,$ins)){
    mysqli_stmt_bind_param($st,'is',$idempleado,$mes);
    mysqli_stmt_execute($st);
    $id=(int)mysqli_insert_id($con);
    mysqli_stmt_close($st);
  }
  return $id;
}

function apply_day_entry_merge(mysqli $con, int $idempleado, string $fechaYmd, string $tipoNew, float $hoursNew, string $descNew): void {
  $mes = substr($fechaYmd,0,7);
  $id_am = ensure_header($con, $idempleado, $mes);

  $horasDiaMax = get_pref_hours_dia($con);

  $old = null;
  $q="SELECT id, tipo, horas, descripcion
      FROM ausencias_detalle
      WHERE id_ausencias_mes=? AND fecha=?
      LIMIT 1";
  if($st=mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'is',$id_am,$fechaYmd);
    mysqli_stmt_execute($st);
    $rs=mysqli_stmt_get_result($st);
    $old = $rs ? mysqli_fetch_assoc($rs) : null;
    mysqli_stmt_close($st);
  }

  $tipo = $tipoNew;
  $horas = $hoursNew;
  $desc = trim($descNew);

  if ($old) {
    $oldTipo = (string)$old['tipo'];
    $oldHoras = (float)$old['horas'];
    $oldDesc = (string)($old['descripcion'] ?? '');

    if ($oldDesc !== '') $desc = $oldDesc.' | '.$desc;

    if ($oldTipo === 'AUSENCIA') {
      $tipo = 'AUSENCIA';
      $horas = $oldHoras;
    } elseif ($oldTipo === 'RETRASO' && $tipoNew === 'RETRASO') {
      $tipo = 'RETRASO';
      $horas = $oldHoras + $hoursNew;
    } elseif ($oldTipo === 'RETRASO' && $tipoNew === 'AUSENCIA') {
      $tipo = 'AUSENCIA';
      $horas = $oldHoras + $hoursNew;
    }

    if ($horas > $horasDiaMax) $horas = $horasDiaMax;
    if (strlen($desc) > 255) $desc = substr($desc,0,255);

    $u="UPDATE ausencias_detalle
        SET tipo=?, horas=?, descripcion=?, updated_by=0, updated_at=NOW()
        WHERE id=?";
    if($st=mysqli_prepare($con,$u)){
      $idRow=(int)$old['id'];
      mysqli_stmt_bind_param($st,'sdsi',$tipo,$horas,$desc,$idRow);
      @mysqli_stmt_execute($st);
      mysqli_stmt_close($st);
    }
  } else {
    if ($horas > $horasDiaMax) $horas = $horasDiaMax;
    if (strlen($desc) > 255) $desc = substr($desc,0,255);

    $i="INSERT INTO ausencias_detalle (id_ausencias_mes, fecha, tipo, horas, descripcion, creado_por)
        VALUES (?,?,?,?,?,0)";
    if($st=mysqli_prepare($con,$i)){
      mysqli_stmt_bind_param($st,'issds',$id_am,$fechaYmd,$tipo,$horas,$desc);
      @mysqli_stmt_execute($st);
      mysqli_stmt_close($st);
    }
  }
}
