<?php
// rrhh_home/asistencia/inc/auth.php
declare(strict_types=1);

date_default_timezone_set('America/Costa_Rica');

function require_gestor_login(): array {
  if (session_status() === PHP_SESSION_NONE) session_start();

  if (empty($_SESSION['gestor_id']) || (int)$_SESSION['gestor_id'] <= 0) {
    header('Location: login.php'); exit;
  }

  $idempleado = (int)($_SESSION['gestor_empleado_id'] ?? 0);

  // Compatibilidad: algunos archivos usan $_SESSION['idempleado']
  if ($idempleado > 0 && empty($_SESSION['idempleado'])) {
    $_SESSION['idempleado'] = $idempleado;
  }

  return [
    'gestor_id'  => (int)$_SESSION['gestor_id'],
    'idempleado' => $idempleado,
    'nombre'     => (string)($_SESSION['gestor_nombre'] ?? 'Gestor')
  ];
}

/**
 * Conecta usando el $con global (dbcon.php).
 */
function asistencia_get_con(): ?mysqli {
  global $con;

  if (isset($con) && ($con instanceof mysqli)) {
    @mysqli_set_charset($con, 'utf8mb4');
    return $con;
  }

  // auth.php está en asistencia/inc → dbcon.php está en rrhh_home/
  require_once __DIR__ . '/../../dbcon.php';
  global $con;

  if (isset($con) && ($con instanceof mysqli)) {
    @mysqli_set_charset($con, 'utf8mb4');
    return $con;
  }

  return null;
}

/**
 * Obtiene la ÚLTIMA marca del empleado (sin filtrar por día).
 * Devuelve:
 * - fecha_ymd (YYYY-mm-dd)
 * - fecha_hora (YYYY-mm-dd HH:ii:ss)
 * - evento
 * - estado
 * - inferred_state: WORKING|AWAY|DONE|SIN_MARCA
 */
function asistencia_last_mark(mysqli $con, int $idempleado): array {
  $out = [
    'fecha_ymd' => '',
    'fecha_hora' => '',
    'evento' => '',
    'estado' => '',
    'inferred_state' => 'SIN_MARCA',
  ];

  $q = "SELECT evento, estado,
               DATE_FORMAT(fecha_hora,'%Y-%m-%d %H:%i:%s') AS fh,
               DATE_FORMAT(fecha_hora,'%Y-%m-%d') AS ymd
        FROM ausencias_marcajes
        WHERE idempleado=?
        ORDER BY id DESC
        LIMIT 1";

  if ($st = mysqli_prepare($con, $q)) {
    mysqli_stmt_bind_param($st, 'i', $idempleado);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    if ($rs && ($r = mysqli_fetch_assoc($rs))) {
      $out['evento'] = (string)($r['evento'] ?? '');
      $out['estado'] = (string)($r['estado'] ?? '');
      $out['fecha_hora'] = (string)($r['fh'] ?? '');
      $out['fecha_ymd']  = (string)($r['ymd'] ?? '');

      // Inferencia robusta (porque tu "estado" viene raro con OUT_FINAL)
      $out['inferred_state'] = asistencia_infer_state($out['evento'], $out['estado']);
    }
    mysqli_stmt_close($st);
  }

  return $out;
}

/**
 * Inferir el estado real a partir del evento y estado.
 * Regla:
 * - Si evento = OUT_FINAL => DONE
 * - Si evento empieza por OUT_ => AWAY (pendiente/pausa)
 * - Si evento empieza por IN_ => WORKING
 * - Si no hay evento => SIN_MARCA
 * - Si estado viene bien (WORKING/AWAY/DONE) lo usamos como fallback
 */
function asistencia_infer_state(string $evento, string $estado): string {
  $evento = strtoupper(trim($evento));
  $estado = strtoupper(trim($estado));

  if ($evento === '') {
    // fallback al estado si viene correcto
    if (in_array($estado, ['WORKING','AWAY','DONE'], true)) return $estado;
    return 'SIN_MARCA';
  }

  if ($evento === 'OUT_FINAL') return 'DONE';

  if (str_starts_with($evento, 'OUT_')) return 'AWAY';
  if (str_starts_with($evento, 'IN_'))  return 'WORKING';

  // fallback al estado si viene correcto
  if (in_array($estado, ['WORKING','AWAY','DONE'], true)) return $estado;

  return 'SIN_MARCA';
}

/**
 * REGLA FINAL (la que tú pediste):
 * - Si la ÚLTIMA fecha_hora es de AYER => BLOQUEAR TODO
 * - Si la ÚLTIMA fecha_hora es de HOY => SOLO permitir si está WORKING
 * - Cualquier otro caso => BLOQUEAR
 */
function require_working_or_scan(array $user, string $currentPage): void {
  if (session_status() === PHP_SESSION_NONE) session_start();

  // Páginas permitidas cuando está bloqueado
  $allowedWhenLocked = ['scan.php', 'logout.php', 'login.php'];
  if (in_array($currentPage, $allowedWhenLocked, true)) return;

  // Si tu scan.php usa este flag para bloquear por horas extra u otra regla, lo respetamos
  if (!empty($_SESSION['gestor_blocked'])) {
    header('Location: scan.php?locked=1&reason=blocked'); exit;
  }

  $idempleado = (int)($user['idempleado'] ?? 0);
  if ($idempleado <= 0) {
    header('Location: scan.php?locked=1&reason=no_employee'); exit;
  }

  $con = asistencia_get_con();
  if (!$con) {
    // Si no podemos validar, por seguridad bloqueamos
    header('Location: scan.php?locked=1&reason=db_error'); exit;
  }

  $today = date('Y-m-d');
  $last  = asistencia_last_mark($con, $idempleado);

  // Si no hay marcas -> bloqueado
  if ($last['fecha_ymd'] === '') {
    $_SESSION['gestor_estado'] = 'SIN_MARCA';
    $_SESSION['gestor_estado_date'] = $today;
    header('Location: scan.php?locked=1&reason=no_marks'); exit;
  }

  // ✅ 1) Si la última marca NO es de HOY => BLOQUEAR (ayer u otro día)
  if ($last['fecha_ymd'] !== $today) {
    $_SESSION['gestor_estado'] = $last['inferred_state'];
    $_SESSION['gestor_estado_date'] = $last['fecha_ymd']; // guardamos la real (ayer)
    header('Location: scan.php?locked=1&reason=wrong_day'); exit;
  }

  // ✅ 2) Si es HOY, debe estar WORKING
  $_SESSION['gestor_estado'] = $last['inferred_state'];
  $_SESSION['gestor_estado_date'] = $today;

  if ($last['inferred_state'] !== 'WORKING') {
    header('Location: scan.php?locked=1&reason=not_working'); exit;
  }

  // ✅ Si llegó aquí: HOY + WORKING => acceso total a TODO.
}
