<?php
// rrhh_home/asistencia/historial.php
// Historial con tabs: Asistencia / Horas Extra
// Requisito: solo accesible cuando el gestor esté TRABAJANDO (WORKING).
// Mejora: Mostrar GPS + Foto desde tabla ausencias_marcajes_evid (marcaje_id, lat,lng,acc,photo_path).
// ✅ UI: MISMO Bootstrap/AdminLTE + layout-top-nav (como scan_extra.php). SIN tocar consultas.
// PHP 8.1.33

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');

if (!isset($_SESSION['gestor_id']) || (int)$_SESSION['gestor_id'] <= 0) {
  header('Location: login.php'); exit;
}

require_once __DIR__ . '/../dbcon.php';
mysqli_set_charset($con, 'utf8mb4');

$gestorId     = (int)($_SESSION['gestor_id'] ?? 0);
$gestorNombre = (string)($_SESSION['gestor_nombre'] ?? 'Gestor');

// Compatibilidad: algunos guardan idempleado como idempleado o como gestor_empleado_id
$idempleado = (int)($_SESSION['idempleado'] ?? 0);
if ($idempleado <= 0) $idempleado = (int)($_SESSION['gestor_empleado_id'] ?? 0);
if ($idempleado <= 0) $idempleado = (int)($_SESSION['idempleado_planilla'] ?? 0);

if ($idempleado <= 0) {
  echo "Gestor sin empleado asociado"; exit;
}

// TAB: asistencia | extra
$tab = (string)($_GET['tab'] ?? 'asistencia');
$tab = ($tab === 'extra') ? 'extra' : 'asistencia';

// Traer estado actual (último marcaje)
$estadoActual = '';
$eventoActual = '';
$fhActual     = '';

$q0 = "SELECT estado, evento, DATE_FORMAT(fecha_hora,'%Y-%m-%d %H:%i:%s') fh
       FROM ausencias_marcajes
       WHERE idempleado=?
       ORDER BY id DESC
       LIMIT 1";
if ($st0 = mysqli_prepare($con, $q0)) {
  mysqli_stmt_bind_param($st0, 'i', $idempleado);
  mysqli_stmt_execute($st0);
  $rs0 = mysqli_stmt_get_result($st0);
  $r0  = $rs0 ? mysqli_fetch_assoc($rs0) : null;
  mysqli_stmt_close($st0);

  if ($r0) {
    $estadoActual = (string)($r0['estado'] ?? '');
    $eventoActual = (string)($r0['evento'] ?? '');
    $fhActual     = (string)($r0['fh'] ?? '');
  }
}

// Regla del panel: si no está WORKING, no puede ver historial
if ($estadoActual !== 'WORKING') {
  header('Location: scan.php?locked=1'); exit;
}

// Helpers
function h(string $s): string { return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }

function evento_es(string $e): string {
  return match($e){
    'IN' => 'ENTRADA',
    'IN_INICIO' => 'ENTRADA (INICIO)',
    'IN_REANUDA' => 'ENTRADA (REANUDACIÓN)',
    'OUT_PENDING' => 'SALIDA (PENDIENTE)',
    'OUT_PAUSE' => 'SALIDA (PAUSA)',
    'OUT_LUNCH' => 'SALIDA (ALMUERZO)',
    'OUT_CAFE1' => 'SALIDA (CAFÉ 1)',
    'OUT_CAFE2' => 'SALIDA (CAFÉ 2)',
    'IN_LUNCH' => 'ENTRADA (REGRESA ALMUERZO)',
    'IN_CAFE1' => 'ENTRADA (REGRESA CAFÉ 1)',
    'IN_CAFE2' => 'ENTRADA (REGRESA CAFÉ 2)',
    'OUT_FINAL' => 'SALIDA FINAL (JORNADA)',
    default => $e
  };
}

function estado_es(string $s): string {
  return match($s){
    'WORKING' => 'TRABAJANDO',
    'AWAY' => 'AUSENTE / PAUSA',
    'DONE' => 'FINALIZADO',
    default => $s
  };
}

function fmt_hhmm(?string $t): string {
  $t = trim((string)$t);
  if ($t === '') return '';
  return substr($t, 0, 5);
}

function build_maps_url(?string $lat, ?string $lng): string {
  $lat = trim((string)$lat);
  $lng = trim((string)$lng);
  if ($lat === '' || $lng === '') return '';
  $q = rawurlencode($lat . ',' . $lng);
  return "https://www.google.com/maps/search/?api=1&query={$q}";
}

function normalize_photo_url(string $path): string {
  $p = trim($path);
  if ($p === '') return '';
  if (preg_match('~^https?://~i', $p)) return $p;
  if ($p[0] === '/') return $p;

  if (stripos($p, 'asistencia/') === 0) return '/' . $p;
  if (stripos($p, 'uploads/') === 0) return '/asistencia/' . $p;

  return '/' . $p;
}

// =======================
// DATA SEGÚN TAB
// =======================
$rowsAsistencia = [];
$rowsExtra = [];

if ($tab === 'asistencia') {

  $q = "
    SELECT
      m.id,
      m.evento,
      m.estado,
      DATE_FORMAT(m.fecha_hora,'%Y-%m-%d %H:%i:%s') fh,
      e.lat,
      e.lng,
      e.acc,
      e.photo_path
    FROM ausencias_marcajes m
    LEFT JOIN ausencias_marcajes_evid e
      ON e.marcaje_id = m.id
    WHERE m.idempleado=?
    ORDER BY m.id DESC
    LIMIT 200
  ";

  if ($st = mysqli_prepare($con, $q)) {
    mysqli_stmt_bind_param($st,'i',$idempleado);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    while ($rs && ($r = mysqli_fetch_assoc($rs))) $rowsAsistencia[] = $r;
    mysqli_stmt_close($st);
  }

} else {

  $q = "SELECT
          fecha,
          schedule_end,
          planned_end,
          actual_end,
          overtime_minutes,
          factor,
          DATE_FORMAT(planned_at,'%Y-%m-%d %H:%i:%s') planned_at,
          DATE_FORMAT(closed_at,'%Y-%m-%d %H:%i:%s') closed_at
        FROM overtime_qr
        WHERE idempleado=?
        ORDER BY fecha DESC
        LIMIT 120";

  if ($st = mysqli_prepare($con, $q)) {
    mysqli_stmt_bind_param($st,'i',$idempleado);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    while ($rs && ($r = mysqli_fetch_assoc($rs))) $rowsExtra[] = $r;
    mysqli_stmt_close($st);
  }
}

// UI links (top-nav como scan_extra.php)
$backUrl = 'dashboard.php';
$tabLabel = ($tab === 'extra') ? 'Horas Extra' : 'Asistencia';
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <meta http-equiv="X-UA-Compatible" content="IE=edge">
  <title>Historial | <?= h($tabLabel) ?></title>
  <meta content="width=device-width, initial-scale=1, maximum-scale=1, user-scalable=no" name="viewport">

  <!-- MISMO stack que scan_extra.php (Bootstrap 3 + AdminLTE 2) -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/twitter-bootstrap/3.4.1/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/4.7.0/css/font-awesome.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/2.4.18/css/AdminLTE.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/2.4.18/css/skins/skin-blue.min.css">

  <style>
    .user-image-circle{
      width:28px;height:28px;border-radius:50%;object-fit:cover;border:2px solid rgba(255,255,255,.7);
    }
    .badge-pill{
      display:inline-block;padding:6px 10px;border-radius:999px;background:#f3f4f6;border:1px solid #e5e7eb;font-weight:800;
    }
    .table td{ vertical-align:middle!important; }
    .tabbar{ margin-bottom:12px; }
    .tabbar .btn{ font-weight:800; }
    .small-muted{ color:#6b7280; font-size:12px; }

    .thumb{
      width:52px;height:52px;border-radius:10px;object-fit:cover;
      border:1px solid #e5e7eb; box-shadow:0 2px 6px rgba(0,0,0,.08);
      background:#f3f4f6;
    }
    .gps-box{ font-size:12px; color:#374151; line-height:1.25; }
    .gps-box small{ color:#6b7280; display:block; margin-top:2px; }
    .nowrap{ white-space:nowrap; }

    /* Botón volver pro (igual vibe del scan_extra.php) */
    .btn-back-pro{
      border-radius:12px !important;
      font-weight:900 !important;
      padding:10px 14px !important;
      border:2px solid rgba(255,255,255,.35) !important;
      box-shadow:0 10px 22px rgba(0,0,0,.12);
      position:relative;
      overflow:hidden;
    }
    .btn-back-pro:before{
      content:'';
      position:absolute; top:-40%; left:-40%;
      width:80%; height:180%;
      background:rgba(255,255,255,.18);
      transform:rotate(25deg);
      transition:all .25s ease;
    }
    .btn-back-pro:hover:before{ left:20%; }
  </style>
</head>

<body class="hold-transition skin-blue layout-top-nav">
<div class="wrapper">

  <!-- TOP NAV (sin sidebar) -->
  <header class="main-header">
    <nav class="navbar navbar-static-top" role="navigation" style="margin-left:0;">
      <div class="container">

        <div class="navbar-header">
          <a href="<?= h($backUrl) ?>" class="navbar-brand">
            <b>Historial</b> | <?= h($tabLabel) ?>
          </a>

          <button type="button" class="navbar-toggle collapsed" data-toggle="collapse" data-target="#navbar-collapse">
            <i class="fa fa-bars"></i>
          </button>
        </div>

        <div class="collapse navbar-collapse pull-left" id="navbar-collapse">
          <ul class="nav navbar-nav">
            <li>
              <a href="<?= h($backUrl) ?>"><i class="fa fa-dashboard"></i> Dashboard</a>
            </li>

            <li class="dropdown <?= ($tab==='asistencia' ? 'active' : '') ?>">
              <a href="#" class="dropdown-toggle" data-toggle="dropdown">
                <i class="fa fa-qrcode"></i> Asistencia <span class="caret"></span>
              </a>
              <ul class="dropdown-menu" role="menu">
                <li><a href="scan.php"><i class="fa fa-camera"></i> Escanear</a></li>
                <li><a href="historial.php?tab=asistencia"><i class="fa fa-list"></i> Historial</a></li>
              </ul>
            </li>

            <li class="dropdown <?= ($tab==='extra' ? 'active' : '') ?>">
              <a href="#" class="dropdown-toggle" data-toggle="dropdown">
                <i class="fa fa-clock-o"></i> Horas Extra <span class="caret"></span>
              </a>
              <ul class="dropdown-menu" role="menu">
                <li><a href="scan_extra.php"><i class="fa fa-camera"></i> Escanear</a></li>
                <li><a href="historial.php?tab=extra"><i class="fa fa-list"></i> Historial</a></li>
              </ul>
            </li>
          </ul>
        </div>

        <div class="navbar-custom-menu">
          <ul class="nav navbar-nav">

            <!-- Botón volver visual (derecha) -->
            <li class="hidden-xs">
              <a class="btn-back-pro" style="margin:8px 10px; color:#fff; background:rgba(0,0,0,.12);"
                 href="<?= h($backUrl) ?>">
                <i class="fa fa-arrow-left"></i> Volver
              </a>
            </li>

            <li class="dropdown user user-menu">
              <a href="#" class="dropdown-toggle" data-toggle="dropdown">
                <img src="https://via.placeholder.com/80x80.png?text=U" class="user-image user-image-circle" alt="User">
                <span class="hidden-xs"><?= h($gestorNombre) ?></span>
              </a>
              <ul class="dropdown-menu">
                <li class="user-header">
                  <img src="https://via.placeholder.com/120x120.png?text=U" class="img-circle" alt="User">
                  <p style="margin-top:10px;">
                    <?= h($gestorNombre) ?><br>
                    <small>Estado: <b><?= h(estado_es($estadoActual)) ?></b></small><br>
                    <?php if ($fhActual): ?><small>Último: <?= h($fhActual) ?></small><?php endif; ?>
                  </p>
                </li>
                <li class="user-footer">
                  <div class="pull-left">
                    <a href="dashboard.php" class="btn btn-default btn-flat">Dashboard</a>
                  </div>
                  <div class="pull-right">
                    <a href="logout.php" class="btn btn-default btn-flat">Salir</a>
                  </div>
                </li>
              </ul>
            </li>

          </ul>
        </div>

      </div>
    </nav>
  </header>

  <!-- Content -->
  <div class="content-wrapper" style="margin-left:0;">
    <div class="container">

      <section class="content-header">
        <h1><i class="fa fa-list"></i> Historial <small><?= h($tabLabel) ?></small></h1>
      </section>

      <section class="content">

        <!-- Botón volver dentro del contenido (móvil) -->
        <div class="visible-xs" style="margin-bottom:10px;">
          <a href="<?= h($backUrl) ?>" class="btn btn-primary" style="width:100%; border-radius:12px; font-weight:900;">
            <i class="fa fa-arrow-left"></i> Volver
          </a>
        </div>

        <div class="box">
          <div class="box-header with-border">
            <h3 class="box-title">
              <i class="fa fa-list"></i> Historial
            </h3>
            <div class="pull-right">
              <span class="badge-pill">Estado actual: <?= h(estado_es($estadoActual)) ?></span>
            </div>
          </div>

          <div class="box-body">

            <!-- Tabs botones -->
            <div class="tabbar">
              <div class="btn-group">
                <a href="historial.php?tab=asistencia"
                   class="btn <?= ($tab==='asistencia' ? 'btn-primary' : 'btn-default') ?>">
                  <i class="fa fa-qrcode"></i> Asistencia
                </a>
                <a href="historial.php?tab=extra"
                   class="btn <?= ($tab==='extra' ? 'btn-primary' : 'btn-default') ?>">
                  <i class="fa fa-clock-o"></i> Horas extra
                </a>
              </div>
              <div class="small-muted" style="margin-top:6px;">
                Solo visible mientras estás <b>TRABAJANDO</b>.
              </div>
            </div>

            <?php if ($tab === 'asistencia'): ?>

              <?php if (empty($rowsAsistencia)): ?>
                <div class="alert alert-info" style="border-radius:12px;">Aún no hay marcajes registrados.</div>
              <?php else: ?>
                <div class="table-responsive">
                  <table class="table table-bordered table-striped">
                    <thead>
                      <tr>
                        <th style="width:180px;">Fecha y hora</th>
                        <th>Evento</th>
                        <th style="width:160px;">Estado</th>
                        <th style="width:90px;">Foto</th>
                        <th style="width:280px;">GPS / Mapa</th>
                      </tr>
                    </thead>
                    <tbody>
                      <?php foreach($rowsAsistencia as $r): ?>
                        <?php
                          $id  = (int)($r['id'] ?? 0);
                          $fh  = (string)($r['fh'] ?? '');
                          $ev  = (string)($r['evento'] ?? '');
                          $es  = (string)($r['estado'] ?? '');

                          $lat = trim((string)($r['lat'] ?? ''));
                          $lng = trim((string)($r['lng'] ?? ''));
                          $acc = trim((string)($r['acc'] ?? ''));

                          $photoPath = trim((string)($r['photo_path'] ?? ''));
                          $photoUrl  = ($photoPath !== '') ? normalize_photo_url($photoPath) : '';

                          $mapsUrl = ($lat !== '' && $lng !== '') ? build_maps_url($lat, $lng) : '';
                          $modalId = 'img_' . $id;
                        ?>
                        <tr>
                          <td class="nowrap"><?= h($fh) ?></td>
                          <td><?= h(evento_es($ev)) ?></td>
                          <td><?= h(estado_es($es)) ?></td>

                          <td>
                            <?php if ($photoUrl !== ''): ?>
                              <a href="#" data-toggle="modal" data-target="#<?= h($modalId) ?>" title="Ver foto">
                                <img src="<?= h($photoUrl) ?>" class="thumb" alt="Foto">
                              </a>

                              <div class="modal fade" id="<?= h($modalId) ?>" tabindex="-1" role="dialog">
                                <div class="modal-dialog modal-lg" role="document">
                                  <div class="modal-content" style="border-radius:14px;">
                                    <div class="modal-header">
                                      <button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>
                                      <h4 class="modal-title"><i class="fa fa-camera"></i> Evidencia • <?= h($fh) ?></h4>
                                    </div>
                                    <div class="modal-body">
                                      <div class="text-center">
                                        <img src="<?= h($photoUrl) ?>" style="max-width:100%; border-radius:8px; border:1px solid #e5e7eb;" alt="Foto">
                                      </div>
                                      <hr>
                                      <div class="row">
                                        <div class="col-sm-6">
                                          <b>Evento:</b> <?= h(evento_es($ev)) ?><br>
                                          <b>Estado:</b> <?= h(estado_es($es)) ?><br>
                                          <?php if ($lat !== '' && $lng !== ''): ?>
                                            <b>GPS:</b> <?= h($lat) ?>, <?= h($lng) ?> <?php if ($acc !== ''): ?>(±<?= h($acc) ?>m)<?php endif; ?><br>
                                          <?php else: ?>
                                            <b>GPS:</b> --<br>
                                          <?php endif; ?>
                                        </div>
                                        <div class="col-sm-6 text-right">
                                          <?php if ($mapsUrl !== ''): ?>
                                            <a class="btn btn-primary" target="_blank" rel="noopener" href="<?= h($mapsUrl) ?>" style="border-radius:10px;font-weight:900;">
                                              <i class="fa fa-map-marker"></i> Ver en Google Maps
                                            </a>
                                          <?php endif; ?>
                                          <a class="btn btn-default" target="_blank" rel="noopener" href="<?= h($photoUrl) ?>" style="border-radius:10px;font-weight:900;">
                                            <i class="fa fa-external-link"></i> Abrir imagen
                                          </a>
                                        </div>
                                      </div>
                                    </div>
                                    <div class="modal-footer">
                                      <button type="button" class="btn btn-default" data-dismiss="modal" style="border-radius:10px;font-weight:900;">Cerrar</button>
                                    </div>
                                  </div>
                                </div>
                              </div>

                            <?php else: ?>
                              <span class="text-muted">--</span>
                            <?php endif; ?>
                          </td>

                          <td>
                            <?php if ($lat !== '' && $lng !== ''): ?>
                              <div class="gps-box">
                                <div><i class="fa fa-map-marker text-red"></i> <b><?= h($lat) ?>, <?= h($lng) ?></b></div>
                                <?php if ($acc !== ''): ?>
                                  <small>Precisión aprox: ±<?= h($acc) ?> m</small>
                                <?php else: ?>
                                  <small>Precisión: --</small>
                                <?php endif; ?>
                                <div style="margin-top:6px;">
                                  <a class="btn btn-xs btn-primary" target="_blank" rel="noopener" href="<?= h($mapsUrl) ?>" style="border-radius:10px;font-weight:900;">
                                    <i class="fa fa-external-link"></i> Ver en Maps
                                  </a>
                                </div>
                              </div>
                            <?php else: ?>
                              <span class="text-muted">--</span>
                            <?php endif; ?>
                          </td>

                        </tr>
                      <?php endforeach; ?>
                    </tbody>
                  </table>
                </div>
              <?php endif; ?>

            <?php else: ?>

              <?php if (empty($rowsExtra)): ?>
                <div class="alert alert-info" style="border-radius:12px;">Aún no hay registros de horas extra.</div>
              <?php else: ?>
                <div class="table-responsive">
                  <table class="table table-bordered table-striped">
                    <thead>
                      <tr>
                        <th style="width:120px;">Fecha</th>
                        <th style="width:120px;">Salida oficial</th>
                        <th style="width:120px;">Plan</th>
                        <th style="width:120px;">Real</th>
                        <th style="width:140px;">Minutos</th>
                        <th style="width:140px;">Horas</th>
                        <th style="width:100px;">Factor</th>
                      </tr>
                    </thead>
                    <tbody>
                      <?php foreach($rowsExtra as $r):
                        $mins = (int)($r['overtime_minutes'] ?? 0);
                        $hours = round($mins / 60, 2);
                      ?>
                        <tr>
                          <td><?= h((string)$r['fecha']) ?></td>
                          <td><?= h(fmt_hhmm((string)$r['schedule_end'])) ?></td>
                          <td><?= h(fmt_hhmm((string)$r['planned_end'])) ?></td>
                          <td><?= h(fmt_hhmm((string)$r['actual_end'])) ?></td>
                          <td><?= h((string)$mins) ?></td>
                          <td><?= h((string)$hours) ?></td>
                          <td><?= h((string)($r['factor'] ?? '1.5')) ?></td>
                        </tr>
                      <?php endforeach; ?>
                    </tbody>
                  </table>
                </div>
              <?php endif; ?>

            <?php endif; ?>

          </div>
        </div>

      </section>
    </div>
  </div>

  <footer class="main-footer" style="font-size:12px;margin-left:0;">
    <div class="container">
      <div class="pull-right hidden-xs">Asistencia Gestores</div>
      <strong>© <?= date('Y') ?> JPortales</strong>
    </div>
  </footer>

</div>

<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.7.1/jquery.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/twitter-bootstrap/3.4.1/js/bootstrap.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/2.4.18/js/adminlte.min.js"></script>
</body>
</html>
