<?php
// rrhh_home/asistencia/ficha_personal.php
// "Ficha personal" (Asistencia) – muestra la ficha del colaborador ligado al gestor en sesión.
// PHP 8.1.33

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');

/* 🔧 DEBUG rápido (si necesitas ver el error exacto):
ini_set('display_errors','1');
ini_set('display_startup_errors','1');
error_reporting(E_ALL);
*/

require_once __DIR__ . '/inc/bootstrap.php';
$user = require_gestor_login();
require_working_or_scan($user, basename($_SERVER['SCRIPT_NAME'])); // regla: nada se usa si no está trabajando

// DB RRHH (mysqli) está en rrhh_home/dbcon.php
require_once __DIR__ . '/../dbcon.php';
mysqli_set_charset($con, 'utf8mb4');

// empleado del gestor (solo el suyo)
$idempleado = (int)($_SESSION['idempleado'] ?? 0);
if ($idempleado <= 0) $idempleado = (int)($_SESSION['gestor_empleado_id'] ?? 0);

$page_title = 'Ficha personal';

if ($idempleado <= 0) {
  include __DIR__ . '/inc/layout_top.php';
  echo "<div class='box box-danger'><div class='box-body'><b>Error:</b> No hay idempleado asociado en sesión.</div></div>";
  include __DIR__ . '/inc/layout_bottom.php';
  exit;
}

/* ===================== Helpers (sin chocar con bootstrap) ===================== */
if (!function_exists('fp_h')) {
  function fp_h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
}
if (!function_exists('fp_fmtFecha')) {
  function fp_fmtFecha($f): string {
    if (!$f || $f === '0000-00-00') return '';
    $ts = strtotime((string)$f);
    if (!$ts) return fp_h((string)$f);
    return date('d/m/Y', $ts);
  }
}
if (!function_exists('fp_asset_url')) {
  // Ajusta rutas guardadas en DB para que funcionen desde /asistencia/
  function fp_asset_url(?string $p): string {
    $p = trim((string)$p);
    if ($p === '') return '';
    if (preg_match('#^(https?://|//)#i', $p)) return $p;
    if (str_starts_with($p, '/')) return $p;
    return '../' . ltrim($p, '/');
  }
}

/* ===================== 1) Empleado ===================== */
$sqlEmp = "
  SELECT id, categoria, nombre_completo, tipo_documento, cedula, nacionalidad,
         fecha_nacimiento, numero_emergencia, estado_civil,
         year_ingreso, mes_ingreso, dia_ingreso, imagen1, sumar, restar
  FROM empleados_planilla
  WHERE id = $idempleado
  LIMIT 1
";
$empRes = mysqli_query($con, $sqlEmp);
if (!$empRes) {
  include __DIR__ . '/inc/layout_top.php';
  echo "<div class='box box-danger'><div class='box-body'><b>Error DB:</b> ".fp_h(mysqli_error($con))."</div></div>";
  include __DIR__ . '/inc/layout_bottom.php';
  exit;
}
$emp = mysqli_fetch_assoc($empRes);
if (!$emp) {
  include __DIR__ . '/inc/layout_top.php';
  echo "<div class='box box-warning'><div class='box-body'>No se encontró colaborador con id ".fp_h((string)$idempleado).".</div></div>";
  include __DIR__ . '/inc/layout_bottom.php';
  exit;
}

// Fecha ingreso armada
$fecha_ingreso = '';
if (!empty($emp['year_ingreso']) && !empty($emp['mes_ingreso']) && !empty($emp['dia_ingreso'])) {
  $y = (int)$emp['year_ingreso'];
  $m = str_pad((string)(int)$emp['mes_ingreso'], 2, '0', STR_PAD_LEFT);
  $d = str_pad((string)(int)$emp['dia_ingreso'], 2, '0', STR_PAD_LEFT);
  $fecha_ingreso = "$y-$m-$d";
}

/* ===================== 2) Vacaciones ===================== */
$vac = [];
$qVac = "SELECT id, fecha_desde, fecha_hasta, motivo, cantidad_dias, imagen1, fecha_documento
         FROM ingreso_vacaciones
         WHERE idempleado = $idempleado AND status='1'
         ORDER BY fecha_desde DESC";
$rVac = mysqli_query($con, $qVac);
if ($rVac) while ($row = mysqli_fetch_assoc($rVac)) $vac[] = $row;

/* ===================== 3) Vacaciones Extra ===================== */
$vacx = [];
$qVacX = "SELECT id, fecha_desde, fecha_hasta, motivo, cantidad_dias, imagen1, fecha_documento
          FROM ingreso_vacaciones_extra
          WHERE idempleado = $idempleado
          ORDER BY fecha_desde DESC";
$rVacX = mysqli_query($con, $qVacX);
if ($rVacX) while ($row = mysqli_fetch_assoc($rVacX)) $vacx[] = $row;

/* ===================== 4) Incapacidades ===================== */
$inc = [];
$qInc = "SELECT id, fecha_desde, fecha_hasta, dividido_35, dias_calculo, motivo, cantidad_dias, imagen1, fecha_documento
         FROM ingreso_inscapacidad
         WHERE idempleado = $idempleado
         ORDER BY fecha_desde DESC";
$rInc = mysqli_query($con, $qInc);
if ($rInc) while ($row = mysqli_fetch_assoc($rInc)) $inc[] = $row;

/* ===================== 5) Documentos ===================== */
$docGenerales = [];
$docEspecificos = [];

$empCat = mysqli_real_escape_string($con, (string)($emp['categoria'] ?? ''));

$qG = "
  SELECT id, idempleado, categoria, descripcion, comentario, imagen1, fecha_documento
  FROM ingreso_documentosgenerales_empleadosplanilla
  WHERE (idempleado = $idempleado)
     OR ((idempleado IS NULL OR idempleado = 0) AND categoria = '$empCat')
  ORDER BY fecha_documento DESC, id DESC
";
$rG = mysqli_query($con, $qG);
if ($rG) while ($row = mysqli_fetch_assoc($rG)) $docGenerales[] = $row;

$qE = "SELECT id, idempleado, categoria, descripcion, comentario, imagen1, fecha_documento
       FROM ingreso_documentos_empleadosplanilla
       WHERE idempleado = $idempleado
       ORDER BY fecha_documento DESC, id DESC";
$rE = mysqli_query($con, $qE);
if ($rE) while ($row = mysqli_fetch_assoc($rE)) $docEspecificos[] = $row;

/* ===================== 6) Feriados por vacaciones (si existe tabla) ===================== */
$feriadosPorVac = [];
$tblCheck = mysqli_query($con, "SHOW TABLES LIKE 'vacaciones_feriados'");
if ($tblCheck && mysqli_num_rows($tblCheck) > 0) {
  foreach ($vac as $v) {
    $idv = (int)$v['id'];
    $feriadosPorVac[$idv] = [];
    $qr = mysqli_query($con, "SELECT fecha FROM vacaciones_feriados WHERE idvacacion = $idv ORDER BY fecha");
    if ($qr) while ($fr = mysqli_fetch_assoc($qr)) $feriadosPorVac[$idv][] = (string)$fr['fecha'];
  }
}

include __DIR__ . '/inc/layout_top.php';
?>

<style>
.viewer-overlay{position:fixed; inset:0; background:rgba(0,0,0,.7); display:none; z-index:9999; overflow:auto; padding:40px 20px;}
.viewer-box{background:#fff; margin:auto; max-width:1100px; border-radius:10px; box-shadow:0 10px 30px rgba(0,0,0,.4); position:relative;}
.viewer-header{padding:12px 16px; border-bottom:1px solid #eee; font-weight:700; background:#f7f7f7;}
.viewer-close{position:absolute; right:12px; top:8px; font-size:20px; width:36px; height:36px; border-radius:50%; cursor:pointer; color:#fff; background:#d9534f; border:2px solid #fff; display:flex; align-items:center; justify-content:center;}
.viewer-content{padding:12px;}
.viewer-img{max-width:100%; transform-origin:center center; cursor:grab;}
.viewer-instructions{padding:6px 12px 14px; font-size:12px; color:#666;}
</style>

<section class="content-header">
  <h1><i class="fa fa-id-card"></i> Ficha personal <small>Detalle del colaborador</small></h1>
</section>

<section class="content">

  <!-- Datos del colaborador -->
  <div class="box box-primary">
    <div class="box-header with-border">
      <h3 class="box-title" style="font-size:17px;">Datos del colaborador</h3>
      <div class="box-tools pull-right">
        <!-- si existe el pdf en raíz, lo abrimos -->
        <a class="btn btn-default btn-sm" href="<?= fp_h(fp_asset_url('colaborador_detalle_pdf.php?id='.(int)$idempleado)) ?>" target="_blank" rel="noopener">
          <i class="fa fa-file-pdf-o"></i> Generar PDF
        </a>
      </div>
    </div>
    <div class="box-body">
      <div class="row">
        <div class="col-sm-9">
          <table class="table table-condensed">
            <tr><th style="width:180px">Nombre completo</th><td><?= fp_h($emp['nombre_completo'] ?? '') ?></td></tr>
            <tr><th>Categoría</th><td><?= fp_h($emp['categoria'] ?? '') ?></td></tr>
            <tr><th>Documento</th><td><?= fp_h(($emp['tipo_documento'] ?? '').' — '.($emp['cedula'] ?? '')) ?></td></tr>
            <tr><th>Nacionalidad</th><td><?= fp_h($emp['nacionalidad'] ?? '') ?></td></tr>
            <tr><th>Fecha nacimiento</th><td><?= fp_fmtFecha($emp['fecha_nacimiento'] ?? '') ?></td></tr>
            <tr><th>Fecha ingreso</th><td><?= fp_fmtFecha($fecha_ingreso) ?></td></tr>
            <tr><th>Estado civil</th><td><?= fp_h($emp['estado_civil'] ?? '') ?></td></tr>
            <tr><th>Tel. emergencia</th><td><?= fp_h($emp['numero_emergencia'] ?? '') ?></td></tr>
          </table>
        </div>

        <div class="col-sm-3 text-center">
          <?php if (!empty($emp['imagen1'])): ?>
            <?php $u = fp_asset_url((string)$emp['imagen1']); ?>
            <img src="<?= fp_h($u) ?>"
                 style="max-width:100%;max-height:220px;border-radius:6px;border:1px solid #ddd;cursor:zoom-in;"
                 onclick="openImageViewer('<?= fp_h($u) ?>','Documento / foto del colaborador')">
          <?php else: ?>
            <div style="width:100%;height:220px;border:1px dashed #ccc;display:flex;align-items:center;justify-content:center;border-radius:6px;">
              <span class="text-muted">Sin imagen</span>
            </div>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>

  <!-- Vacaciones -->
  <div class="box box-success">
    <div class="box-header with-border"><h3 class="box-title" style="font-size:17px;">Vacaciones</h3></div>
    <div class="box-body">
      <?php if (!$vac): ?>
        <p class="text-muted">Sin registros de vacaciones.</p>
      <?php else: ?>
        <div class="table-responsive">
          <table class="table table-striped table-bordered">
            <thead>
              <tr>
                <th>#</th><th>Fecha doc</th><th>Desde</th><th>Hasta</th><th>Días</th><th>Motivo</th><th>Adjunto</th><th>Feriados</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($vac as $i=>$v): ?>
                <tr>
                  <td><?= $i+1 ?></td>
                  <td><?= fp_fmtFecha($v['fecha_documento'] ?? '') ?></td>
                  <td><?= fp_fmtFecha($v['fecha_desde'] ?? '') ?></td>
                  <td><?= fp_fmtFecha($v['fecha_hasta'] ?? '') ?></td>
                  <td><?= fp_h($v['cantidad_dias'] ?? '') ?></td>
                  <td><?= fp_h($v['motivo'] ?? '') ?></td>
                  <td>
                    <?php if (!empty($v['imagen1'])): ?>
                      <?php $url = fp_asset_url((string)$v['imagen1']); $isPdf = stripos($url,'.pdf')!==false; ?>
                      <button class="btn btn-xs btn-info" onclick="<?= $isPdf ? "openPdfViewer('".fp_h($url)."','Adjunto vacaciones')" : "openImageViewer('".fp_h($url)."','Adjunto vacaciones')" ?>">
                        <i class="fa fa-file"></i> Ver
                      </button>
                    <?php else: ?><span class="text-muted">—</span><?php endif; ?>
                  </td>
                  <td>
                    <?php
                      $idv = (int)($v['id'] ?? 0);
                      if (!empty($feriadosPorVac[$idv])) echo fp_h(implode(', ', $feriadosPorVac[$idv]));
                      else echo '<span class="text-muted">—</span>';
                    ?>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      <?php endif; ?>
    </div>
  </div>

  <!-- Vacaciones adicionales -->
  <div class="box box-info">
    <div class="box-header with-border"><h3 class="box-title" style="font-size:17px;">Vacaciones adicionales</h3></div>
    <div class="box-body">
      <?php if (!$vacx): ?>
        <p class="text-muted">Sin registros de vacaciones adicionales.</p>
      <?php else: ?>
        <div class="table-responsive">
          <table class="table table-striped table-bordered">
            <thead><tr><th>#</th><th>Fecha doc</th><th>Días</th><th>Motivo</th><th>Adjunto</th></tr></thead>
            <tbody>
              <?php foreach ($vacx as $i=>$v): ?>
                <tr>
                  <td><?= $i+1 ?></td>
                  <td><?= fp_fmtFecha($v['fecha_documento'] ?? '') ?></td>
                  <td><?= fp_h($v['cantidad_dias'] ?? '') ?></td>
                  <td><?= fp_h($v['motivo'] ?? '') ?></td>
                  <td>
                    <?php if (!empty($v['imagen1'])): ?>
                      <?php $url = fp_asset_url((string)$v['imagen1']); $isPdf = stripos($url,'.pdf')!==false; ?>
                      <button class="btn btn-xs btn-info" onclick="<?= $isPdf ? "openPdfViewer('".fp_h($url)."','Adjunto vacaciones adicionales')" : "openImageViewer('".fp_h($url)."','Adjunto vacaciones adicionales')" ?>">
                        <i class="fa fa-file"></i> Ver
                      </button>
                    <?php else: ?><span class="text-muted">—</span><?php endif; ?>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      <?php endif; ?>
    </div>
  </div>

  <!-- Incapacidades -->
  <div class="box box-warning">
    <div class="box-header with-border"><h3 class="box-title" style="font-size:17px;">Incapacidades</h3></div>
    <div class="box-body">
      <?php if (!$inc): ?>
        <p class="text-muted">Sin registros de incapacidad.</p>
      <?php else: ?>
        <div class="table-responsive">
          <table class="table table-striped table-bordered">
            <thead>
              <tr>
                <th>#</th><th>Fecha doc</th><th>Desde</th><th>Hasta</th><th>Días</th><th>Dividido 35</th><th>Días cálculo</th><th>Motivo</th><th>Adjunto</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($inc as $i=>$v): ?>
                <tr>
                  <td><?= $i+1 ?></td>
                  <td><?= fp_fmtFecha($v['fecha_documento'] ?? '') ?></td>
                  <td><?= fp_fmtFecha($v['fecha_desde'] ?? '') ?></td>
                  <td><?= fp_fmtFecha($v['fecha_hasta'] ?? '') ?></td>
                  <td><?= fp_h($v['cantidad_dias'] ?? '') ?></td>
                  <td><?= fp_h($v['dividido_35'] ?? '') ?></td>
                  <td><?= fp_h($v['dias_calculo'] ?? '') ?></td>
                  <td><?= fp_h($v['motivo'] ?? '') ?></td>
                  <td>
                    <?php if (!empty($v['imagen1'])): ?>
                      <?php $url = fp_asset_url((string)$v['imagen1']); $isPdf = stripos($url,'.pdf')!==false; ?>
                      <button class="btn btn-xs btn-info" onclick="<?= $isPdf ? "openPdfViewer('".fp_h($url)."','Adjunto incapacidad')" : "openImageViewer('".fp_h($url)."','Adjunto incapacidad')" ?>">
                        <i class="fa fa-file"></i> Ver
                      </button>
                    <?php else: ?><span class="text-muted">—</span><?php endif; ?>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      <?php endif; ?>
    </div>
  </div>

  <!-- Documentos -->
  <div class="box box-default">
    <div class="box-header with-border"><h3 class="box-title" style="font-size:17px;">Documentos</h3></div>
    <div class="box-body">
      <?php if (!$docGenerales && !$docEspecificos): ?>
        <p class="text-muted">Sin documentos.</p>
      <?php else: ?>
        <div class="row">
          <div class="col-sm-6">
            <h4>Generales <small class="text-muted">(por categoría)</small></h4>
            <div class="table-responsive">
              <table class="table table-striped table-bordered">
                <thead><tr><th>#</th><th>Fecha</th><th>Categoría</th><th>Descripción</th><th>Comentario</th><th>Adjunto</th></tr></thead>
                <tbody>
                  <?php foreach ($docGenerales as $i=>$d): ?>
                    <tr>
                      <td><?= $i+1 ?></td>
                      <td><?= fp_fmtFecha($d['fecha_documento'] ?? '') ?></td>
                      <td><?= fp_h($d['categoria'] ?? '') ?></td>
                      <td><?= fp_h($d['descripcion'] ?? '') ?></td>
                      <td><?= fp_h($d['comentario'] ?? '') ?></td>
                      <td>
                        <?php if (!empty($d['imagen1'])): ?>
                          <?php $url = fp_asset_url((string)$d['imagen1']); $isPdf = stripos($url,'.pdf')!==false; ?>
                          <button class="btn btn-xs btn-info" onclick="<?= $isPdf ? "openPdfViewer('".fp_h($url)."','Documento general')" : "openImageViewer('".fp_h($url)."','Documento general')" ?>">
                            <i class="fa fa-file"></i> Ver
                          </button>
                        <?php else: ?><span class="text-muted">—</span><?php endif; ?>
                      </td>
                    </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          </div>

          <div class="col-sm-6">
            <h4>Específicos <small class="text-muted">(por empleado)</small></h4>
            <div class="table-responsive">
              <table class="table table-striped table-bordered">
                <thead><tr><th>#</th><th>Fecha</th><th>Categoría</th><th>Descripción</th><th>Comentario</th><th>Adjunto</th></tr></thead>
                <tbody>
                  <?php foreach ($docEspecificos as $i=>$d): ?>
                    <tr>
                      <td><?= $i+1 ?></td>
                      <td><?= fp_fmtFecha($d['fecha_documento'] ?? '') ?></td>
                      <td><?= fp_h($d['categoria'] ?? '') ?></td>
                      <td><?= fp_h($d['descripcion'] ?? '') ?></td>
                      <td><?= fp_h($d['comentario'] ?? '') ?></td>
                      <td>
                        <?php if (!empty($d['imagen1'])): ?>
                          <?php $url = fp_asset_url((string)$d['imagen1']); $isPdf = stripos($url,'.pdf')!==false; ?>
                          <button class="btn btn-xs btn-info" onclick="<?= $isPdf ? "openPdfViewer('".fp_h($url)."','Documento específico')" : "openImageViewer('".fp_h($url)."','Documento específico')" ?>">
                            <i class="fa fa-file"></i> Ver
                          </button>
                        <?php else: ?><span class="text-muted">—</span><?php endif; ?>
                      </td>
                    </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          </div>
        </div>
      <?php endif; ?>
    </div>
  </div>

</section>

<!-- Viewer -->
<div id="viewer" class="viewer-overlay" onclick="closeViewer()">
  <div class="viewer-box" onclick="event.stopPropagation()">
    <div class="viewer-header">
      <span id="viewerTitle">Documento</span>
      <span class="viewer-close" onclick="closeViewer()" title="Cerrar">×</span>
    </div>
    <div class="viewer-content">
      <div id="viewerImgWrap" style="display:none;text-align:center;">
        <img id="viewerImg" class="viewer-img" src="" alt="Imagen">
      </div>
      <embed id="viewerPdf" type="application/pdf" style="width:100%; height:80vh; border:0; display:none;" />
    </div>
    <div class="viewer-instructions">
      <b>Imagen:</b> rueda/pinch para zoom, arrastra para mover. &nbsp; <b>PDF:</b> usa controles del visor.
    </div>
  </div>
</div>

<script>
function closeViewer(){
  const o = document.getElementById('viewer');
  const img = document.getElementById('viewerImg');
  const pdf = document.getElementById('viewerPdf');
  o.style.display = 'none';
  img.src = '';
  pdf.src = '';
  document.onmousemove = null;
  document.onmouseup = null;
}
function openPdfViewer(url, title){
  document.getElementById('viewerTitle').textContent = title || 'PDF';
  document.getElementById('viewerImgWrap').style.display = 'none';
  const pdf = document.getElementById('viewerPdf');
  pdf.style.display = 'block';
  pdf.src = url;
  document.getElementById('viewer').style.display = 'block';
}
function openImageViewer(url, title){
  document.getElementById('viewerTitle').textContent = title || 'Imagen';
  document.getElementById('viewerPdf').style.display = 'none';
  const w = document.getElementById('viewerImgWrap');
  w.style.display = 'block';
  const img = document.getElementById('viewerImg');
  img.src = url;
  document.getElementById('viewer').style.display = 'block';
}
document.addEventListener('keydown', (e)=>{ if(e.key==='Escape') closeViewer(); });
</script>

<?php include __DIR__ . '/inc/layout_bottom.php'; ?>
