<?php
// rrhh_home/asistencia/extra_api.php
// Horas extra por QR dinámico
// ✅ BLOQUEO DURO: si existe overtime_qr de AYER sin cerrar => lock total (no permite scan, plan, delete, nada)
// ✅ Mensaje estilo "manual": QR bloqueado
//
// PHP 8.1.x – UTF-8

declare(strict_types=1);

session_start();
date_default_timezone_set('America/Costa_Rica');
header('Content-Type: application/json; charset=utf-8');

if (!isset($_SESSION['gestor_id']) || (int)$_SESSION['gestor_id'] <= 0) {
  echo json_encode(['ok'=>false,'msg'=>'Sesión expirada']); exit;
}

require_once __DIR__ . '/../dbcon.php';
if (!isset($con) || !($con instanceof mysqli)) {
  echo json_encode(['ok'=>false,'msg'=>'Sin conexión a BD']); exit;
}
mysqli_set_charset($con,'utf8mb4');

function respond(array $a){ echo json_encode($a, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }

function norm_time(?string $s): ?string {
  $s = trim((string)$s);
  if ($s === '') return null;
  if (preg_match('/^\d{2}:\d{2}$/', $s)) $s .= ':00';
  if (!preg_match('/^\d{2}:\d{2}:\d{2}$/', $s)) return null;
  $dt = DateTime::createFromFormat('H:i:s', $s);
  return $dt ? $dt->format('H:i:s') : null;
}

function minutes_between(string $start, string $end): int {
  $base = '2000-01-01 ';
  $a = strtotime($base.$start);
  $b = strtotime($base.$end);
  if ($a===false || $b===false) return 0;
  return (int)floor(($b-$a)/60);
}

function fmt_hhmm(string $t): string { return substr($t,0,5); }

function round_up_half_hours(int $minutes): float {
  if ($minutes <= 0) return 0.0;
  $blocks = (int)ceil($minutes / 30);
  return $blocks * 0.5;
}

// =====================================================
// ✅ MISMO TOKEN que jornada_api.php (QR dinámico firmado)
// =====================================================
const TOKEN_SECRET  = 'CAMBIA_ESTA_LLAVE_LARGA_32+CHARS_2026_RRHH_JPORTALES';
const TOKEN_SECONDS = 40;

function b64url(string $bin): string { return rtrim(strtr(base64_encode($bin), '+/', '-_'), '='); }
function make_token(int $windowStart): string {
  $payload = 'J1.' . $windowStart;
  $sigBin = hash_hmac('sha256', $payload, TOKEN_SECRET, true);
  $sig = b64url(substr($sigBin, 0, 16));
  return $payload . '.' . $sig;
}
function validate_token(string $token): bool {
  $p = explode('.', $token);
  if (count($p) !== 3) return false;
  if ($p[0] !== 'J1') return false;

  $windowStart = (int)$p[1];
  if ($windowStart <= 0) return false;
  if (($windowStart % TOKEN_SECONDS) !== 0) return false;

  $now = time();
  $current = (int)(floor($now / TOKEN_SECONDS) * TOKEN_SECONDS);
  $prev    = $current - TOKEN_SECONDS;

  if ($windowStart !== $current && $windowStart !== $prev) return false;

  $expected = make_token($windowStart);
  return hash_equals($expected, $token);
}

function assert_qr_token_or_fail(string $token): void {
  $token = trim($token);
  if ($token === '') respond(['ok'=>false,'msg'=>'Token vacío']);
  if (!validate_token($token)) {
    respond(['ok'=>false,'msg'=>'Token inválido o expirado. Escanea el QR actual.']);
  }
}

// ============================
// Tabla overtime_qr (si no existe)
// ============================
@mysqli_query($con, "
CREATE TABLE IF NOT EXISTS overtime_qr (
  id INT AUTO_INCREMENT PRIMARY KEY,
  idempleado INT NOT NULL,
  gestor_id INT NOT NULL,
  fecha DATE NOT NULL,
  schedule_end TIME NOT NULL,
  planned_end TIME NULL,
  planned_at DATETIME NULL,
  actual_end TIME NULL,
  closed_at DATETIME NULL,
  overtime_minutes INT NOT NULL DEFAULT 0,
  factor DECIMAL(3,1) NOT NULL DEFAULT 1.5,
  note VARCHAR(255) NULL,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NULL,
  UNIQUE KEY uq_emp_fecha (idempleado, fecha),
  KEY idx_gestor (gestor_id),
  KEY idx_fecha (fecha)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
");

// ============================
// Map gestor -> idempleado
// ============================
function get_idempleado_from_gestor(mysqli $con, int $gestor_id): int {
  if (isset($_SESSION['idempleado']) && (int)$_SESSION['idempleado'] > 0) return (int)$_SESSION['idempleado'];
  if (isset($_SESSION['gestor_empleado_id']) && (int)$_SESSION['gestor_empleado_id'] > 0) return (int)$_SESSION['gestor_empleado_id'];
  if (isset($_SESSION['idempleado_planilla']) && (int)$_SESSION['idempleado_planilla'] > 0) return (int)$_SESSION['idempleado_planilla'];

  $q = "SELECT idempleado FROM planilla_gestores WHERE id=? LIMIT 1";
  if ($st = mysqli_prepare($con, $q)) {
    mysqli_stmt_bind_param($st,'i',$gestor_id);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$idempleado);
    if (mysqli_stmt_fetch($st)) { mysqli_stmt_close($st); return (int)$idempleado; }
    mysqli_stmt_close($st);
  }
  return 0;
}

// ============================
// Hora oficial salida hoy
// ============================
function get_today_schedule_end(mysqli $con): ?string {
  $dow = (int)date('N');
  $q = "SELECT active, end_time FROM work_schedule WHERE day_of_week=? LIMIT 1";
  if ($st = mysqli_prepare($con,$q)) {
    mysqli_stmt_bind_param($st,'i',$dow);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$active,$end);
    if (mysqli_stmt_fetch($st)) {
      mysqli_stmt_close($st);
      if ((int)$active !== 1) return null;
      return $end ? (string)$end : null;
    }
    mysqli_stmt_close($st);
  }
  return null;
}

// ============================
// Inyectar a extras_mes/extras_detalle
// ============================
function ensure_extras_header(mysqli $con, int $idempleado, string $mes, int $uid=0): int {
  $id = 0;
  $q = "SELECT id FROM extras_mes WHERE idempleado=? AND mes=? LIMIT 1";
  if ($st = mysqli_prepare($con,$q)) {
    mysqli_stmt_bind_param($st,'is',$idempleado,$mes);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$idtmp);
    if (mysqli_stmt_fetch($st)) $id = (int)$idtmp;
    mysqli_stmt_close($st);
  }
  if ($id>0) return $id;

  $ins = "INSERT INTO extras_mes (idempleado, mes, creado_por) VALUES (?,?,?)";
  if ($st = mysqli_prepare($con,$ins)) {
    mysqli_stmt_bind_param($st,'isi',$idempleado,$mes,$uid);
    mysqli_stmt_execute($st);
    $id = (int)mysqli_insert_id($con);
    mysqli_stmt_close($st);
  }
  return $id;
}

function upsert_extras_day_overwrite(mysqli $con, int $idempleado, string $fecha, float $horas, string $factor, string $desc, int $uid=0): bool {
  $mes = substr($fecha,0,7);
  $id_extras_mes = ensure_extras_header($con,$idempleado,$mes,$uid);

  $sql = "INSERT INTO extras_detalle (id_extras_mes, fecha, horas, factor, descripcion, creado_por)
          VALUES (?,?,?,?,?,?)
          ON DUPLICATE KEY UPDATE
            horas=VALUES(horas),
            factor=VALUES(factor),
            descripcion=VALUES(descripcion),
            updated_by=VALUES(creado_por),
            updated_at=NOW()";
  if (!$st = mysqli_prepare($con,$sql)) return false;

  mysqli_stmt_bind_param($st,'issdsi',$id_extras_mes,$fecha,$horas,$factor,$desc,$uid);
  $ok = mysqli_stmt_execute($st);
  mysqli_stmt_close($st);
  return $ok;
}

function delete_extras_day(mysqli $con, int $idempleado, string $fecha): void {
  $mes = substr($fecha,0,7);
  $id_extras_mes = 0;
  $q = "SELECT id FROM extras_mes WHERE idempleado=? AND mes=? LIMIT 1";
  if ($st=mysqli_prepare($con,$q)) {
    mysqli_stmt_bind_param($st,'is',$idempleado,$mes);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st,$tmp);
    if (mysqli_stmt_fetch($st)) $id_extras_mes = (int)$tmp;
    mysqli_stmt_close($st);
  }
  if ($id_extras_mes>0) {
    $d="DELETE FROM extras_detalle WHERE id_extras_mes=? AND fecha=? LIMIT 1";
    if ($st=mysqli_prepare($con,$d)) {
      mysqli_stmt_bind_param($st,'is',$id_extras_mes,$fecha);
      @mysqli_stmt_execute($st);
      mysqli_stmt_close($st);
    }
  }
}

// ============================
// ASISTENCIA: cerrar jornada desde horas extra
// ============================
function asistencia_last_state(mysqli $con, int $idempleado, string $fecha): array {
  $q="SELECT id, evento, estado, DATE_FORMAT(fecha_hora,'%Y-%m-%d %H:%i:%s') fh
      FROM ausencias_marcajes
      WHERE idempleado=? AND DATE(fecha_hora)=?
      ORDER BY id DESC LIMIT 1";
  if($st=mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'is',$idempleado,$fecha);
    mysqli_stmt_execute($st);
    $rs=mysqli_stmt_get_result($st);
    $r=$rs?mysqli_fetch_assoc($rs):null;
    mysqli_stmt_close($st);
    if($r){
      return [
        'id'=>(int)$r['id'],
        'evento'=>(string)$r['evento'],
        'estado'=>(string)$r['estado'],
        'fh'=>(string)$r['fh'],
      ];
    }
  }
  return [];
}

function asistencia_insert_mark(mysqli $con, int $idempleado, int $gestor_id, string $token, string $evento, string $estado): void {
  $ip = $_SERVER['REMOTE_ADDR'] ?? '';
  $ua = substr((string)($_SERVER['HTTP_USER_AGENT'] ?? ''),0,200);

  $sql="INSERT INTO ausencias_marcajes (idempleado, gestor_id, token, evento, estado, fecha_hora, ip, user_agent)
        VALUES (?,?,?,?,?,NOW(),?,?)";
  if($st=mysqli_prepare($con,$sql)){
    mysqli_stmt_bind_param($st,'iisssss',$idempleado,$gestor_id,$token,$evento,$estado,$ip,$ua);
    @mysqli_stmt_execute($st);
    mysqli_stmt_close($st);
  }
}

// ============================
// ✅ BLOQUEO: extra pendiente AYER
// Regla: existe overtime_qr de AYER con actual_end IS NULL
// ============================
function get_lock_yesterday(mysqli $con, int $idempleado): ?array {
  $y = date('Y-m-d', strtotime('-1 day'));
  $q = "SELECT fecha, schedule_end, planned_end
        FROM overtime_qr
        WHERE idempleado=? AND fecha=? AND actual_end IS NULL
        LIMIT 1";
  if ($st = mysqli_prepare($con,$q)) {
    mysqli_stmt_bind_param($st,'is',$idempleado,$y);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    $row = $rs ? mysqli_fetch_assoc($rs) : null;
    mysqli_stmt_close($st);
    if ($row) {
      return [
        'date' => (string)$row['fecha'],
        'schedule_end' => $row['schedule_end'] ? fmt_hhmm((string)$row['schedule_end']) : null,
        'planned_end'  => $row['planned_end']  ? fmt_hhmm((string)$row['planned_end'])  : null,
      ];
    }
  }
  return null;
}

// ============================
// Acciones
// ============================
$action = $_POST['action'] ?? $_GET['action'] ?? '';

$gestor_id = (int)$_SESSION['gestor_id'];
$idempleado = get_idempleado_from_gestor($con, $gestor_id);
if ($idempleado <= 0) respond(['ok'=>false,'msg'=>'Gestor sin empleado asociado (idempleado).']);

$today = date('Y-m-d');

// ✅ Primero: chequeo bloqueo AYER (antes de todo)
$lockInfo = get_lock_yesterday($con, $idempleado);
if ($lockInfo) {
  $msg = "🔒 QR BLOQUEADO: Existe una Hora Extra activa del {$lockInfo['date']} (no cerrada). "
       . "Por seguridad, el módulo de Horas Extra está bloqueado. Contacta al administrador para regularizar.";
  // status sí puede responder ok=true pero lock=true
  if ($action === 'status') {
    respond([
      'ok' => true,
      'lock' => true,
      'lock_msg' => $msg,
      'lock' => true,
      'lock_date' => $lockInfo['date'],
      'lock_schedule_end' => $lockInfo['schedule_end'],
      'lock_planned_end'  => $lockInfo['planned_end'],
    ]);
  }
  // cualquier otra acción: no permitir nada
  respond([
    'ok' => false,
    'lock' => true,
    'msg' => $msg,
    'lock_date' => $lockInfo['date'],
    'lock_schedule_end' => $lockInfo['schedule_end'],
    'lock_planned_end'  => $lockInfo['planned_end'],
  ]);
}

// Desde aquí: NO hay lock de ayer
$schedule_end = get_today_schedule_end($con);
if (!$schedule_end) respond(['ok'=>false,'msg'=>'No hay horario activo hoy o falta hora de salida en work_schedule.']);

// ============================
// status (solo hoy)
// ============================
if ($action === 'status') {
  $row = null;
  $q="SELECT planned_end, actual_end, schedule_end FROM overtime_qr WHERE idempleado=? AND fecha=? LIMIT 1";
  if($st=mysqli_prepare($con,$q)){
    mysqli_stmt_bind_param($st,'is',$idempleado,$today);
    mysqli_stmt_execute($st);
    $rs=mysqli_stmt_get_result($st);
    $row=$rs?mysqli_fetch_assoc($rs):null;
    mysqli_stmt_close($st);
  }
  if(!$row){
    respond([
      'ok'=>true,
      'lock'=>false,
      'has'=>false,
      'today'=>$today,
      'schedule_end'=>fmt_hhmm($schedule_end)
    ]);
  }
  respond([
    'ok'=>true,
    'lock'=>false,
    'has'=>true,
    'today'=>$today,
    'schedule_end'=>fmt_hhmm((string)$row['schedule_end']),
    'planned_end'=>($row['planned_end']?fmt_hhmm((string)$row['planned_end']):null),
    'actual_end'=>($row['actual_end']?fmt_hhmm((string)$row['actual_end']):null),
    'state'=> !empty($row['actual_end']) ? 'CLOSED' : (!empty($row['planned_end']) ? 'PLANNED' : 'OPEN')
  ]);
}

// ============================
// scan
// ============================
if ($action === 'scan') {
  $token = (string)($_POST['token'] ?? '');
  assert_qr_token_or_fail($token);

  $row = null;
  $q = "SELECT id, planned_end, actual_end, schedule_end FROM overtime_qr WHERE idempleado=? AND fecha=? LIMIT 1";
  if ($st = mysqli_prepare($con,$q)) {
    mysqli_stmt_bind_param($st,'is',$idempleado,$today);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);
    $row = $rs ? mysqli_fetch_assoc($rs) : null;
    mysqli_stmt_close($st);
  }

  if (!$row) {
    respond([
      'ok'=>true,
      'need_plan'=>true,
      'msg'=>'Indica la hora estimada hasta la que te quedarás hoy.',
      'today'=>$today,
      'schedule_end'=>fmt_hhmm($schedule_end)
    ]);
  }

  if (!empty($row['actual_end'])) {
    respond([
      'ok'=>true,
      'closed'=>true,
      'msg'=>'Ya existe un cierre de horas extra para hoy.',
      'today'=>$today,
      'schedule_end'=>fmt_hhmm((string)$row['schedule_end']),
      'actual_end'=>fmt_hhmm((string)$row['actual_end']),
      'planned_end'=>($row['planned_end']?fmt_hhmm((string)$row['planned_end']):null)
    ]);
  }

  if (empty($row['planned_end'])) {
    respond([
      'ok'=>true,
      'need_plan'=>true,
      'msg'=>'Completa tu estimación (hora fin) para programar la extra.',
      'today'=>$today,
      'schedule_end'=>fmt_hhmm((string)$row['schedule_end'])
    ]);
  }

  // Cerrar
  $now = date('H:i:s');

  if (minutes_between((string)$schedule_end, $now) <= 0) {
    respond([
      'ok'=>false,
      'msg'=>'Aún no ha pasado la hora oficial de salida (' . fmt_hhmm((string)$schedule_end) . ').'
    ]);
  }

  $mins = minutes_between((string)$schedule_end, $now);
  if ($mins < 0) $mins = 0;

  $hours = round_up_half_hours($mins);

  $upd = "UPDATE overtime_qr
          SET actual_end=?, closed_at=NOW(), overtime_minutes=?, updated_at=NOW()
          WHERE idempleado=? AND fecha=? LIMIT 1";
  if ($st = mysqli_prepare($con,$upd)) {
    mysqli_stmt_bind_param($st,'siis',$now,$mins,$idempleado,$today);
    mysqli_stmt_execute($st);
    mysqli_stmt_close($st);
  }

  $planned = (string)$row['planned_end'];
  $desc = "QR Extra | Salida oficial: ".fmt_hhmm((string)$schedule_end)
        ." | Plan: ".fmt_hhmm($planned)
        ." | Real: ".fmt_hhmm($now)
        ." | (Horas en 0.5)";

  $okIns = upsert_extras_day_overwrite($con, $idempleado, $today, (float)$hours, '1.5', $desc, 0);

  $last = asistencia_last_state($con, $idempleado, $today);
  if (empty($last) || ($last['estado'] ?? '') !== 'DONE') {
    asistencia_insert_mark($con, $idempleado, $gestor_id, 'EXTRA_CLOSE.'.$token, 'OUT_FINAL', 'DONE');
  }

  respond([
    'ok'=>true,
    'closed'=>true,
    'msg'=> $okIns ? '✅ Extra cerrada y jornada cerrada (Salida Final).' : 'Extra cerrada, pero no pude inyectar en extras_detalle.',
    'today'=>$today,
    'schedule_end'=>fmt_hhmm((string)$schedule_end),
    'planned_end'=>fmt_hhmm($planned),
    'actual_end'=>fmt_hhmm($now),
    'minutes'=>$mins,
    'hours'=>$hours,
    'redirect'=>'dashboard.php'
  ]);
}

// ============================
// save_plan
// ============================
if ($action === 'save_plan') {
  $token = (string)($_POST['token'] ?? '');
  assert_qr_token_or_fail($token);

  $planned = norm_time($_POST['planned_end'] ?? '');
  if (!$planned) respond(['ok'=>false,'msg'=>'Hora estimada inválida.']);

  if (minutes_between((string)$schedule_end, (string)$planned) <= 0) {
    respond(['ok'=>false,'msg'=>'La hora estimada debe ser mayor a la salida oficial (' . fmt_hhmm((string)$schedule_end) . ').']);
  }

  $ins = "INSERT INTO overtime_qr
          (idempleado, gestor_id, fecha, schedule_end, planned_end, planned_at, created_at, updated_at)
          VALUES (?,?,?,?,?,NOW(),NOW(),NOW())
          ON DUPLICATE KEY UPDATE
            schedule_end=VALUES(schedule_end),
            planned_end=VALUES(planned_end),
            planned_at=NOW(),
            actual_end=NULL,
            closed_at=NULL,
            overtime_minutes=0,
            updated_at=NOW()";
  if ($st = mysqli_prepare($con,$ins)) {
    mysqli_stmt_bind_param($st,'iisss',$idempleado,$gestor_id,$today,$schedule_end,$planned);
    mysqli_stmt_execute($st);
    mysqli_stmt_close($st);
    respond([
      'ok'=>true,
      'msg'=>'Horas extra programadas.',
      'today'=>$today,
      'schedule_end'=>fmt_hhmm((string)$schedule_end),
      'planned_end'=>fmt_hhmm((string)$planned)
    ]);
  }
  respond(['ok'=>false,'msg'=>'No se pudo guardar: '.mysqli_error($con)]);
}

// ============================
// delete_today
// ============================
if ($action === 'delete_today') {
  delete_extras_day($con, $idempleado, $today);

  $d="DELETE FROM overtime_qr WHERE idempleado=? AND fecha=? LIMIT 1";
  if($st=mysqli_prepare($con,$d)){
    mysqli_stmt_bind_param($st,'is',$idempleado,$today);
    @mysqli_stmt_execute($st);
    mysqli_stmt_close($st);
  }

  respond([
    'ok'=>true,
    'msg'=>'Se eliminó la programación/cierre de horas extra para hoy.',
    'today'=>$today
  ]);
}

// ============================
// clear_close
// ============================
if ($action === 'clear_close') {
  delete_extras_day($con, $idempleado, $today);

  $u="UPDATE overtime_qr
      SET actual_end=NULL, closed_at=NULL, overtime_minutes=0, updated_at=NOW()
      WHERE idempleado=? AND fecha=? LIMIT 1";
  if($st=mysqli_prepare($con,$u)){
    mysqli_stmt_bind_param($st,'is',$idempleado,$today);
    @mysqli_stmt_execute($st);
    mysqli_stmt_close($st);
  }

  respond([
    'ok'=>true,
    'msg'=>'Se reabrió la extra (se quitó el cierre).',
    'today'=>$today
  ]);
}

respond(['ok'=>false,'msg'=>'Acción no válida']);
