<?php
// rrhh_home/asistencia/calendario_api.php
// Eventos para FullCalendar: AUSENCIAS + RETRASOS + HORAS EXTRA (Planilla)
// + detalle exacto de entrada/salida/pausas desde ausencias_marcajes
// + explicación del porqué (comparación planilla vs marcajes vs horario oficial)
// PHP 8.1.33

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Costa_Rica');

header('Content-Type: application/json; charset=utf-8');

function json_out(array $a, int $code=200): void {
  http_response_code($code);
  echo json_encode($a, JSON_UNESCAPED_UNICODE);
  exit;
}

if (!isset($_SESSION['gestor_id']) || (int)$_SESSION['gestor_id'] <= 0) {
  json_out(['ok'=>false,'msg'=>'Sesión expirada.'], 401);
}

require_once __DIR__ . '/../dbcon.php';
mysqli_set_charset($con, 'utf8mb4');

$idempleado = (int)($_SESSION['idempleado'] ?? 0);
if ($idempleado <= 0) $idempleado = (int)($_SESSION['gestor_empleado_id'] ?? 0);
if ($idempleado <= 0) json_out(['ok'=>false,'msg'=>'No hay idempleado en sesión.'], 400);

function table_exists(mysqli $con, string $table): bool {
  $t = mysqli_real_escape_string($con, $table);
  $rs = mysqli_query($con, "SHOW TABLES LIKE '{$t}'");
  return ($rs && mysqli_num_rows($rs) > 0);
}

function safe_date(string $s): ?string {
  $s = trim($s);
  if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $s)) return null;
  return $s;
}

function fmt_h(float $h): string {
  $x = rtrim(rtrim(number_format($h, 2, '.', ''), '0'), '.');
  return $x === '' ? '0' : $x;
}

function minutes_between_hhmm(string $a, string $b): int {
  // a/b HH:MM
  $t1 = strtotime('2000-01-01 '.$a.':00');
  $t2 = strtotime('2000-01-01 '.$b.':00');
  if ($t1===false || $t2===false) return 0;
  return (int)floor(($t2-$t1)/60);
}

function minutes_between_dt(string $dtA, string $dtB): int {
  $t1 = strtotime($dtA);
  $t2 = strtotime($dtB);
  if ($t1===false || $t2===false) return 0;
  return (int)floor(($t2-$t1)/60);
}

$start = safe_date((string)($_GET['start'] ?? ''));
$end   = safe_date((string)($_GET['end'] ?? ''));

if (!$start || !$end) {
  $start = date('Y-m-01');
  $end   = date('Y-m-t');
}

/* ========================= Cargar horario oficial (work_schedule) =========================
   Esperamos columnas: day_of_week (1=Lun..7=Dom), active, start_time, end_time
   Si tu tabla usa otros nombres, dímelos y lo ajusto.
*/
$scheduleByDow = []; // 1..7 => ['start'=>'08:00','end'=>'16:30']
if (table_exists($con, 'work_schedule')) {
  $cols = [];
  $rs = mysqli_query($con, "SHOW COLUMNS FROM work_schedule");
  while ($rs && ($r = mysqli_fetch_assoc($rs))) $cols[] = (string)$r['Field'];

  $hasDow = in_array('day_of_week', $cols, true);
  $hasActive = in_array('active', $cols, true);

  $startCol = in_array('start_time', $cols, true) ? 'start_time' : (in_array('start', $cols, true) ? 'start' : null);
  $endCol   = in_array('end_time',   $cols, true) ? 'end_time'   : (in_array('end',   $cols, true) ? 'end'   : null);

  if ($hasDow && $hasActive && $startCol && $endCol) {
    $q = "SELECT day_of_week, active, $startCol AS st, $endCol AS en FROM work_schedule";
    $r2 = mysqli_query($con, $q);
    while ($r2 && ($row = mysqli_fetch_assoc($r2))) {
      $dow = (int)($row['day_of_week'] ?? 0);
      if ($dow < 1 || $dow > 7) continue;
      if ((int)($row['active'] ?? 0) !== 1) continue;
      $st = (string)($row['st'] ?? '');
      $en = (string)($row['en'] ?? '');
      if ($st !== '' && $en !== '') {
        $scheduleByDow[$dow] = ['start'=>substr($st,0,5), 'end'=>substr($en,0,5)];
      }
    }
  }
}

/* ========================= Cargar marcajes del rango (1 query) =========================
   Tabla: ausencias_marcajes
   Eventos: IN_INICIO, IN_REANUDA, OUT_PAUSE, OUT_FINAL
   Con esto calculamos:
   - primera entrada del día (first_in)
   - salida final del día (final_out)
   - pausas: pares OUT_PAUSE -> IN_REANUDA (sumatoria)
*/
$marksByDate = []; // 'YYYY-MM-DD' => ['first_in'=>..., 'final_out'=>..., 'pauses'=>[], 'pause_min'=>int]
if (table_exists($con, 'ausencias_marcajes')) {
  $q = "
    SELECT evento, DATE_FORMAT(fecha_hora,'%Y-%m-%d') AS d, DATE_FORMAT(fecha_hora,'%H:%i') AS hm,
           DATE_FORMAT(fecha_hora,'%Y-%m-%d %H:%i:%s') AS dt
    FROM ausencias_marcajes
    WHERE idempleado=?
      AND fecha_hora >= CONCAT(?, ' 00:00:00')
      AND fecha_hora <= CONCAT(?, ' 23:59:59')
      AND UPPER(evento) IN ('IN_INICIO','IN_REANUDA','OUT_PAUSE','OUT_FINAL')
    ORDER BY fecha_hora ASC
  ";
  if ($st = mysqli_prepare($con, $q)) {
    mysqli_stmt_bind_param($st, 'iss', $idempleado, $start, $end);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);

    while ($rs && ($r = mysqli_fetch_assoc($rs))) {
      $d = (string)($r['d'] ?? '');
      $ev = strtoupper((string)($r['evento'] ?? ''));
      $hm = (string)($r['hm'] ?? '');
      $dt = (string)($r['dt'] ?? '');

      if ($d === '' || $hm === '' || $dt === '') continue;
      if (!isset($marksByDate[$d])) {
        $marksByDate[$d] = [
          'first_in'=>null,
          'final_out'=>null,
          'pauses'=>[],           // cada pause = ['out'=>'HH:MM','in'=>'HH:MM', 'min'=>x]
          'pause_min'=>0,
          '_open_pause_dt'=>null, // helper interno
          '_open_pause_hm'=>null,
        ];
      }
      $ref = &$marksByDate[$d];

      if ($ev === 'IN_INICIO' || $ev === 'IN_REANUDA') {
        // primera entrada
        if ($ref['first_in'] === null) $ref['first_in'] = $hm;

        // si venimos de pausa abierta, cerrarla
        if ($ref['_open_pause_dt']) {
          $mins = minutes_between_dt((string)$ref['_open_pause_dt'], $dt);
          if ($mins < 0) $mins = 0;
          $ref['pauses'][] = [
            'out' => (string)$ref['_open_pause_hm'],
            'in'  => $hm,
            'min' => $mins
          ];
          $ref['pause_min'] += $mins;
          $ref['_open_pause_dt'] = null;
          $ref['_open_pause_hm'] = null;
        }
      }

      if ($ev === 'OUT_PAUSE') {
        // abre pausa (si ya había una abierta, la sobreescribimos)
        $ref['_open_pause_dt'] = $dt;
        $ref['_open_pause_hm'] = $hm;
      }

      if ($ev === 'OUT_FINAL') {
        $ref['final_out'] = $hm;
        // si quedó pausa abierta y luego OUT_FINAL, cerramos pausa hasta OUT_FINAL (para no perder tiempo)
        if ($ref['_open_pause_dt']) {
          $mins = minutes_between_dt((string)$ref['_open_pause_dt'], $dt);
          if ($mins < 0) $mins = 0;
          $ref['pauses'][] = [
            'out' => (string)$ref['_open_pause_hm'],
            'in'  => $hm,
            'min' => $mins
          ];
          $ref['pause_min'] += $mins;
          $ref['_open_pause_dt'] = null;
          $ref['_open_pause_hm'] = null;
        }
      }

      unset($ref);
    }

    mysqli_stmt_close($st);
  }
}

// limpiar helpers internos
foreach ($marksByDate as $d => $arr) {
  unset($marksByDate[$d]['_open_pause_dt'], $marksByDate[$d]['_open_pause_hm']);
}

/* ========================= Construcción de eventos + totales ========================= */
$events = [];
$tot_aus = 0.0;
$tot_ret = 0.0;
$tot_ot  = 0.0;

function build_day_context(string $date, array $scheduleByDow, array $marksByDate): array {
  $dow = (int)date('N', strtotime($date));
  $sch = $scheduleByDow[$dow] ?? null;

  $mk = $marksByDate[$date] ?? null;

  $first_in = $mk['first_in'] ?? null;
  $final_out = $mk['final_out'] ?? null;
  $pause_min = (int)($mk['pause_min'] ?? 0);
  $pauses = $mk['pauses'] ?? [];

  $schedule_start = $sch['start'] ?? null;
  $schedule_end   = $sch['end'] ?? null;

  // retraso estimado por marcaje: first_in - schedule_start
  $late_min = 0;
  if ($first_in && $schedule_start) {
    $m = minutes_between_hhmm($schedule_start, $first_in);
    if ($m > 0) $late_min = $m;
  }

  // OT estimada por marcaje: final_out - schedule_end
  $ot_min = 0;
  if ($final_out && $schedule_end) {
    $m = minutes_between_hhmm($schedule_end, $final_out);
    if ($m > 0) $ot_min = $m;
  }

  return [
    'schedule_start'=>$schedule_start,
    'schedule_end'=>$schedule_end,
    'first_in'=>$first_in,
    'final_out'=>$final_out,
    'pauses'=>$pauses,
    'pause_min'=>$pause_min,
    'late_min'=>$late_min,
    'ot_min'=>$ot_min,
  ];
}

/* ========================= AUSENCIAS / RETRASOS (Planilla) ========================= */
if (table_exists($con, 'ausencias_mes') && table_exists($con, 'ausencias_detalle')) {

  $sql = "
    SELECT d.id, d.fecha, UPPER(d.tipo) AS tipo, d.horas, d.descripcion
    FROM ausencias_detalle d
    INNER JOIN ausencias_mes m ON m.id = d.id_ausencias_mes
    WHERE m.idempleado=?
      AND d.fecha BETWEEN ? AND ?
    ORDER BY d.fecha ASC, d.id ASC
  ";
  if ($st = mysqli_prepare($con, $sql)) {
    mysqli_stmt_bind_param($st, 'iss', $idempleado, $start, $end);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);

    while ($rs && ($r = mysqli_fetch_assoc($rs))) {
      $id = (int)($r['id'] ?? 0);
      $d  = (string)($r['fecha'] ?? '');
      $tipo = (string)($r['tipo'] ?? 'AUSENCIA');
      $horas = (float)($r['horas'] ?? 0);
      $desc  = trim((string)($r['descripcion'] ?? ''));

      if ($d === '') continue;
      if ($tipo !== 'AUSENCIA' && $tipo !== 'RETRASO') $tipo = 'AUSENCIA';

      if ($tipo === 'RETRASO') $tot_ret += $horas;
      else $tot_aus += $horas;

      $ctx = build_day_context($d, $scheduleByDow, $marksByDate);

      $hTxt = fmt_h($horas);
      $title = ($tipo === 'RETRASO') ? "RETRASO {$hTxt}h" : "AUSENCIA {$hTxt}h";

      // explicación:
      $detailLines = [];
      $detailLines[] = "<b>Origen (Planilla):</b> ausencias_mes + ausencias_detalle";
      $detailLines[] = "<b>Horas registradas (Planilla):</b> {$hTxt}h";
      $detailLines[] = "<hr style='margin:8px 0'>";

      if ($ctx['schedule_start'] && $ctx['schedule_end']) {
        $detailLines[] = "<b>Horario oficial:</b> {$ctx['schedule_start']} - {$ctx['schedule_end']}";
      } else {
        $detailLines[] = "<b>Horario oficial:</b> (no disponible en work_schedule)";
      }

      if ($ctx['first_in'] || $ctx['final_out']) {
        $detailLines[] = "<b>Marcajes reales:</b> Entrada: ".($ctx['first_in'] ?? '—')." | Salida final: ".($ctx['final_out'] ?? '—');
      } else {
        $detailLines[] = "<b>Marcajes reales:</b> No hay marcajes del día en ausencias_marcajes";
      }

      // tipo-specific
      if ($tipo === 'RETRASO') {
        $lateH = $ctx['late_min'] > 0 ? fmt_h($ctx['late_min']/60) : '0';
        $detailLines[] = "<b>Retraso estimado (por marcaje vs horario):</b> {$lateH}h";
        $detailLines[] = "<small class='text-muted'>Cálculo: Entrada real - Hora inicio oficial (si entrada &gt; inicio).</small>";
      } else {
        $pauseH = $ctx['pause_min'] > 0 ? fmt_h($ctx['pause_min']/60) : '0';
        $detailLines[] = "<b>Pausas estimadas (por marcajes):</b> {$pauseH}h";
        if (!empty($ctx['pauses'])) {
          $pItems = [];
          foreach ($ctx['pauses'] as $p) {
            $pItems[] = "{$p['out']} → {$p['in']} (".fmt_h(((int)$p['min'])/60)."h)";
          }
          $detailLines[] = "<b>Detalle pausas:</b><br>".implode("<br>", array_map('htmlspecialchars', $pItems));
        }
        $detailLines[] = "<small class='text-muted'>Cálculo: suma de (OUT_PAUSE → IN_REANUDA) y/o (OUT_PAUSE → OUT_FINAL).</small>";
      }

      if ($desc !== '') {
        $detailLines[] = "<hr style='margin:8px 0'>";
        $detailLines[] = "<b>Descripción:</b> ".htmlspecialchars($desc, ENT_QUOTES, 'UTF-8');
      }

      $events[] = [
        'id' => ($tipo === 'RETRASO' ? 'R-' : 'A-') . $id,
        'title' => $title,
        'start' => $d,
        'allDay' => true,
        'extendedProps' => [
          'tipo' => $tipo,
          'detalle' => implode('<br>', $detailLines),
          'ctx' => $ctx, // por si luego quieres mostrar más en UI
        ]
      ];
    }

    mysqli_stmt_close($st);
  }
}

/* ========================= HORAS EXTRA (Planilla) ========================= */
if (table_exists($con, 'extras_mes') && table_exists($con, 'extras_detalle')) {

  $sql = "
    SELECT d.id, d.fecha, d.horas, d.factor, d.descripcion
    FROM extras_detalle d
    INNER JOIN extras_mes m ON m.id = d.id_extras_mes
    WHERE m.idempleado=?
      AND d.fecha BETWEEN ? AND ?
    ORDER BY d.fecha ASC, d.id ASC
  ";
  if ($st = mysqli_prepare($con, $sql)) {
    mysqli_stmt_bind_param($st, 'iss', $idempleado, $start, $end);
    mysqli_stmt_execute($st);
    $rs = mysqli_stmt_get_result($st);

    while ($rs && ($r = mysqli_fetch_assoc($rs))) {
      $id = (int)($r['id'] ?? 0);
      $d  = (string)($r['fecha'] ?? '');
      $horas = (float)($r['horas'] ?? 0);
      $factor = (string)($r['factor'] ?? '1.5');
      $desc  = trim((string)($r['descripcion'] ?? ''));

      if ($d === '') continue;

      $tot_ot += $horas;

      $ctx = build_day_context($d, $scheduleByDow, $marksByDate);

      $hTxt = fmt_h($horas);
      $title = "OT {$hTxt}h x{$factor}";

      $detailLines = [];
      $detailLines[] = "<b>Origen (Planilla):</b> extras_mes + extras_detalle";
      $detailLines[] = "<b>Horas registradas (Planilla):</b> {$hTxt}h";
      $detailLines[] = "<b>Factor:</b> {$factor}";
      $detailLines[] = "<hr style='margin:8px 0'>";

      if ($ctx['schedule_start'] && $ctx['schedule_end']) {
        $detailLines[] = "<b>Horario oficial:</b> {$ctx['schedule_start']} - {$ctx['schedule_end']}";
      } else {
        $detailLines[] = "<b>Horario oficial:</b> (no disponible en work_schedule)";
      }

      if ($ctx['first_in'] || $ctx['final_out']) {
        $detailLines[] = "<b>Marcajes reales:</b> Entrada: ".($ctx['first_in'] ?? '—')." | Salida final: ".($ctx['final_out'] ?? '—');
      } else {
        $detailLines[] = "<b>Marcajes reales:</b> No hay marcajes del día en ausencias_marcajes";
      }

      $otEstH = $ctx['ot_min'] > 0 ? fmt_h($ctx['ot_min']/60) : '0';
      $detailLines[] = "<b>OT estimada (por marcaje vs salida oficial):</b> {$otEstH}h";
      $detailLines[] = "<small class='text-muted'>Cálculo: Salida final real - Hora fin oficial (si salida &gt; fin).</small>";

      // Nota si difiere mucho
      $diff = abs(($ctx['ot_min']/60) - $horas);
      if ($ctx['final_out'] && $ctx['schedule_end'] && $diff >= 0.25) {
        $detailLines[] = "<hr style='margin:8px 0'>";
        $detailLines[] = "<b>Nota:</b> La OT de Planilla ({$hTxt}h) difiere de la OT estimada por marcaje ({$otEstH}h).";
        $detailLines[] = "<small class='text-muted'>Esto es normal si OT fue registrada manualmente o si hubo reglas especiales.</small>";
      }

      if ($desc !== '') {
        $detailLines[] = "<hr style='margin:8px 0'>";
        $detailLines[] = "<b>Descripción:</b> ".htmlspecialchars($desc, ENT_QUOTES, 'UTF-8');
      }

      $events[] = [
        'id' => 'OT-' . $id,
        'title' => $title,
        'start' => $d,
        'allDay' => true,
        'extendedProps' => [
          'tipo' => 'OT',
          'detalle' => implode('<br>', $detailLines),
          'ctx' => $ctx,
        ]
      ];
    }

    mysqli_stmt_close($st);
  }
}

json_out([
  'ok' => true,
  'start' => $start,
  'end' => $end,
  'idempleado' => $idempleado,
  'totals' => [
    'ausencias_h' => (float)number_format($tot_aus, 2, '.', ''),
    'retrasos_h'  => (float)number_format($tot_ret, 2, '.', ''),
    'ot_h'        => (float)number_format($tot_ot, 2, '.', ''),
  ],
  'events' => $events
]);
