<?php
/**
 * dbcon.php — compatible con sistemas viejos (mysqli, variable $con)
 * - Soporta variables de entorno (DB_HOST, DB_USER, DB_PASS, DB_NAME, DB_PORT, DB_SOCKET)
 * - Fuerza UTF-8 (utf8mb4) en la conexión (SET NAMES + collation de sesión)
 * - No hace echo de errores en producción (no rompe JSON/AJAX). Loggea a error_log.
 * - Mantiene $con como instancia mysqli para no romper includes existentes.
 * - Ajusta zona horaria a America/Costa_Rica.
 */

declare(strict_types=1);

/* ========= Config ========= */
date_default_timezone_set('America/Costa_Rica'); // antes estaba Lima; cambiamos a CR

// Si pones APP_DEBUG=1 en el entorno, verás errores de conexión en pantalla.
$APP_DEBUG = (getenv('APP_DEBUG') === '1');

/* ========= Credenciales =========
 * Usa variables de entorno si existen; si no, cae en tus valores actuales.
 */
$DB_HOST   = getenv('DB_HOST')   ?: 'localhost';
$DB_USER   = getenv('DB_USER')   ?: 'jportales_pos_rrhhbritanico';
$DB_PASS   = getenv('DB_PASS')   ?: 'portales123';
$DB_NAME   = getenv('DB_NAME')   ?: 'jportales_pos_rrhhbritanico';
$DB_PORT   = (int)(getenv('DB_PORT') ?: 3306);
$DB_SOCKET = getenv('DB_SOCKET') ?: null; // e.g. '/var/lib/mysql/mysql.sock'

/* ========= Conexión segura ========= */
mysqli_report(MYSQLI_REPORT_OFF); // evita warnings que rompan la salida

$con = mysqli_init();
if ($con === false) {
  $msg = 'mysqli_init() failed';
  error_log($msg);
  if ($APP_DEBUG) { http_response_code(500); echo $msg; }
  exit; // sin conexión no podemos continuar
}

// Ajusta timeout de conexión (útil en VPS/containers)
@mysqli_options($con, MYSQLI_OPT_CONNECT_TIMEOUT, 10);

// Conexión: host/usuario/pass/db/port/socket
$connected = @mysqli_real_connect(
  $con,
  $DB_HOST,
  $DB_USER,
  $DB_PASS,
  $DB_NAME,
  $DB_PORT ?: null,
  $DB_SOCKET ?: null,
  0
);

if (!$connected) {
  $msg = 'Failed to connect to MySQL: ' . mysqli_connect_error();
  error_log($msg);
  if ($APP_DEBUG) { http_response_code(500); echo $msg; }
  exit;
}

/* ========= Charset & collation de sesión =========
 * Forzamos utf8mb4 para que ñ/acentos funcionen siempre.
 */
if (!@mysqli_set_charset($con, 'utf8mb4')) {
  // Si falla, intenta por SET NAMES
  @mysqli_query($con, "SET NAMES utf8mb4");
}
@mysqli_query($con, "SET collation_connection = 'utf8mb4_unicode_ci'");

// (Opcional) fuerza SQL_MODE sólo si tú lo quieres. Lo dejo comentado para no “romper” lógicas viejas.
// @mysqli_query($con, "SET SESSION sql_mode=''");

/* ========= Zona horaria MySQL (sesión) =========
 * Intentamos usar el nombre de zona. Si el servidor MySQL no tiene tzdata cargado,
 * caemos al offset estándar de CR (-06:00).
 */
if (!@mysqli_query($con, "SET time_zone = 'America/Costa_Rica'")) {
  @mysqli_query($con, "SET time_zone = '-06:00'");
}

/* ========= Helpers opcionales ========= */
/** Escapado HTML seguro (para endpoints que imprimen HTML) */
if (!function_exists('esc')) {
  function esc($v): string { return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
}

/** Comprueba que la conexión sigue viva (reconnect si hace falta) */
if (!function_exists('db_ping')) {
  function db_ping(mysqli $con): bool {
    if (@mysqli_ping($con)) return true;
    // En sistemas legacy simplemente devolvemos false; reabrir aquí rompería credenciales locales.
    return false;
  }
}
