<?php
// add_proveedor.php — Alta de proveedor con UTF-8 y consultas preparadas
declare(strict_types=1);

session_start();
require_once __DIR__ . '/dbcon.php'; // ya fuerza utf8mb4 en la sesión MySQL

// -----------------------------------------------------------------------------
// Config / helpers
// -----------------------------------------------------------------------------

// Normaliza cualquier string a UTF-8 válido y quita controles no imprimibles.
function norm(string $s): string {
  if (!mb_detect_encoding($s, 'UTF-8', true)) {
    $s = @mb_convert_encoding($s, 'UTF-8', 'Windows-1252,ISO-8859-1,UTF-8') ?: $s;
  }
  return preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/u', '', $s);
}

// Inserta proveedor con consulta preparada (evita errores por comillas/apóstrofos).
function insertar_proveedor(mysqli $con, string $nombre, string $representante, string $posicion, string $telefono, string $email): bool {
  $sql = "INSERT INTO proveedores_colegio (nombre, representante, posicion, telefono, email)
          VALUES (?, ?, ?, ?, ?)";
  if (!$stmt = $con->prepare($sql)) {
    error_log("prepare failed: ".$con->error);
    return false;
  }
  $stmt->bind_param("sssss", $nombre, $representante, $posicion, $telefono, $email);
  $ok = $stmt->execute();
  if (!$ok) error_log("execute failed: ".$stmt->error);
  $stmt->close();
  return $ok;
}

// -----------------------------------------------------------------------------
// Lee entrada
// -----------------------------------------------------------------------------

$idsucursal = isset($_SESSION['idsucursal']) ? (string)$_SESSION['idsucursal'] : ''; // por compatibilidad

$idempleado   = $_POST["idempleado"]   ?? '';
$fecha        = $_POST["fecha"]        ?? '';
$area         = $_POST["area"]         ?? '';

$resp1        = $_POST["responsabilidad"]  ?? '';
$resp2        = $_POST["responsabilidad2"] ?? '';
$responsabilidad = norm($resp1) . ' | ' . norm($resp2); // (no se usa abajo, pero lo dejamos por compat)

$nombre        = norm($_POST["nombre"]         ?? '');
$representante = norm($_POST["representante"]  ?? '');
$posicion      = norm($_POST["posicion"]       ?? '');
$telefono      = norm($_POST["telefono"]       ?? '');
$email         = norm($_POST["email"]          ?? '');

// -----------------------------------------------------------------------------
// Subida de imagen (opcional) — No se guarda en la tabla proveedores_colegio,
// pero lo mantenemos por compatibilidad de UI si ya tienes el input "imagen".
// -----------------------------------------------------------------------------

$target_dir = __DIR__ . "/Files/Compras/";        // ruta absoluta para mover
$public_dir = "Files/Compras/";                   // ruta pública a guardar/usar en HTML

$imagen_ruta_publica = "";                        // por si quieres guardarla luego en otra tabla

if (!empty($_FILES['imagen']['name']) && is_uploaded_file($_FILES['imagen']['tmp_name'])) {
  // Asegura el directorio
  if (!is_dir($target_dir)) @mkdir($target_dir, 0775, true);

  // Sanea el nombre de archivo
  $baseName = basename($_FILES['imagen']['name']);
  $baseName = preg_replace('/[^A-Za-z0-9_\-.]/', '_', $baseName);

  $target_file_abs = $target_dir . $baseName;
  $target_file_rel = $public_dir . $baseName;

  // Límite de tamaño (50MB como en tu código)
  if ((int)$_FILES["imagen"]["size"] > 50 * 1024 * 1024) {
    die("Archivo demasiado grande (máx 50MB).");
  }

  // Mueve el archivo
  if (!@move_uploaded_file($_FILES["imagen"]["tmp_name"], $target_file_abs)) {
    die("No se pudo subir el archivo de imagen.");
  }

  $imagen_ruta_publica = $target_file_rel; // si luego lo necesitas
}

// -----------------------------------------------------------------------------
// Inserta proveedor (USANDO PREPARED STATEMENT) — evita errores por apóstrofes
// -----------------------------------------------------------------------------

if (!insertar_proveedor($con, $nombre, $representante, $posicion, $telefono, $email)) {
  // Si quieres mostrar el error real en desarrollo: activa APP_DEBUG=1 en el entorno.
  die("No se pudo guardar el proveedor.");
}

// -----------------------------------------------------------------------------
// Redirección final
// -----------------------------------------------------------------------------

// Evita seguir ejecutando cosas innecesarias
echo "<script>document.location='bandeja_proveedor.php'</script>";
exit;
