<?php
// add_ingreso_vacaciones_extra.php
// PHP 8.1.33 — Inserta Vacaciones EXTRA enviando TODOS los datos,
// convierte adjuntos (imagen/pdf) a UN PDF con recortes y guarda por Mes-Año.

// ===== Debug (0 en producción) =====
ini_set('display_errors','1');
ini_set('display_startup_errors','1');
error_reporting(E_ALL);

// ===== Recursos =====
ini_set('memory_limit','1024M');
set_time_limit(120);

if (session_status() !== PHP_SESSION_ACTIVE) session_start();
date_default_timezone_set('America/Costa_Rica');

require_once __DIR__.'/dbcon.php';
if (isset($con) && $con instanceof mysqli) { @mysqli_set_charset($con,'utf8mb4'); }

$idusuario  = $_SESSION['idusuario']  ?? null;
$idsucursal = $_SESSION['idsucursal'] ?? null;

/* ==================== Helpers de fechas ==================== */
/**
 * Devuelve un literal SQL NO NULO para fechas:
 *  - '' → usa $fallback
 *  - dd/mm/yyyy → STR_TO_DATE(...)
 *  - yyyy-mm-dd → 'yyyy-mm-dd'
 *  - cualquier otro → usa $fallback
 */
function date_sql_not_null(mysqli $con, $s, $fallback){
  $s  = trim((string)$s);
  $fb = trim((string)$fallback);

  if ($fb === '' || $fb === null) $fb = date('Y-m-d');

  // Normaliza fallback
  if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $fb)) {
    $fb_sql = "STR_TO_DATE('".mysqli_real_escape_string($con,$fb)."','%d/%m/%Y')";
  } elseif (preg_match('/^\d{4}-\d{2}-\d{2}$/', $fb)) {
    $fb_sql = "'".mysqli_real_escape_string($con,$fb)."'";
  } else {
    $today  = date('Y-m-d');
    $fb_sql = "'".mysqli_real_escape_string($con,$today)."'";
  }

  if ($s === '') return $fb_sql;

  if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $s)) {
    return "STR_TO_DATE('".mysqli_real_escape_string($con,$s)."','%d/%m/%Y')";
  }
  if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $s)) {
    return "'".mysqli_real_escape_string($con,$s)."'";
  }
  return $fb_sql;
}

/* ==================== Helpers de carpeta Mes-Año ==================== */
function month_name_es($m){
  static $N=[1=>'Enero','Febrero','Marzo','Abril','Mayo','Junio','Julio','Agosto','Septiembre','Octubre','Noviembre','Diciembre'];
  $mi=(int)$m; return $N[$mi] ?? 'Desconocido';
}
function month_folder_from_date($datestr){
  if (!$datestr || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $datestr)) $datestr=date('Y-m-d');
  [$Y,$m]=explode('-', substr($datestr,0,7));
  return month_name_es((int)$m).'-'.$Y; // p.ej. Octubre-2025
}

/* ==================== Helpers de imágenes → PDF ==================== */
function _parse_crop_list($json){ $d=json_decode($json,true); return is_array($d)?$d:[]; }
function _clampv($v,$min,$max){ return max($min, min($max, (int)$v)); }

if (class_exists('Imagick')) {
  try {
    $rl = new Imagick();
    $rl->setResourceLimit(Imagick::RESOURCETYPE_MEMORY, 256);
    $rl->setResourceLimit(Imagick::RESOURCETYPE_MAP, 256);
    $rl->clear(); $rl->destroy();
  } catch(Exception $e){}
}

function _imagick_add_image_page($doc, $tmp, $crop=null){
  $img = new Imagick();
  $img->readImage($tmp);

  if (is_array($crop) && isset($crop['rotate']) && abs((float)$crop['rotate'])>0.01) {
    $img->rotateImage(new ImagickPixel('white'), (float)$crop['rotate']);
  }
  if (is_array($crop) && !empty($crop['width']) && !empty($crop['height'])) {
    $iw=$img->getImageWidth(); $ih=$img->getImageHeight();
    $x=_clampv($crop['x']??0,0,$iw-1);
    $y=_clampv($crop['y']??0,0,$ih-1);
    $cw=_clampv($crop['width'],1,$iw);
    $ch=_clampv($crop['height'],1,$ih);
    if ($x+$cw>$iw) $cw=$iw-$x; if ($y+$ch>$ih) $ch=$ih-$y;
    if ($cw>1 && $ch>1){ $img->cropImage($cw,$ch,$x,$y); $img->setImagePage(0,0,0,0); }
  }

  $MAX_PX=40_000_000;
  $iw=$img->getImageWidth(); $ih=$img->getImageHeight();
  if (($iw*$ih)>$MAX_PX) { $img->sampleImage(3000,3000,true); }

  if (method_exists($img,'setImageAlphaChannel')) $img->setImageAlphaChannel(Imagick::ALPHACHANNEL_REMOVE);
  $img->setImageColorspace(Imagick::COLORSPACE_SRGB);

  $maxW=2500; $maxH=3500;
  $iw=$img->getImageWidth(); $ih=$img->getImageHeight();
  if ($iw>$maxW || $ih>$maxH) $img->thumbnailImage($maxW,$maxH,true,true);

  $img->setImageFormat('pdf');
  $img->setImageCompression(Imagick::COMPRESSION_JPEG);
  $img->setImageCompressionQuality(80);
  $img->setImageUnits(Imagick::RESOLUTION_PIXELSPERINCH);
  $img->setImageResolution(150,150);

  $doc->addImage($img);
  $img->clear(); $img->destroy();
  return true;
}

function _gd_image_to_jpeg_bytes($tmp, $ext, $crop=null){
  $info=@getimagesize($tmp); if(!$info) return null;
  $wI=(int)$info[0]; $hI=(int)$info[1];
  if (($wI*$hI)>40_000_000) return null;

  switch(strtolower($ext)){
    case 'jpg':case 'jpeg': $im=@imagecreatefromjpeg($tmp); break;
    case 'png': $im=@imagecreatefrompng($tmp); break;
    case 'gif': $im=@imagecreatefromgif($tmp); break;
    case 'webp': $im=@imagecreatefromwebp($tmp); break;
    default: return null;
  }
  if(!$im) return null;

  if (is_array($crop) && isset($crop['rotate']) && abs((float)$crop['rotate'])>0.01) {
    $bg=imagecolorallocate($im,255,255,255);
    $rot=imagerotate($im, -(float)$crop['rotate'], $bg);
    if($rot){ imagedestroy($im); $im=$rot; }
  }

  $w=imagesx($im); $h=imagesy($im);
  if (is_array($crop) && !empty($crop['width']) && !empty($crop['height'])){
    $x=_clampv($crop['x']??0,0,$w-1);
    $y=_clampv($crop['y']??0,0,$h-1);
    $cw=_clampv($crop['width'],1,$w);
    $ch=_clampv($crop['height'],1,$h);
    if ($x+$cw>$w) $cw=$w-$x; if ($y+$ch>$h) $ch=$h-$y;
    if ($cw>1 && $ch>1){
      $dst=imagecreatetruecolor($cw,$ch);
      imagecopyresampled($dst,$im,0,0,$x,$y,$cw,$ch,$cw,$ch);
      imageresolution($dst,150,150);
      imagedestroy($im); $im=$dst; $w=$cw; $h=$ch;
    }
  }

  $maxW=2500; $maxH=3500;
  if ($w>$maxW || $h>$maxH){
    $ratio=min($maxW/$w,$maxH/$h);
    $nw=(int)round($w*$ratio); $nh=(int)round($h*$ratio);
    $res=imagecreatetruecolor($nw,$nh);
    imagecopyresampled($res,$im,0,0,0,0,$nw,$nh,$w,$h);
    imageresolution($res,150,150);
    imagedestroy($im); $im=$res; $w=$nw; $h=$nh;
  }

  ob_start(); imagejpeg($im,null,80); $data=ob_get_clean();
  imagedestroy($im);
  if(function_exists('gc_collect_cycles')) gc_collect_cycles();
  return ['data'=>$data,'w'=>$w,'h'=>$h];
}

function _build_multi_pdf_from_jpegs($pages,$targetPdf){
  $unit=72; $a4w=8.27*$unit; $a4h=11.69*$unit; $dpi=150.0;
  $pdf="%PDF-1.3\n"; $ofs=[]; $objNum=1;
  $ofs[] = strlen($pdf);
  $pdf .= ($objNum)." 0 obj<< /Type /Catalog /Pages 2 0 R >>endobj\n"; $catalog=$objNum; $objNum++;
  $pagesObj=$objNum; $objNum++; $kids=[];
  foreach($pages as $pg){
    $w=$pg['w']; $h=$pg['h']; $imgData=$pg['data'];
    $imgWpt=($w/$dpi)*72.0; $imgHpt=($h/$dpi)*72.0;
    $scale=min(($a4w*0.92)/$imgWpt, ($a4h*0.92)/$imgHpt); if($scale<=0)$scale=1;
    $dw=$imgWpt*$scale; $dh=$imgHpt*$scale; $x=($a4w-$dw)/2; $y=($a4h-$dh)/2;

    $imgObj=$objNum++; $ofs[] = strlen($pdf);
    $pdf.=$imgObj." 0 obj<< /Type /XObject /Subtype /Image /Width ".intval($w)." /Height ".intval($h)." /ColorSpace /DeviceRGB /BitsPerComponent 8 /Filter /DCTDecode /Length ".strlen($imgData)." >>stream\n".$imgData."\nendstream\nendobj\n";

    $stream="q\n".sprintf('%.2f 0 0 %.2f %.2f %.2f cm',$dw,$dh,$x,$y)."\n/Im0 Do\nQ\n";
    $cntObj=$objNum++; $ofs[] = strlen($pdf);
    $pdf.=$cntObj." 0 obj<< /Length ".strlen($stream)." >>stream\n".$stream."endstream\nendobj\n";

    $pageObj=$objNum++; $ofs[] = strlen($pdf);
    $pdf.=$pageObj." 0 obj<< /Type /Page /Parent ".$pagesObj." 0 R /MediaBox [0 0 ".sprintf('%.2f %.2f',$a4w,$a4h)."] /Resources << /XObject << /Im0 ".$imgObj." 0 R >> /ProcSet [/PDF /ImageC] >> /Contents ".$cntObj." 0 R >>endobj\n";
    $kids[]=$pageObj." 0 R";
  }
  $ofs[] = strlen($pdf);
  $pdf.=$pagesObj." 0 obj<< /Type /Pages /Count ".count($kids)." /Kids [ ".implode(' ',$kids)." ] >>endobj\n";
  $xrefPos=strlen($pdf);
  $pdf.="xref\n0 ".($objNum)."\n0000000000 65535 f \n";
  foreach($ofs as $o) $pdf.=sprintf("%010d 00000 n \n",$o);
  $pdf.="trailer<< /Size ".($objNum)." /Root ".$catalog." 0 R >>\nstartxref\n".$xrefPos."\n%%EOF";
  return (bool)file_put_contents($targetPdf,$pdf);
}

/* ==================== 1) Recibir TODOS los datos del formulario ==================== */
$all_post       = $_POST; // Trazabilidad completa
$crop_json_list = $_POST['crop_json_list'] ?? '[]';
$feriados_json  = $_POST['feriados_json']  ?? '[]';

$empleado        = $_POST['empleado']         ?? '';   // id select
$idempleado      = $_POST['idempleado']       ?? $empleado;
$fecha_documento = $_POST['fecha_documento']  ?? date('Y-m-d');
$desde           = $_POST['desde']            ?? '';
$hasta           = $_POST['hasta']            ?? '';
$motivo          = $_POST['motivo']           ?? '';
$cantidad_dias   = $_POST['cantidad_dias']    ?? 0;

/* ==================== 2) Carpeta por Mes-Año (según fecha_documento o hoy) ==================== */
$BASE_DIR   = "Files/Ingreso_vacaciones_extra/";
$subFolder  = month_folder_from_date(preg_match('/^\d{4}-\d{2}-\d{2}$/',$fecha_documento)?$fecha_documento:date('Y-m-d'));
$TARGET_DIR = rtrim($BASE_DIR,'/').'/'.$subFolder.'/';
if (!is_dir($TARGET_DIR)) { @mkdir($TARGET_DIR, 0775, true); }

/* ==================== 3) Construir PDF de adjuntos ==================== */
function construirPDFAdjunto($target_dir, $fieldName='imagen1', $crop_json_list='[]'){
  if (!isset($_FILES[$fieldName])) return '';
  $isMultiple = is_array($_FILES[$fieldName]['name']);
  $names = $isMultiple ? $_FILES[$fieldName]['name'] : [ $_FILES[$fieldName]['name'] ];
  $tmps  = $isMultiple ? $_FILES[$fieldName]['tmp_name'] : [ $_FILES[$fieldName]['tmp_name'] ];
  $sizes = $isMultiple ? $_FILES[$fieldName]['size'] : [ $_FILES[$fieldName]['size'] ];
  $crops = _parse_crop_list($crop_json_list);

  $hasAny=false;
  foreach($names as $i=>$n){
    if(!empty($n) && (int)$sizes[$i]>0 && is_uploaded_file($tmps[$i])) { $hasAny=true; break; }
  }
  if(!$hasAny) return '';

  if(!is_dir($target_dir)) @mkdir($target_dir,0775,true);
  $safePdf   = 'vacextra_'.date('Ymd_His').'_'.mt_rand(1000,9999).'.pdf';
  $targetPdf = rtrim($target_dir,'/').'/'.$safePdf;

  $useImagick = class_exists('Imagick');
  if ($useImagick){
    $doc=new Imagick(); $added=0;
    foreach($names as $i=>$origName){
      $tmp=$tmps[$i]; $sz=(int)$sizes[$i];
      if($sz<=0 || !is_uploaded_file($tmp)) continue;
      $ext=strtolower(pathinfo($origName, PATHINFO_EXTENSION));
      $crop=(isset($crops[$i]) && is_array($crops[$i])) ? $crops[$i] : null;

      if ($ext==='pdf'){
        try{ $doc->readImage($tmp); $added++; }catch(Exception $e){}
      } elseif (in_array($ext,['jpg','jpeg','png','gif','webp'])){
        if (_imagick_add_image_page($doc,$tmp,$crop)) $added++;
      }
      if (function_exists('gc_collect_cycles')) gc_collect_cycles();
    }
    if($added>0){
      $doc->setImageFormat('pdf');
      foreach($doc as $p){
        $p->setImageCompression(Imagick::COMPRESSION_JPEG);
        $p->setImageCompressionQuality(80);
        $p->setImageUnits(Imagick::RESOLUTION_PIXELSPERINCH);
        $p->setImageResolution(150,150);
      }
      $doc->writeImages($targetPdf,true);
      $doc->clear(); $doc->destroy();
      return file_exists($targetPdf) ? $targetPdf : '';
    }
    $doc->clear(); $doc->destroy();
    return '';
  }

  // Sin Imagick
  $pdfCount=0; $firstPdf=-1;
  foreach($names as $i=>$n){
    $ext=strtolower(pathinfo($n, PATHINFO_EXTENSION));
    if($ext==='pdf'){ $pdfCount++; if($firstPdf<0)$firstPdf=$i; }
  }
  if($pdfCount>0){
    $tmp=$tmps[$firstPdf];
    if(is_uploaded_file($tmp)){
      if(!@move_uploaded_file($tmp,$targetPdf)) @copy($tmp,$targetPdf);
      return file_exists($targetPdf) ? $targetPdf : '';
    }
    return '';
  }

  $pages=[];
  foreach($names as $i=>$n){
    $tmp=$tmps[$i]; $sz=(int)$sizes[$i];
    if($sz<=0 || !is_uploaded_file($tmp)) continue;
    $ext=strtolower(pathinfo($n, PATHINFO_EXTENSION));
    if(!in_array($ext,['jpg','jpeg','png','gif','webp'])) continue;
    $crop=(isset($crops[$i]) && is_array($crops[$i])) ? $crops[$i] : null;
    $gd=_gd_image_to_jpeg_bytes($tmp,$ext,$crop);
    if($gd && isset($gd['data'])) $pages[]=['data'=>$gd['data'],'w'=>$gd['w'],'h'=>$gd['h']];
    if (function_exists('gc_collect_cycles')) gc_collect_cycles();
  }
  if(!empty($pages)){
    if(_build_multi_pdf_from_jpegs($pages,$targetPdf)) return $targetPdf;
  }
  return '';
}
$imagen1 = construirPDFAdjunto($TARGET_DIR, 'imagen1', $crop_json_list); // '' si no hay válidos

/* ==================== 4) Guardar sidecar JSON con TODO $_POST ==================== */
try{
  $meta = [
    'saved_at'   => date('Y-m-d H:i:s'),
    'by_user_id' => $idusuario,
    'by_sucursal'=> $idsucursal,
    'client_ip'  => $_SERVER['REMOTE_ADDR'] ?? '',
    'post'       => $all_post
  ];
  $baseName = $imagen1 ? pathinfo($imagen1, PATHINFO_FILENAME) : ('vacextra_'.date('Ymd_His').'_'.mt_rand(1000,9999));
  file_put_contents($TARGET_DIR.$baseName.'.meta.json', json_encode($meta, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT));
}catch(Throwable $e){ /* no interrumpe */ }

/* ==================== 5) INSERT con fechas NO NULAS ==================== */
// Si vienen vacías, usamos:
//  - fecha_documento: hoy por defecto
//  - fecha_desde: fallback = fecha_documento
//  - fecha_hasta: fallback = (desde !== '' ? desde : fecha_documento)

$fecha_documento_sql = date_sql_not_null($con, $fecha_documento, date('Y-m-d'));
$desde_sql           = date_sql_not_null($con, $desde, $fecha_documento);
$hasta_sql           = date_sql_not_null($con, $hasta, ($desde !== '' ? $desde : $fecha_documento));

$idempleado_sql = mysqli_real_escape_string($con,(string)$idempleado);
$motivo_sql     = mysqli_real_escape_string($con,$motivo);
$cant_sql       = mysqli_real_escape_string($con,(string)$cantidad_dias);
$imagen1_sql    = mysqli_real_escape_string($con,$imagen1);

$sql = "
  INSERT INTO ingreso_vacaciones_extra(
    fecha_documento,
    fecha_desde,
    fecha_hasta,
    idempleado,
    motivo,
    cantidad_dias,
    imagen1
  ) VALUES (
    $fecha_documento_sql,
    $desde_sql,
    $hasta_sql,
    '$idempleado_sql',
    '$motivo_sql',
    '$cant_sql',
    '$imagen1_sql'
  )";
mysqli_query($con,$sql) or die(mysqli_error($con));

$idvacacion_extra = mysqli_insert_id($con);

/* // (Opcional) Si también quieres registrar feriados para EXTRA, descomenta:
function guardarFeriados($con, $idvacacion, $feriados_json, $desde, $hasta){
  $arr = json_decode($feriados_json, true); if (!is_array($arr)) return;
  foreach ($arr as $f){
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/',$f)) continue;
    if ($desde !== '' && $f < $desde) continue;
    if ($hasta !== '' && $f > $hasta) continue;
    $w=(int)date('w', strtotime($f)); if ($w===0 || $w===6) continue;
    $f_sql=mysqli_real_escape_string($con,$f);
    mysqli_query($con,"INSERT IGNORE INTO vacaciones_feriados (idvacacion, fecha) VALUES (".intval($idvacacion).",'$f_sql')");
  }
}
guardarFeriados($con, $idvacacion_extra, $feriados_json, $desde, $hasta);
*/

/* ==================== 6) Redirecciones ==================== */
if ($idusuario && $idsucursal) {
  echo "<script>document.location='consulta_vacaciones_extra.php?idusuario=$idusuario&idsucursal=$idsucursal'</script>";
} else {
  echo "<script>document.location='consulta_vacaciones_extra.php'</script>";
}
echo "<script>document.location='paso2.php'</script>";
