<?php
// alta_categoria_empleadosplanilla.php (antes tu script)
// - UTF-8 robusto (dbcon.php ya fuerza utf8mb4)
// - Prepared statements (sin errores por comillas/apóstrofes)
// - Upload opcional de imagen1 (si la necesitas para otra tabla)

declare(strict_types=1);

session_start();
require_once __DIR__ . '/dbcon.php'; // mantiene $con (mysqli) con utf8mb4

/* ---------------- Helpers ---------------- */

// Normaliza strings a UTF-8 válido y elimina controles no imprimibles.
function norm(string $s): string {
  if (!mb_detect_encoding($s, 'UTF-8', true)) {
    $s = @mb_convert_encoding($s, 'UTF-8', 'Windows-1252,ISO-8859-1,UTF-8') ?: $s;
  }
  return preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/u', '', $s);
}

// Inserta categoría usando prepared statement
function insertar_categoria(mysqli $con, string $categoria): bool {
  $sql = "INSERT INTO categoria_empleados (categoria) VALUES (?)";
  if (!$stmt = $con->prepare($sql)) {
    error_log("prepare(insertar_categoria) failed: ".$con->error);
    return false;
  }
  $stmt->bind_param("s", $categoria);
  $ok = $stmt->execute();
  if (!$ok) error_log("execute(insertar_categoria) failed: ".$stmt->error);
  $stmt->close();
  return $ok;
}

// Elimina inventario por id (si existe)
function eliminar_inventario(mysqli $con, int $id): bool {
  $sql = "DELETE FROM inventario WHERE id = ?";
  if (!$stmt = $con->prepare($sql)) {
    error_log("prepare(eliminar_inventario) failed: ".$con->error);
    return false;
  }
  $stmt->bind_param("i", $id);
  $ok = $stmt->execute();
  if (!$ok) error_log("execute(eliminar_inventario) failed: ".$stmt->error);
  $stmt->close();
  return $ok;
}

// Guarda imagen1 opcionalmente y devuelve la ruta pública (o cadena vacía)
function guardar_imagen1_opcional(): string {
  if (empty($_FILES['imagen1']['name']) || !is_uploaded_file($_FILES['imagen1']['tmp_name'])) {
    return '';
  }

  $target_dir_abs = __DIR__ . "/Files/Empleados_planilla/"; // ruta absoluta
  $target_dir_pub = "Files/Empleados_planilla/";            // ruta pública

  if (!is_dir($target_dir_abs)) @mkdir($target_dir_abs, 0775, true);

  // Sanea nombre de archivo
  $baseName = basename($_FILES['imagen1']['name']);
  $baseName = preg_replace('/[^A-Za-z0-9_\-.]/', '_', $baseName);

  $abs = $target_dir_abs . $baseName;
  $pub = $target_dir_pub . $baseName;

  // Verificación básica de imagen
  $check = @getimagesize($_FILES['imagen1']['tmp_name']);
  if ($check === false) {
    // Si quieres permitir PDF, quita esta validación.
    die("El archivo subido en imagen1 no es una imagen válida.");
  }

  // Tamaño máx 50MB
  if ((int)$_FILES['imagen1']['size'] > 50 * 1024 * 1024) {
    die("El archivo de imagen es demasiado grande (máx 50MB).");
  }

  if (!@move_uploaded_file($_FILES['imagen1']['tmp_name'], $abs)) {
    die("No se pudo subir el archivo de imagen1.");
  }

  return $pub;
}

/* ---------------- Entrada ---------------- */

$idsucursal = $_SESSION['idsucursal'] ?? '';
$idusuario  = $_SESSION['idusuario']  ?? ''; // para redirección compat

$categoria  = norm($_POST["categoria"] ?? '');

// id para borrar de inventario (si viene)
$ideliminar = isset($_GET["id"]) ? (int)$_GET["id"] : 0;

/* ---------------- Upload opcional ---------------- */

$ruta_imagen1 = guardar_imagen1_opcional(); // si luego la necesitas, úsala

/* ---------------- Operaciones DB ---------------- */

// Si hay id válido, intenta borrar de inventario. Si no existe, no es error fatal.
if ($ideliminar > 0) {
  eliminar_inventario($con, $ideliminar);
}

// Inserta la categoría de forma segura
if (!insertar_categoria($con, $categoria)) {
  die("No se pudo guardar la categoría.");
}

/* ---------------- Redirección final ---------------- */

// Redirige a tu listado (con params como usabas antes)
$qs = 'idusuario=' . rawurlencode((string)$idusuario) . '&idsucursal=' . rawurlencode((string)$idsucursal);
echo "<script>document.location='consulta_categoria_empleadosplanilla.php?{$qs}'</script>";
exit;
